package ru.yandex.calendar.frontend.api.event;

import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.UserOrResourceAvailabilityIntervals;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SpecialUsers;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.email.Email;

/**
 * @author dbrylev
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetResourcesEvents extends ActionWithIntervalParams {

    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;

    @RequestParam("has")
    private Option<String> filter;
    @RequestParam(value = "email", required = false)
    private ListF<Email> emails;
    @RequestParam(value = "officeAbbr", required = false)
    private ListF<String> officeAbbrs;
    @RequestParam(value = "officeCity", required = false)
    private ListF<String> officeCities;

    @RequestParam(required = false)
    private boolean outTs;

    @Override
    protected XmlOrJsonWritable doExecute() {
        ListF<ResourceInfo> resources = resourceRoutines.getDomainResourcesCanBookWithLayersAndOffices(
                SpecialUsers.ARTJOCK, getOfficeFilter(), getResourceFilter());

        AvailabilityRequest request = AvailabilityRequest.interval(getFrom(), getTo())
                .includeEventsOrganizers()
                .includeEventsParticipants()
                .withoutPermsCheck();

        ListF<UserOrResourceAvailabilityIntervals> avails = availRoutines.getAvailabilityIntervalss(
                SpecialUsers.ARTJOCK, Cf.list(), resources.map(ResourceInfo::getResource), request, getActionInfo());

        MapF<Long, ResourceInfo> resourceById = resources.toMapMappingToKey(ResourceInfo::getResourceId);

        return w -> {
            w.startArray("resources");
            for (UserOrResourceAvailabilityIntervals avail : avails) {
                w.startObject("resource");

                ResourceInfo resource = resourceById.getOrThrow(avail.getSubjectId().getResourceId());
                w.addTextField("name", resource.getName().getOrNull());
                w.addTextField("email", resource.getEmail().getEmail());

                w.startObject("office");
                w.addTextField("abbr", resource.getOffice().getAbbr().getOrNull());
                w.addTextField("city", resource.getOffice().getCityName().getOrNull());
                w.endObject();

                DateTimeFormatter dateFormatter = ISODateTimeFormat.dateTimeNoMillis()
                        .withZone(OfficeManager.getOfficeTimeZone(resource.getOffice()));

                w.startArray("events");
                    for (AvailabilityInterval interval : avail.getIntervalsO().get().unmerged()) {
                        w.startObject("event");

                        if (outTs) {
                            w.addNumberField("start", interval.getInterval().getStartMillis());
                            w.addNumberField("end", interval.getInterval().getEndMillis());
                        } else {
                            w.addTextField("start", dateFormatter.print(interval.getInterval().getStart()));
                            w.addTextField("end", dateFormatter.print(interval.getInterval().getEnd()));
                        }
                        Function1V<SettingsInfo> writeYandexUserFieldsF = si -> {
                            w.addTextField("name", si.getCommon().getUserName().getOrNull());
                            w.addTextField("login", si.getCommon().getUserLogin().getOrNull());
                            w.addTextField("email", Emails.getUnicoded(si.getEmail()));
                            w.addNumberField("uid", si.getUid().getUid());
                        };

                        interval.getOrganizer().forEach(si -> {
                            w.startObject("organizer");
                            writeYandexUserFieldsF.apply(si);
                            w.endObject();
                        });
                        ListF<UserParticipantInfo> attendees = interval.getParticipants().get().isMeeting()
                                ? interval.getParticipants().get().getAllUserAttendeesButNotOrganizer()
                                : Cf.list();

                        w.startArray("attendees");
                        attendees.forEach(p -> p.getSettingsIfYandexUser().forEach(si -> {
                            w.startObject("attendee");
                            writeYandexUserFieldsF.apply(si);
                            w.addTextField("decision", p.getDecision().toDbValue());
                            w.endObject();
                        }));
                        w.endArray();

                        w.endObject();
                    }
                w.endArray();
                w.endObject();
            }
            w.endArray();
        };
    }

    private OfficeFilter getOfficeFilter() {
        OfficeFilter officeFilter = OfficeFilter.any();

        if (officeAbbrs.isNotEmpty()) {
            officeFilter = officeFilter.withAbbr(officeAbbrs);
        }
        if (officeCities.isNotEmpty()) {
            officeFilter = officeFilter.withCityName(officeCities);
        }
        return officeFilter;
    }

    private ResourceFilter getResourceFilter() {
        ResourceFilter resourceFilter = ResourceFilter.any();

        if (filter.isPresent()) {
            resourceFilter = resourceFilter.withFilter(filter.get());
        }
        if (emails.isNotEmpty()) {
            resourceFilter = resourceFilter.withEmailFilter(emails);
        }
        return resourceFilter;
    }
}
