package ru.yandex.calendar.frontend.api.ics;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.api.ApiManager;
import ru.yandex.calendar.frontend.api.EventAndNeighbourEvents;
import ru.yandex.calendar.frontend.api.XmlOrJsonApiActionSupport;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.frontend.api.event.EventSerializer;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImportStats;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVEvent;
import ru.yandex.calendar.logic.ics.iv5j.ical.component.IcsVToDo;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsAttendee;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;

/**
 * @author gutman
 */
public class GetCalendarInfoByIcsUrlAction extends XmlOrJsonApiActionSupport {

    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ApiManager apiManager;

    @RequestParam
    private String icsUrl;
    @RequestParam
    private PassportUid uid;

    @Override
    protected XmlOrJsonWritable doExecute() {
        byte[] calendarBytes = ApacheHttpClientUtils.download(UrlUtils.urlDecode(icsUrl));
        IcsCalendar calendar = IcsUtils.parseBytesInGuessedEncoding(calendarBytes);

        // XXX: require some debug parameters (e.g. message-id)

        IcsMethod method = calendar.getMethod();

        if (!(method.sameAs(IcsMethod.REQUEST) && !(method.sameAs(IcsMethod.PUBLISH)))) {
            return XmlOrJsonWritable.textField("not-a-request-or-publish", "");
        }

        ListF<IcsVEvent> events = calendar.getEvents();
        ListF<IcsVToDo> todos = calendar.getTodos();

        if (events.isNotEmpty() && todos.isNotEmpty()) {
            return XmlOrJsonWritable.textField("both-events-ant-todos-in-calendar", "");
        }

        if (events.size() > 1) {
            return XmlOrJsonWritable.textField("more-than-one-event", "");
        }

        if (method.sameAs(IcsMethod.REQUEST)) {
            return handleRequest(calendar);
        }

        if (method.sameAs(IcsMethod.REPLY)) {
            return handleReply(calendar);
        }

        if (method.sameAs(IcsMethod.PUBLISH)) {
            if (events.isNotEmpty()) {
                return handleEventsPublish(calendar);
            } else if (todos.isNotEmpty()) {
                return handleTodosPublish(calendar);
            }
        }
        return XmlOrJsonWritable.empty();
    }

    private XmlOrJsonWritable handleEventsPublish(IcsCalendar calendar) {
        IcsVEvent vevent = calendar.getEvents().single();
        String externalId = vevent.getUid().get();
        IcsVTimeZones tzs = IcsVTimeZones.cons(
                calendar.getTimezones(), dateTimeManager.getTimeZoneForUid(uid), vevent.isRepeatingOrRecurrence());

        Option<Instant> recurrenceId = vevent.getRecurrenceIdInstant(tzs);
        Option<Event> eventOnUserLayer = eventRoutines.findEvent(
                Cf.list(UidOrResourceId.user(uid)), externalId,
                recurrenceId);

        Option<Event> event = eventRoutines.findEventByExternalIdAndRecurrence(externalId, recurrenceId);

        final DateTimeZone tz = DateTimeZone.forID(settingsRoutines.getTimeZoneJavaId(uid));
        final EventAndNeighbourEvents eventAndNeighbours = apiManager.getEventAndNeighbourEventsByEvent(event.get(),
                uid, tz);
        final boolean alreadyAddedToCalendar = eventOnUserLayer.isPresent();

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                EventSerializer.serializeEventAndNeighbourEvents(w, tz, eventAndNeighbours);
                xmlizeActions(w, alreadyAddedToCalendar);
            }
        };
    }

    private XmlOrJsonWritable handleTodosPublish(IcsCalendar calendar) {
        return XmlOrJsonWritable.textField("todo-list", "");
    }

    public XmlOrJsonWritable handleRequest(IcsCalendar calendar) {
        // XXX: proper mode (that acts like mailhook)
        IcsImportStats stats = icsImporter.importIcsStuff(uid, calendar, IcsImportMode.incomingEmailFromMailhook());

        long eventId = stats.getProcessedEventIds().single();

        Event event = eventDbManager.getEventById(eventId);
        final Option<EventUser> eventUser = eventUserDao.findEventUserByEventIdAndUid(event.getId(), uid);

        final DateTimeZone tz = DateTimeZone.forID(settingsRoutines.getTimeZoneJavaId(uid));
        final EventAndNeighbourEvents eventAndNeighbours = apiManager.getEventAndNeighbourEventsByEvent(event, uid, tz);

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                EventSerializer.serializeEventAndNeighbourEvents(w, tz, eventAndNeighbours);
                if (eventUser.isPresent() && eventUser.get().getIsAttendee() && !eventUser.get().getIsOrganizer()) {
                    EventSerializer.serializeDecisions(w, eventUser.get());
                }
            }
        };
    }

    private XmlOrJsonWritable handleReply(IcsCalendar calendar) {
        IcsImportStats stats = icsImporter.importIcsStuff(uid, calendar, IcsImportMode.incomingEmailFromMailhook());

        long eventId = stats.getProcessedEventIds().single();

        final Event event = eventDbManager.getEventById(eventId);

        final DateTimeZone tz = DateTimeZone.forID(settingsRoutines.getTimeZoneJavaId(uid));

        final IcsAttendee attendee = calendar.getEvents().single().getAttendees().single();

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                EventSerializer.serializeEventFields(w, event, Option.of(tz));
                w.startObject("reply");
                w.addTextField("attendee", attendee.getUnicodedEmail());
                w.addTextField("decision", Decision.findByIcsPartStat(attendee.getPartStat().get()).get().value());
                w.endObject();
            }
        };
    }

    private void xmlizeActions(XmlOrJsonWriter xw, boolean alreadyAddedToCalendar) {
        xw.startArray("actions");

        if (alreadyAddedToCalendar) {
            xw.addTextField("action", "detach");
        } else {
            xw.addTextField("action", "attach");
        }

        xw.endArray();
    }
}
