package ru.yandex.calendar.frontend.api.ics;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.Validate;

/**
 * @author gutman
 */
public class TodoImportResult {

    private static enum Situation {
        IMPORTED_TO_NEW_LIST,
        IMPORTED_TO_DEFAULT_LIST,
        LIST_ALREADY_EXISTS,
        LIST_ALREADY_EXISTS_AND_OVERWRITTEN,
    }
    private final Option<String> newTodoListExternalId;
    private final Option<String> deletedListExternalId;
    private final Situation situation;

    private TodoImportResult(
            Situation situation, Option<String> newTodoListExternalId, Option<String> deletedListExternalId)
    {
        if (situation == Situation.IMPORTED_TO_NEW_LIST) {
            Validate.isTrue(newTodoListExternalId.isPresent());
        }
        this.situation = situation;
        this.newTodoListExternalId = newTodoListExternalId;
        this.deletedListExternalId = deletedListExternalId;
    }

    static TodoImportResult toNewList(String listId) {
        return new TodoImportResult(Situation.IMPORTED_TO_NEW_LIST, Option.of(listId), Option.<String>empty());
    }

    static TodoImportResult toDefaultList() {
        return new TodoImportResult(Situation.IMPORTED_TO_DEFAULT_LIST, Option.<String>empty(), Option.<String>empty());
    }

    static TodoImportResult listAlreadyExists() {
        return new TodoImportResult(Situation.LIST_ALREADY_EXISTS, Option.<String>empty(), Option.<String>empty());
    }

    static TodoImportResult listAlreadyExistsAndOverwritten(String newTodoListExternalId, String deletedListExternalId)
    {
        return new TodoImportResult(Situation.LIST_ALREADY_EXISTS_AND_OVERWRITTEN,
                Option.of(newTodoListExternalId), Option.of(deletedListExternalId));
    }

    public boolean getListAlreadyExists() {
        return situation == Situation.LIST_ALREADY_EXISTS;
    }

    public boolean getListAlreadyExistsAndOverwritten() {
        return situation == Situation.LIST_ALREADY_EXISTS_AND_OVERWRITTEN;
    }

    public boolean getImportedToNewList() {
        return situation == Situation.IMPORTED_TO_NEW_LIST;
    }

    public String getNewTodoListExternalId() {
        return newTodoListExternalId.get();
    }

    public String getDeletedListExternalId() {
        return deletedListExternalId.get();
    }

}
