package ru.yandex.calendar.frontend.api.inviter;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.converters.ConverterToInstantMillis;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author dbrylev
 */
public class GetResourcesBusyIntervals extends InviterActionSupport {

    @Autowired
    private InviterManager inviterManager;

    @RequestParam("has")
    private Option<String> filter;
    @RequestParam(value = "email", required = false)
    private ListF<Email> emails;
    @RequestParam(value = "officeAbbr", required = false)
    private ListF<String> officeAbbrs;
    @RequestParam(value = "officeCity", required = false)
    private ListF<String> officeCities;

    @RequestParam(customConverter = ConverterToInstantMillis.class)
    private Option<Instant> fromTs;
    @RequestParam(customConverter = ConverterToInstantMillis.class)
    private Option<Instant> toTs;

    @Override
    protected XmlOrJsonWritable doExecute() {
        InstantInterval interval = new InstantInterval(
                fromTs.getOrElse(Instant.now()), toTs.getOrElse(Instant.now().plus(Duration.standardDays(3))));

        final ListF<ResourceBusyIntervals> busyIntervalss =
                inviterManager.getResourcesBusyIntervals(interval, getOfficeFilter(), getResourceFilter());

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                w.startArray("resources");
                for (ResourceBusyIntervals busyIntervals : busyIntervalss) {
                    w.startObject("resource");

                    InviterSerializer.writeResourceInfoFields(w, busyIntervals.getResourceInfo());

                    w.startArray("busy_intervals");
                    for (InstantInterval interval : busyIntervals.getIntervals()) {
                        w.startObject("busy_interval");

                        w.addNumberField("start_ts", interval.getStartMillis());
                        w.addNumberField("end_ts", interval.getEndMillis());

                        w.endObject();
                    }
                    w.endArray();

                    w.endObject();
                }
                w.endArray();
            }
        };
    }

    private OfficeFilter getOfficeFilter() {
        OfficeFilter officeFilter = OfficeFilter.any();

        if (officeAbbrs.isNotEmpty()) {
            officeFilter = officeFilter.withAbbr(officeAbbrs);
        }
        if (officeCities.isNotEmpty()) {
            officeFilter = officeFilter.withCityName(officeCities);
        }
        return officeFilter;
    }

    private ResourceFilter getResourceFilter() {
        ResourceFilter resourceFilter = ResourceFilter.any();

        if (filter.isPresent()) {
            resourceFilter = resourceFilter.withFilter(filter.get());
        }
        if (emails.isNotEmpty()) {
            resourceFilter = resourceFilter.withEmailFilter(emails);
        }
        return resourceFilter;
    }
}
