package ru.yandex.calendar.frontend.api.inviter;

import java.util.List;
import java.util.stream.Stream;

import lombok.val;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.Validate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventResource;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.UserOrResourceAvailabilityIntervals;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.schedule.ResourceEventsAndReservations;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.SpecialUsers;
import ru.yandex.calendar.micro.yt.StaffCache;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;

import static ru.yandex.calendar.frontend.api.inviter.CoParticipant.Meeting;
import static ru.yandex.calendar.frontend.api.inviter.UserOccupiedResources.OccupiedResource;

public class InviterManager {
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private StaffCache staffCache;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;

    public ListF<CoParticipant> getUserCoParticipants(
            final PassportUid uid, InstantInterval interval, boolean includeAllUsers)
    {
        Validate.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "no inviter for public");

        ListF<EventWithRelations> meetings = eventDbManager.getEventsWithRelationsByEvents(
                getUserNonRepeatingMeetingsStartingIn(uid, interval));

        ListF<CoParticipant> coParticipants = meetings.flatMap(getUserCoParticipantsF(uid))
                .groupBy(CoParticipant::getUid).values().toList().map(mergeCoParticipantF());

        return includeAllUsers
                ? coParticipants.plus(getAllUsersAsCoParticipants()).stableUniqueBy(CoParticipant::getUid)
                : coParticipants;
    }

    public ListF<UserBusyIntervals> getUsersBusyIntervals(
            ListF<PassportUid> uids, InstantInterval interval, boolean includeAbsences)
    {
        Validate.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "no inviter for public");

        ListF<UserInfo> users = getUserInfos(uids);
        AvailabilityRequest req = AvailabilityRequest.interval(interval).includeEventsResources().withoutPermsCheck();

        if (!includeAbsences) req = req.excludeAbsencesEvents();

        ListF<UserOrResourceAvailabilityIntervals> avails = availRoutines.getAvailabilityIntervalss(
                SpecialUsers.ARTJOCK, users.map(UserInfo.getUidF().andThen(UidOrResourceId.userF())),
                req, getActionInfo());

        Tuple2List<UserInfo, ListF<AvailabilityInterval>> usersAvails = users.zipWith(UserInfo.getUidF()
                .andThen(Cf2.f(avails.toMapMappingToKey(UserOrResourceAvailabilityIntervals.getUidF())::getOrThrow))
                .andThen(UserOrResourceAvailabilityIntervals.getUnmergedF()));

        return usersAvails.map(UserBusyIntervals.fromAvailabilityIntervalsF());
    }

    public ListF<ResourceBusyIntervals> getResourcesBusyIntervals(
            InstantInterval interval, OfficeFilter officeFilter, ResourceFilter resourceFilter)
    {
        Validate.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "no inviter for public");

        ListF<ResourceInfo> resources = resourceRoutines.getDomainResourcesCanBookWithLayersAndOffices(
                SpecialUsers.ARTJOCK, officeFilter, resourceFilter);

        Tuple2List<ResourceInfo, ResourceEventsAndReservations> schedule = resourceScheduleManager
                .getResourceScheduleDataForInterval(
                        Option.empty(), resources,
                        interval, MoscowTime.TZ, Option.empty(), getActionInfo());

        return schedule.map2(ResourceEventsAndReservations::getMergedIntervals).map(ResourceBusyIntervals.consF());
    }

    public ListF<UserOccupiedResources> getUsersOccupiedResources(ListF<PassportUid> uids, InstantInterval interval) {
        Validate.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "no inviter for public");

        Tuple2List<PassportUid, ListF<Event>> userOrganizedEvents =
                getUsersOrganizedNonRepeatingMeetingsStartingIn(uids, interval);

        final MapF<Long, Event> eventById = userOrganizedEvents.get2().<Event>flatten().toMapMappingToKey(Event.getIdF());

        final ListF<EventResource> eventResources = eventResourceDao.findEventResourcesByEventIds(eventById.keys());

        MapF<Long, ListF<EventResource>> eventResourcesByEventId = eventResources.groupBy(EventResource::getEventId);

        final Function<Long, ListF<EventResource>> eventResourcesByEventIdF =
                id -> eventResourcesByEventId.getOrElse(id, Cf.list());

        final MapF<PassportUid, UserInfo> userInfoByUid = getUserInfos(uids).toMapMappingToKey(UserInfo.getUidF());

        final MapF<Long, ResourceInfo> resourceInfoById = resourceDao
                .findResourceInfosByIds(eventResources.map(EventResource.getResourceIdF()).stableUnique())
                .toMapMappingToKey(ResourceInfo.resourceIdF());

        return userOrganizedEvents.map((uid, events) -> {
            ListF<EventResource> ers = events.flatMap(Event.getIdF().andThen(eventResourcesByEventIdF));

            ListF<OccupiedResource> resources = ers.groupBy(EventResource.getResourceIdF()).mapEntries(
                (resourceId, ers1) -> new OccupiedResource(
                        resourceInfoById.getOrThrow(ers1.first().getResourceId()),
                        ers1.map(er -> eventById.getOrThrow(er.getEventId())))
            );
            return new UserOccupiedResources(userInfoByUid.getOrThrow(uid), resources);
        });
    }

    public Event createMeeting(PassportUid uid, InviterEventData eventData) {
        EventData data = InviterEventDataConverter.convert(eventData, dateTimeManager.getTimeZoneForUid(uid));

        return eventWebManager.createUserEvent(
                uid, data, NotificationsData.useLayerDefaultIfCreate(),
                InvitationProcessingMode.SAVE_ATTACH_SEND, getActionInfo()).getEvent();
    }

    private ListF<UserInfo> getUserInfos(ListF<PassportUid> uids) {
        ListF<SettingsInfo> settings = uids.map(settingsRoutines.getSettingsByUidBatch(uids)::getOrThrow);

        return settings.map(s -> {
            Option<Long> lastActivityOfficeId = s.getYt().filterMap(SettingsYt.getActiveOfficeIdF());
            Option<Long> tableOfficeId = s.getYt().filterMap(SettingsYt.getTableOfficeIdF());
            Option<Integer> tableFloorNum = s.getYt().filterMap(SettingsYt.getTableFloorNumF());

            return new UserInfo(s.getCommon(),
                    officeManager.getOfficesByIds(lastActivityOfficeId).singleO(),
                    officeManager.getOfficesByIds(tableOfficeId).singleO(), tableFloorNum);
        });
    }

    private ListF<CoParticipant> getUserCoParticipants(PassportUid uid, EventWithRelations event) {
        ListF<YandexUserParticipantInfo> attendees = event.getParticipants()
                .getParticipantsSafeWithInconsistent()
                .filterByType(YandexUserParticipantInfo.class)
                .filter(ParticipantInfo.isAttendeeF());

        Function1B<ParticipantInfo> isCurrentUserF = ParticipantInfo.getUidF().andThen(Cf2.isSomeF(uid));

        Option<YandexUserParticipantInfo> userO = attendees.find(isCurrentUserF);
        ListF<YandexUserParticipantInfo> users = attendees.filter(isCurrentUserF.notF());

        if (!userO.isPresent()) return Cf.list();

        final ParticipantInfo user = userO.get();
        final Meeting meeting = new Meeting(event.getEvent().getCreationTs(), attendees.size());

        return users.map(p -> {
            ListF<Meeting> from = p.isOrganizer() ? Cf.list(meeting) : Cf.list();
            ListF<Meeting> to = user.isOrganizer() ? Cf.list(meeting) : Cf.list();
            ListF<Meeting> common = !user.isOrganizer() && !p.isOrganizer() ? Cf.list(meeting) : Cf.list();

            return new CoParticipant(p.getUid().get(), p.getEmail(), from, to, common);
        });
    }

    private List<CoParticipant> getAllUsersAsCoParticipants() {
        return StreamEx.of(staffCache.getUsers())
            .flatMap(user -> {
                val email = Email.parseSafe(user.getInfo().getWorkEmail());
                if (!user.getInfo().isDismissed() && email.isPresent()) {
                    return Stream.of(CoParticipant.empty(PassportUid.cons(user.getUid().getValue()), email.get()));
                } else {
                    return Stream.empty();
                }
            })
            .toImmutableList();
    }

    private Tuple2List<PassportUid, ListF<Event>> getUsersOrganizedNonRepeatingMeetingsStartingIn(
            ListF<PassportUid> uids, InstantInterval interval)
    {
        SqlCondition eventCondition = EventFields.RECURRENCE_ID.column().isNull()
                .and(EventFields.REPETITION_ID.column().isNull())
                .and(EventFields.START_TS.column().ge(interval.getStart()))
                .and(EventFields.START_TS.column().lt(interval.getEnd()));

        return eventUserDao.findUserOrganizedEventsByUids(uids, eventCondition);
    }

    private ListF<Event> getUserNonRepeatingMeetingsStartingIn(PassportUid uid, InstantInterval interval) {

        SqlCondition eventCondition = EventFields.RECURRENCE_ID.column().isNull()
                .and(EventFields.REPETITION_ID.column().isNull())
                .and(EventFields.START_TS.column().ge(interval.getStart()))
                .and(EventFields.START_TS.column().lt(interval.getEnd()));

        SqlCondition eventUserCondition = EventUserFields.UID.eq(uid)
                .and(EventUserFields.IS_ATTENDEE.eq(true));

        return eventUserDao.findEventsByEventUsers(eventCondition, eventUserCondition);
    }

    private Function<EventWithRelations, ListF<CoParticipant>> getUserCoParticipantsF(final PassportUid uid) {
        return e -> getUserCoParticipants(uid, e);
    }

    private Function<ListF<CoParticipant>, CoParticipant> mergeCoParticipantF() {
        return as -> as.reduceLeft(CoParticipant.mergeF());
    }

    private ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }
}
