package ru.yandex.calendar.frontend.api.mail;

import org.jetbrains.annotations.Nullable;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.interceptors.LoggingParameterBinder;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.Parameter;
import ru.yandex.commune.a3.action.parameter.ParameterDescriptor;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.parameter.bind.ParameterBinder;
import ru.yandex.commune.a3.utils.ParameterRef;
import ru.yandex.commune.a3.utils.ParameterRetriever;
import ru.yandex.commune.a3.utils.ParameterRetrieverFactory;
import ru.yandex.commune.mail.ContentType;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

public class DownloadIcsBinder implements ParameterBinder, LoggingParameterBinder {

    private final ParameterRef urlParam = ParameterRef.request("icsUrl");
    private final ParameterRetriever<String> urlRetriever = ParameterRetrieverFactory.cons(String.class, urlParam);

    @Nullable
    @Override
    public Object createAndBind(
            WebRequest webRequest,
            ActionInvocationContext invocationContext,
            ParameterDescriptor parameterDescriptor
    ) {
        Class<?> parameterClass = parameterDescriptor.getParameterType().getActualType().asClass().getClazz();
        Check.equals(DownloadedIcs.class, parameterClass);

        HttpServletRequestX request = webRequest.getHttpServletRequest();
        Option<ContentType> contentType = Option.ofNullable(request.getContentType()).map(ContentType::valueOf);

        if (contentType.exists(type -> type.getSubtype().equals("calendar"))) {
            return Option.of(new DownloadedIcs(Option.empty(), IcsUtils.parse(request.getInputStreamX())));
        }

        Option<String> icsUrl = urlRetriever.retrieve(webRequest, invocationContext);
        ValidateParam.some(urlParam.getName(), icsUrl);

        return Option.of(new DownloadedIcs(icsUrl, downloadIcs(icsUrl.get())));
    }

    @Override
    public Option<String> getNameForLogging(Parameter parameter) {
        return Option.of(urlParam.getName());
    }

    @Override
    public Option<String> getValueForLogging(Parameter parameter) {
        return requireValue(parameter).getUrl();
    }

    private IcsCalendar downloadIcs(String icsUrl) {
        byte[] calendarBytes = ExceptionUtils.executeOrThrow(
                () -> ApacheHttpClientUtils.download(UrlUtils.urlDecode(icsUrl)), "Failed to download ics attachment");
        return ExceptionUtils.executeOrThrow(
                () -> IcsUtils.parseBytesInGuessedEncoding(calendarBytes), "Failed to parse ics attachment");
    }

    public static DownloadedIcs requireValue(Parameter parameter) {
        Object raw = parameter.getValue().get();
        Object value = raw instanceof Option ? ((Option<?>) raw).get() : raw;

        return (DownloadedIcs) value;
    }
}
