package ru.yandex.calendar.frontend.api.mail;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.calendar.frontend.bender.UnicodeEmailMarshaller;
import ru.yandex.calendar.frontend.bender.UnicodeEmailUnmarshaller;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.InstantAsMillisMarshaller;
import ru.yandex.misc.bender.custom.InstantAsMillisUnmarshaller;
import ru.yandex.misc.bender.parse.simpleType.SimpleTypeUnmarshallerSupport;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.simpleType.SimpleTypeMarshallerSupport;
import ru.yandex.misc.bender.serialize.simpleType.StringValueMarshaller;
import ru.yandex.misc.email.Email;

/**
 * @author dbrylev
 */
public class MailBender {

    public static final BenderMapper mapper = new BenderMapper(getConfiguration());

    public static BenderConfiguration getConfiguration() {
        return BenderConfiguration.cons(
                MembersToBind.WITH_ANNOTATIONS, false,
                CustomMarshallerUnmarshallerFactoryBuilder.cons()
                        .add(Duration.class, new DurationAsMinutesMarshaller(), new DurationAsMinutesUnmarshaller())
                        .add(Instant.class, new InstantAsMillisMarshaller(), new InstantAsMillisUnmarshaller())
                        .add(DateTime.class, new DateTimeMarshaller())
                        .add(Email.class, new UnicodeEmailMarshaller(), new UnicodeEmailUnmarshaller())
                        .build());
    }

    private static class DurationAsMinutesMarshaller extends SimpleTypeMarshallerSupport {
        @Override
        protected String toStringValueForXml(Object o) {
            return Long.toString(((Duration) o).getStandardMinutes());
        }

        @Override
        protected void writeJson(BenderJsonWriter json, Object o) {
            json.writeNumber(((Duration) o).getStandardMinutes());
        }
    }

    private static class DurationAsMinutesUnmarshaller extends SimpleTypeUnmarshallerSupport {
        protected Duration convert(String o) {
            return Duration.standardMinutes(Long.parseLong(o));
        }
    }

    private static class DateTimeMarshaller extends StringValueMarshaller {
        protected String toStringValue(Object o) {
            return ((DateTime) o).toString("yyyy-MM-dd'T'HH:mm:ssZZ");
        }
    }
}
