package ru.yandex.calendar.frontend.api.mail;

import java.util.Optional;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Try;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.a3.bind.BindJson;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;

/**
 * @author Sergey Shinderuk
 */
@ActionContainer
public class MailEventActions {

    @Autowired
    private MailEventManager mailEventManager;

    @Action
    public MailEventInfo getEventInfo(@BindJson MailEventData eventData) {
        return mailEventManager.getEventInfo(eventData, actionInfo());
    }

    @Action
    public MailEventInfo createOrUpdateEvent(@BindJson MailEventData eventData) {
        return mailEventManager.createOrUpdateEvent(eventData, actionInfo());
    }

    @Action
    public MailEventInfo getEventInfo2(@BindJson MailEventData eventData) {
        eventData.setSecondsAsMinutesOffsetNotification();
        return getEventInfo(eventData).withSecondsAsMinutesOffsetNotification();
    }

    @Action
    public MailEventInfo createOrUpdateEvent2(@BindJson MailEventData eventData) {
        eventData.setSecondsAsMinutesOffsetNotification();
        return createOrUpdateEvent(eventData).withSecondsAsMinutesOffsetNotification();
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public MailEventInfoOrRefusal getEventInfoByIcsUrl(@RequestParam("uid") PassportUid uid,
                                                       @DownloadIcs DownloadedIcs ics,
                                                       @RequestParam("lang") Optional<Language> lang) {
        return mailEventManager.getEventInfoByIcsUrl(uid, ics.getCalendar(), lang, actionInfo());
    }

    @Action
    public MailEventInfoOrRefusal importEventByIcsUrl(@RequestParam("uid") PassportUid uid,
                                                      @DownloadIcs DownloadedIcs ics,
                                                      @RequestParam("lang") Optional<Language> lang) {
        return mailEventManager.importEventByIcsUrl(uid, ics.getCalendar(), lang, actionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public MailEventInfoOrRefusal getEventInfoByIcsUrl2(
            @RequestParam("uid") PassportUid uid, @DownloadIcs DownloadedIcs ics)
    {
        return mailEventManager.getEventInfoByIcsUrl(uid, ics.getCalendar(), Optional.empty(), actionInfo())
                .withSecondsAsMinutesOffsetNotification();
    }

    @Action
    public MailEventInfoOrRefusal createOrUpdateEventByIcsUrl(
            @RequestParam("uid") PassportUid uid,
            @DownloadIcs DownloadedIcs ics,
            @RequestParam("lang") Option<Language> lang,
            @RequestParam("decision") Option<MailDecision> decision,
            @BindJson Option<MailEventIcsData> eventData)
    {
        MailEventIcsData data = eventData.orElse(() -> decision.map(MailEventIcsData::new)).get();

        return mailEventManager.createOrUpdateEventByIcsUrl(uid, ics.getCalendar(), data, lang, actionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public IncomingIcsMessageInfoOrRefusal getIncomingIcsMessageInfo(
            @RequestParam("uid") PassportUid uid,
            @DownloadIcs DownloadedIcs ics,
            @RequestParam("lang") Option<Language> lang)
    {
        return mailEventManager.getIncomingIcsMessageInfo(uid, ics.getCalendar(), lang);
    }

    @Action
    public OkResultOrRefusal updateDecision(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("externalId") String externalId,
            @RequestParam(value = "recurrenceId", ignoreEmpty = true) Option<String> recurrenceId,
            @RequestParam("decision") Option<MailDecision> decision,
            @BindJson Option<MailEventIcsData> eventData)
    {
        decision = decision.orElse(eventData.map(MailEventIcsData::getDecision));
        ValidateParam.some("decision", decision);

        Try<Option<Instant>> recurId = Try.tryCatchException(() -> recurrenceId.map(Instant::parse));
        ValidateParam.isTrue("recurrenceId", recurId.isSuccess(), "Invalid value: " + recurrenceId.mkString(""));

        return mailEventManager.updateDecision(uid, externalId, recurId.get(), decision.get(), actionInfo());
    }

    private static ActionInfo actionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }

}
