package ru.yandex.calendar.frontend.api.mail;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;

/**
 * @author dbrylev
 */
@Bendable
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
public class MailEventData {
    private PassportUid uid;
    private String externalId;

    private String name;
    private String location;
    private Option<String> description;

    private Instant start;
    private Instant end;
    private int startClientTimezoneOffset;

    private Option<Boolean> isAllDay;

    private Option<MailDecision> decision;
    private Option<Notification> notification;

    private MailEventData(
            PassportUid uid, String externalId, String name, String location, Option<String> description,
            boolean allDay, Instant start, Instant end, int startClientTimezoneOffset,
            Option<MailDecision> decision, Option<Notification> notification)
    {
        this.uid = uid;
        this.externalId = externalId;
        this.name = name;
        this.location = location;
        this.description = description;
        this.isAllDay = Option.of(allDay);
        this.start = start;
        this.end = end;
        this.startClientTimezoneOffset = startClientTimezoneOffset;
        this.decision = decision;
        this.notification = notification;
    }

    // temporary hack for old version of ruchkas
    public void setSecondsAsMinutesOffsetNotification() {
        if (notification.isPresent()) {
            notification = Option.of(new Notification(
                    notification.get().getChannel(),
                    Duration.standardSeconds(notification.get().getOffset().getStandardMinutes())));
        }
    }

    public Event getEvent() {
        Event event = new Event();
        event.setName(name);
        event.setLocation(location);
        description.forEach(event::setDescription);

        event.setStartTs(start);
        event.setEndTs(end);
        event.setIsAllDay(isAllDay());
        event.setRecurrenceIdNull();

        return event;
    }

    public EventUser getEventUser() {
        EventUser eventUser = new EventUser();
        if (decision.isPresent()) {
            eventUser.setDecision(decision.get().toDecision());
            eventUser.setAvailability(Availability.BUSY);
        } else {
            eventUser.setDecision(Decision.UNDECIDED);
            eventUser.setAvailability(Availability.MAYBE);
        }
        return eventUser;
    }

    public DateTimeZone getTimeZone() {
        return DateTimeZone.forOffsetHoursMinutes(
                -startClientTimezoneOffset / 60, Math.abs(startClientTimezoneOffset % 60));
    }

    public PassportUid getUid() {
        return uid;
    }

    public String getExternalId() {
        return externalId;
    }

    public String getName() {
        return name;
    }

    public String getLocation() {
        return location;
    }

    public boolean isAllDay() {
        return isAllDay.getOrElse(false);
    }

    public Instant getStart() {
        return start;
    }

    public Instant getEnd() {
        return end;
    }

    public Option<MailDecision> getDecision() {
        return decision;
    }

    public Option<Notification> getNotification() {
        return notification;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private Option<PassportUid> uid = Option.empty();
        private Option<String> externalId = Option.empty();

        private Option<String> name = Option.empty();
        private Option<String> location = Option.empty();
        private Option<String> description = Option.empty();

        private Option<Instant> start = Option.empty();
        private Option<Instant> end = Option.empty();
        private Option<Integer> startClientTimezoneOffset = Option.empty();

        private Option<Boolean> isAllDay = Option.empty();

        private Option<MailDecision> decision = Option.empty();
        private Option<Notification> notification = Option.empty();

        private Builder() {}

        private Builder(MailEventData e) {
            this.uid = Option.of(e.uid);
            this.externalId = Option.of(e.externalId);

            this.name = Option.of(e.name);
            this.location = Option.of(e.location);

            this.start = Option.of(e.start);
            this.end = Option.of(e.end);
            this.isAllDay = e.isAllDay;
            this.startClientTimezoneOffset = Option.of(e.startClientTimezoneOffset);

            this.decision = e.decision;
            this.notification = e.notification;
        }

        public MailEventData build() {
            return new MailEventData(
                    uid.get(), externalId.get(), name.get(), location.get(), description,
                    isAllDay.get(), start.get(), end.get(), startClientTimezoneOffset.get(), decision, notification);
        }

        public Builder setUid(PassportUid uid) {
            this.uid = Option.of(uid);
            return this;
        }

        public Builder setExternalId(String externalId) {
            this.externalId = Option.of(externalId);
            return this;
        }

        public Builder setName(String name) {
            this.name = Option.of(name);
            return this;
        }

        public Builder setLocation(String location) {
            this.location = Option.of(location);
            return this;
        }

        public Builder setDescription(String description) {
            this.description = Option.of(description);
            return this;
        }

        public Builder setStart(Instant start) {
            this.start = Option.of(start);
            return this;
        }

        public Builder setStartUtc(int year, int month, int day, int hour, int minute) {
            return setStart(new DateTime(year, month, day, hour, minute, DateTimeZone.UTC).toInstant());
        }

        public Builder setEnd(Instant end) {
            this.end = Option.of(end);
            return this;
        }

        public Builder setEndUtc(int year, int month, int day, int hour, int minute) {
            return setEnd(new DateTime(year, month, day, hour, minute, DateTimeZone.UTC).toInstant());
        }

        public Builder setStartClientTimezoneOffset(int startClientTimezoneOffset) {
            this.startClientTimezoneOffset = Option.of(startClientTimezoneOffset);
            return this;
        }

        public Builder setAllDay(boolean allDay) {
            isAllDay = Option.of(allDay);
            return this;
        }

        public Builder setDecision(MailDecision decision) {
            this.decision = Option.of(decision);
            return this;
        }

        public Builder setNotification(Notification notification) {
            this.notification = Option.of(notification);
            return this;
        }

        public Builder clearNotification() {
            this.notification = Option.empty();
            return this;
        }
    }
}
