package ru.yandex.calendar.frontend.api.mail;

import java.util.Optional;

import lombok.Getter;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.a3.bind.JsonActionResult;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.time.MoscowTime;

@JsonActionResult(resultFieldName = "eventInfo")
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
@Getter
public class MailEventInfo {
    private final Optional<Long> eventId;
    private final String externalId;
    private final Option<String> recurrenceId;

    @BenderFlatten
    private final MailEventBaseInfo baseInfo;
    private final DateTime startDt;
    private final DateTime endDt;

    private final MailDecision decision;
    private final Option<Notification> notification;

    private final Option<String> instanceKey;
    private final boolean isCancelled;
    private final boolean isPast;
    private final boolean isNoRsvp;
    private final ListF<String> actions;

    private final Option<String> calendarMailType;
    private final Option<String> calendarUrl;

    private final Option<Conflict> conflict;
    private final Option<String> prodId;

    public MailEventInfo(
            Optional<Long> eventId, String externalId, Option<Instant> recurrenceId, String name, String location, String description,
            Instant start, Instant end, boolean allDay, MailRepetitionInfo repetitionInfo,
            MailParticipantsInfo participants, MailDecision decision, Option<Notification> notification,
            Option<String> instanceKey, boolean isCancelled, boolean isNoRsvp, boolean isPast,
            ListF<String> actions, Option<String> calendarMailType,
            Option<String> calendarUrl, Option<Conflict> conflict, Option<String> prodId, DateTimeZone userTz)
    {
        this.eventId = eventId;
        this.externalId = externalId;
        this.recurrenceId = recurrenceId.map(Instant::toString);
        this.baseInfo = new MailEventBaseInfo(
                name, location, description, start, end, allDay, repetitionInfo, participants);
        this.startDt = start.toDateTime(userTz);
        this.endDt = end.toDateTime(userTz);

        this.decision = decision;
        this.notification = notification;
        this.instanceKey = instanceKey;
        this.isCancelled = isCancelled;
        this.isNoRsvp = isNoRsvp;
        this.isPast = isPast;
        this.actions = actions;
        this.calendarMailType = calendarMailType;
        this.calendarUrl = calendarUrl;
        this.conflict = conflict;
        this.prodId = prodId;
    }

    // temporary hack for old version of ruchkas
    public MailEventInfo withSecondsAsMinutesOffsetNotification() {
        Option<Notification> notification = this.notification;
        if (notification.isPresent()) {
            notification = Option.of(new Notification(
                    notification.get().getChannel(),
                    Duration.standardMinutes(notification.get().getOffset().getStandardSeconds())));
        }
        return new MailEventInfo(
                eventId, externalId, getRecurrenceId(), baseInfo.getName(), baseInfo.getLocation(), baseInfo.getDescription(),
                baseInfo.getStart(), baseInfo.getEnd(), baseInfo.isAllDay(), baseInfo.getRepetitionInfo(),
                baseInfo.getParticipants(), decision, notification, instanceKey, isCancelled, isNoRsvp, isPast,
                actions, calendarMailType, calendarUrl, conflict, prodId, startDt.getZone());
    }

    public static Builder builder() {
        return new Builder();
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    public static class Conflict {
        private final String name;
        private final String location;
        private final Instant start;
        private final Instant end;

        private Conflict(String name, String location, Instant start, Instant end) {
            this.name = name;
            this.location = location;
            this.start = start;
            this.end = end;
        }

        public static Function<EventInstanceInfo, Conflict> consF() {
            return e -> new Conflict(
                    e.getEvent().getName(), e.getEvent().getLocation(),
                    e.getEvent().getStartTs(), e.getEvent().getEndTs());
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private Option<String> name = Option.empty();
            private Option<String> location = Option.empty();
            private Option<Instant> start = Option.empty();
            private Option<Instant> end = Option.empty();

            public Conflict build() {
                return new Conflict(name.get(), location.get(), start.get(), end.get());
            }

            public Builder setName(String name) {
                this.name = Option.of(name);
                return this;
            }

            public Builder setLocation(String location) {
                this.location = Option.of(location);
                return this;
            }

            public Builder setStart(Instant start) {
                this.start = Option.of(start);
                return this;
            }

            public Builder setEnd(Instant end) {
                this.end = Option.of(end);
                return this;
            }
        }

        public String getName() {
            return name;
        }

        public String getLocation() {
            return location;
        }

        public Instant getStart() {
            return start;
        }

        public Instant getEnd() {
            return end;
        }
    }

    public static class Builder {
        private Optional<Long> eventId = Optional.empty();
        private Option<String> externalId = Option.empty();
        private Option<Instant> recurrenceId = Option.empty();

        private Option<String> name = Option.empty();
        private Option<String> location = Option.empty();
        private Option<String> description = Option.empty();

        private Option<Instant> start = Option.empty();
        private Option<Instant> end = Option.empty();
        private Option<Boolean> isAllDay = Option.empty();
        private MailRepetitionInfo repetitionInfo = MailRepetitionInfo.EMPTY;

        private Option<MailParticipantsInfo> participants = Option.empty();

        private Option<MailDecision> decision = Option.empty();
        private Option<Notification> notification = Option.empty();

        private Option<Conflict> conflict = Option.empty();
        private Option<String> icsProdId = Option.empty();

        private Option<DateTimeZone> userTz = Option.empty();

        public MailEventInfo build() {
            return new MailEventInfo(eventId, externalId.get(), recurrenceId, name.get(), location.get(), description.getOrElse(""),
                    start.get(), end.get(), isAllDay.get(), repetitionInfo,
                    participants.getOrElse(MailParticipantsInfo.EMPTY),
                    decision.get(), notification, Option.empty(),
                    false, false, false, Cf.list(), Option.empty(), Option.empty(),
                    conflict, icsProdId, userTz.getOrElse(MoscowTime.TZ));
        }

        public Builder setExternalId(String externalId) {
            this.externalId = Option.of(externalId);
            return this;
        }

        public Builder setEventId(Optional<Long> eventId) {
            this.eventId = eventId;
            return this;
        }

        public Builder setRecurrenceId(Instant recurrenceId) {
            this.recurrenceId = Option.of(recurrenceId);
            return this;
        }

        public Builder setName(String name) {
            this.name = Option.of(name);
            return this;
        }

        public Builder setLocation(String location) {
            this.location = Option.of(location);
            return this;
        }

        public Builder setStart(Instant start) {
            this.start = Option.of(start);
            return this;
        }

        public Builder setStartUtc(int year, int month, int day, int hour, int minute) {
            return setStart(new DateTime(year, month, day, hour, minute, DateTimeZone.UTC).toInstant());
        }

        public Builder setEnd(Instant end) {
            this.end = Option.of(end);
            return this;
        }

        public Builder setEndUtc(int year, int month, int day, int hour, int minute) {
            return setEnd(new DateTime(year, month, day, hour, minute, DateTimeZone.UTC).toInstant());
        }

        public Builder setAllDay(boolean allDay) {
            isAllDay = Option.of(allDay);
            return this;
        }

        public Builder setDecision(MailDecision decision) {
            this.decision = Option.of(decision);
            return this;
        }

        public Builder setNotification(Option<Notification> notification) {
            this.notification = notification;
            return this;
        }

        public Builder setNotification(Notification notification) {
            return setNotification(Option.of(notification));
        }

        public Builder setConflict(Option<Conflict> conflict) {
            this.conflict = conflict;
            return this;
        }

        public Builder setConflict(Conflict conflict) {
            return setConflict(Option.of(conflict));
        }

        public Builder setIcsProdId(Option<String> icsProdId) {
            this.icsProdId = icsProdId;
            return this;
        }

        public Builder setUserTz(DateTimeZone userTz) {
            this.userTz = Option.of(userTz);
            return this;
        }

        public void setRepetitionInfo(MailRepetitionInfo repetitionInfo) {
            this.repetitionInfo = repetitionInfo;
        }
    }

    public Option<Instant> getRecurrenceId() {
        return recurrenceId.map(Instant::parse);
    }

    public String getName() {
        return baseInfo.getName();
    }

    public String getLocation() {
        return baseInfo.getLocation();
    }

    public Instant getStart() {
        return baseInfo.getStart();
    }

    public Instant getEnd() {
        return baseInfo.getEnd();
    }

    public boolean isAllDay() {
        return baseInfo.isAllDay();
    }

    public MailRepetitionInfo getRepetitionInfo() {
        return baseInfo.getRepetitionInfo();
    }
}
