package ru.yandex.calendar.frontend.api.mail;

import java.util.Optional;
import java.util.Set;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import net.fortuna.ical4j.model.Property;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Vec2;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.MainEvent;
import ru.yandex.calendar.logic.beans.generated.Rdate;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.contact.ContactRoutines;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventChangesInfo;
import ru.yandex.calendar.logic.event.EventDateTime;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInfo;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceForUpdate;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInstanceStatusChecker;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventsAttachedUser;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.RecurrenceIdOrMainEvent;
import ru.yandex.calendar.logic.event.meeting.UpdateMode;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.WebReplyData;
import ru.yandex.calendar.logic.event.repetition.EventAndRepetition;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.ics.imp.IcsEventDataConverter;
import ru.yandex.calendar.logic.ics.imp.IcsEventImporter;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVEventGroup;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsVTimeZones;
import ru.yandex.calendar.logic.ics.iv5j.ical.property.IcsMethod;
import ru.yandex.calendar.logic.log.EventIdLogDataJson;
import ru.yandex.calendar.logic.log.EventsLogger;
import ru.yandex.calendar.logic.log.change.EventChangeLogEvents;
import ru.yandex.calendar.logic.log.change.EventChangesJson;
import ru.yandex.calendar.logic.log.change.changes.UserRelatedChangesJson;
import ru.yandex.calendar.logic.notification.EventUserWithNotifications;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.participant.ParticipantBasics;
import ru.yandex.calendar.logic.sharing.participant.ParticipantData;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantsData;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.digest.Md5;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

@Slf4j
public class MailEventManager {
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private UserManager userManager;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private EventChangesFinder eventChangesFinder;
    @Autowired
    private ContactRoutines contactRoutines;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;
    @Autowired
    private IcsEventImporter icsEventImporter;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventInstanceStatusChecker eventInstanceStatusChecker;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private EventsLogger eventsLogger;

    public MailEventInfo getEventInfo(MailEventData eventData, ActionInfo actionInfo) {
        PassportUid uid = eventData.getUid();

        Option<EventInfo> event = findEvent(uid, eventData.getExternalId(), false, actionInfo.getActionSource());
        if (!event.isPresent()) {
            EventData converted = MailEventDataConverter.convert(eventData.getExternalId(), eventData);
            return getEventInfoForNewEvent(eventData.getUid(), converted, Option.empty(),
                    Optional.of(eventData.getTimeZone()), Optional.empty(), actionInfo);
        } else {
            return getEventInfoForExistingEvent(uid, event.get(), Option.empty(), Option.empty(),
                    Optional.of(eventData.getTimeZone()), Optional.empty(), actionInfo);
        }
    }

    public MailEventInfoOrRefusal importEventByIcsUrl(PassportUid uid, IcsCalendar calendar, Optional<Language> lang, ActionInfo actionInfo) {
        final Optional<Email> emailByUid = userManager.getEmailByUid(uid);
        if (emailByUid.isEmpty()) {
            log.warn("user with uid:" + uid + " doesn't exists in userManger skipping importEventByIcsUrl");
            return MailEventInfoOrRefusal.refusal("user with uid:" + uid + " is external");
        }
        try {
            val icsEvent = convertIcsEvent(uid, calendar, actionInfo.getActionSource());
            if (icsEvent.isDeleted() || isEventAlreadyInActualState(uid, icsEvent, actionInfo)) {
                return MailEventInfoOrRefusal.info(getEventInfoForIcsEvent(uid, icsEvent, lang, actionInfo));
            }
            icsImporter.importIcsStuff(uid, calendar, IcsImportMode.mailWidget(uid, actionInfo));
            return MailEventInfoOrRefusal.info(loadEventInfoByIcsUrlFromDataBase(uid, calendar, lang, actionInfo));
        } catch (MailEventException e) {
            return MailEventInfoOrRefusal.refusal(e.getMessage());
        }
    }

    public MailEventInfo loadEventInfoByIcsUrlFromDataBase(PassportUid uid, IcsCalendar calendar, Optional<Language> lang, ActionInfo actionInfo) {
        val icsEvent = convertIcsEvent(uid, calendar, actionInfo.getActionSource());
        if (icsEvent.isDeleted() || isEventExists(uid, icsEvent)) {
            return getEventInfoForIcsEvent(uid, icsEvent, lang, actionInfo);
        }
        throw new MailEventException("Event not found");
    }

    public MailEventInfo parseEvent(PassportUid uid, IcsCalendar calendar, Optional<Language> language, ActionInfo actionInfo) {
        val vevent = getAcceptableGroup(calendar).getMasterOrFirstEvent();
        val eventTz = dateTimeManager.getTimeZoneForUid(uid);
        val tzs = IcsVTimeZones.cons(calendar.getTimezones(), eventTz, vevent.isRepeatingOrRecurrence());
        val eventData = IcsEventDataConverter.convert(tzs, vevent, true);
        val icsEvent = new IcsEvent(calendar, vevent, Optional.empty(), eventData, false);
        return getEventInfoForIcsEvent(uid, icsEvent, language, actionInfo);
    }

    public MailEventInfoOrRefusal getEventInfoByIcsUrl(
            PassportUid uid, IcsCalendar calendar, Optional<Language> language, ActionInfo actionInfo
    ) {
        try {
            return MailEventInfoOrRefusal.info(loadEventInfoByIcsUrlFromDataBase(uid, calendar, language, actionInfo));
        } catch (MailEventException e) {
            try {
                return MailEventInfoOrRefusal.info(parseEvent(uid, calendar, language, actionInfo));
            } catch (MailEventException internalE) {
                return MailEventInfoOrRefusal.refusal(internalE.getMessage());
            }
        }
    }

    private boolean isEventExists(PassportUid uid, IcsEvent icsEvent) {
        return icsEvent.getFoundEvent()
                .map(EventInfo::getEventWithRelations)
                .stream()
                .flatMap(eventWithRelations -> eventWithRelations.getEventUsers().stream())
                .map(EventUser::getUid)
                .anyMatch(uid::equals);
    }

    private boolean isEventAlreadyInActualState(PassportUid uid, IcsEvent icsEvent, ActionInfo actionInfo) {
        val userIsInTheEvent = isEventExists(uid, icsEvent);
        val tzs = IcsVTimeZones.cons(
                icsEvent.getIcs().getTimezones(), icsEvent.getConvertedTz(), icsEvent.getVevent().isRepeatingOrRecurrence());
        val isAlreadyUpdated = icsEvent.getFoundEvent().map(event ->
                eventInstanceStatusChecker.getStatusBySubjectAndSyncData(UidOrResourceId.user(uid), IcsUtils.createIcsSynchData(icsEvent.getVevent(), tzs),
                        event.getEventWithRelations(), actionInfo.getActionSource()).isAlreadyUpdated())
                .orElse(false);
        return isAlreadyUpdated && userIsInTheEvent;
    }

    public MailEventInfoOrRefusal createOrUpdateEventByIcsUrl(
            PassportUid uid, IcsCalendar calendar, MailEventIcsData eventData,
            Option<Language> lang, ActionInfo actionInfo)
    {
        try {
            val icsEvent = convertIcsEvent(uid, calendar, actionInfo.getActionSource());

            if (icsEvent.isDeleted()) {
                return MailEventInfoOrRefusal.info(getEventInfoForIcsEvent(uid, icsEvent, lang.toOptional(), actionInfo));
            }

            icsImporter.importIcsStuff(uid, icsEvent.getIcs(), IcsImportMode.mailWidget(
                    uid, eventData.getDecision().toDecision(), actionInfo));

            val icsEventReloaded = convertIcsEvent(uid, icsEvent.getIcs(), actionInfo.getActionSource());
            return MailEventInfoOrRefusal.info(getEventInfoForIcsEvent(uid, icsEventReloaded, lang.toOptional(), actionInfo));
        } catch (MailEventException e) {
            return MailEventInfoOrRefusal.refusal(e.getMessage());
        }
    }

    public MailEventInfo createOrUpdateEvent(final MailEventData eventData, final ActionInfo actionInfo) {
        PassportUid uid = eventData.getUid();
        LockResource lock = LockResource.event(eventData.getExternalId());

        Tuple2<MailEventInfo, ListF<Long>> res = lockTransactionManager.lockAndDoInTransaction(lock, () -> {
            EventData data = MailEventDataConverter.convert(eventData.getExternalId(), eventData);
            Option<EventInfo> event = findEvent(uid, eventData.getExternalId(), true, actionInfo.getActionSource());

            EventInfo result = !event.isPresent()
                    ? createEvent(uid, data, actionInfo)
                    : updateEvent(uid, event.get(), data, actionInfo);

            return Tuple2.tuple(
                    getEventInfoForExistingEvent(uid, result, Option.empty(), Option.empty(),
                            Optional.of(eventData.getTimeZone()), Optional.empty(), actionInfo),
                    result.getEventWithRelations().getLayerIds());
        });
        xivaNotificationManager.notifyLayersUsersAboutEventsChange(res.get2(), actionInfo);

        return res.get1();
    }

    public IncomingIcsMessageInfoOrRefusal getIncomingIcsMessageInfo(
            PassportUid uid, IcsCalendar calendar, Option<Language> langO
    ) {
        try {
            val veventGroup = getAcceptableGroup(calendar);
            val event = veventGroup.getMasterOrFirstEvent();

            val method = calendar.getMethod();
            val mailType = calendar.getXYandexMailType().filterMap(MailType.R::valueOfO).toOptional();

            val type = getIncomingIcsMessageType(method, mailType);

            val resources = resourceRoutines.getDomainResourcesCanViewWithLayersAndOffices(
                    uid, OfficeFilter.any(), ResourceFilter.byEmail(event.getParticipantEmailsSafe()));

            val resourcesSorted = resources.isEmpty()
                    ? resources
                    : resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(uid, resources);

            val tzs = IcsVTimeZones.cons(
                    calendar.getTimezones(), dateTimeManager.getTimeZoneForUid(uid), event.isRepeatingOrRecurrence());

            val interval = new InstantInterval(event.getDtStart().get().getInstant(tzs), event.getDtEnd().get().getInstant(tzs));

            val incomingIcsMessageInfo = new IncomingIcsMessageInfo(Optional.empty(), event.getSummary().orElse(""), event.getLocation().orElse(""),
                    resourcesSorted, interval, Optional.of(type), langO.getOrElse(() -> settingsRoutines.getLanguage(uid)));
            return IncomingIcsMessageInfoOrRefusal.info(incomingIcsMessageInfo);
        } catch (MailEventException e) {
            return IncomingIcsMessageInfoOrRefusal.refusal(e.getMessage());
        }
    }

    private IncomingIcsMessageInfo.Type getIncomingIcsMessageType(IcsMethod method, Optional<MailType> mailType) {
        if (method.methodIs(IcsMethod.REQUEST)) {
            switch (mailType.orElse(MailType.EVENT_INVITATION)) {
                case EVENT_INVITATION:
                    return IncomingIcsMessageInfo.Type.INVITE;
                case EVENT_UPDATE:
                    return IncomingIcsMessageInfo.Type.UPDATE;
                case EVENT_CANCEL:
                    return IncomingIcsMessageInfo.Type.CANCEL;
            }
        } else if (method.methodIs(IcsMethod.CANCEL)) {
            if (mailType.isEmpty() || mailType.stream().anyMatch(MailType.EVENT_CANCEL::equals)) {
                return IncomingIcsMessageInfo.Type.CANCEL;
            }
        }
        throw new MailEventException("unexpected-mail-type");
    }

    public OkResultOrRefusal updateDecision(
            PassportUid uid, String externalId, Option<Instant> recurrenceId,
            MailDecision decision, ActionInfo actionInfo)
    {
        LockResource lock = LockResource.event(externalId);

        Tuple2<OkResultOrRefusal, ListF<Long>> result = lockTransactionManager.lockAndDoInTransaction(lock, () -> {
            Option<MainEvent> main = eventRoutines.getMainEventBySubjectAndExternalId(
                    UidOrResourceId.user(uid), externalId);

            Option<Event> event = main.flatMapO(me -> eventRoutines.findEventByMainEventIdAndRecurrence(
                    me.getId(), new RecurrenceIdOrMainEvent(recurrenceId)));

            Option<UserParticipantInfo> invitation = event.flatMapO(e -> eventInvitationManager.getUserEventSharing(
                    e.getId(), ParticipantId.yandexUid(uid)));

            if (!invitation.isPresent()) {
                return Tuple2.tuple(OkResultOrRefusal.refusal("event-not-found"), Cf.list());
            }
            if (invitation.get().getDecision() == decision.toDecision()) {
                return Tuple2.tuple(OkResultOrRefusal.refusal("already-decided"), Cf.list());
            }

            EventsAttachedUser attach = eventInvitationManager.handleEventInvitationDecision(
                    uid, invitation.get(), new WebReplyData(decision.toDecision()),
                    event.get().getRepetitionId().isPresent(), actionInfo);

            Lazy<Email> userEmail = Lazy.withSupplier(() ->
                    settingsRoutines.getSettingsByUid(attach.uid.getOrElse(uid)).getEmail());

            attach.events.forEach(e -> eventsLogger.log(EventChangeLogEvents.updated(
                    ActorId.user(uid), new EventIdLogDataJson(externalId, e.event),
                    EventChangesJson.empty().withEvent(e.eventChanges)
                            .withUsersAndLayers(UserRelatedChangesJson.find(userEmail.get(), e))), actionInfo));

            return Tuple2.tuple(OkResultOrRefusal.ok(), attach.getLayers().getAllLayerIds());
        });
        xivaNotificationManager.notifyLayersUsersAboutEventsChange(result.get2(), actionInfo);

        return result.get1();
    }

    private EventInfo createEvent(PassportUid uid, EventData eventData, ActionInfo actionInfo) {
        long mainEventId = eventRoutines.createMainEvent(uid, eventData, actionInfo);

        CreateInfo createInfo = eventRoutines.createUserOrFeedEvent(
                UidOrResourceId.user(uid), EventType.USER, mainEventId, eventData,
                NotificationsData.createFromMail(eventData.getEventUserData().getNotifications()),
                InvitationProcessingMode.SAVE_ONLY, actionInfo);

        return eventInfoDbLoader.getEventInfosByEvents(
                Option.of(uid), Cf.list(createInfo.getEvent()), actionInfo.getActionSource()).single();
    }

    private EventInfo updateEvent(PassportUid uid, EventInfo eventInfo, EventData eventData, ActionInfo actionInfo) {
        EventInstanceForUpdate instance = eventRoutines.getEventInstanceForModifier(
                Option.of(uid), Option.empty(), eventInfo, Option.empty());

        val userInfo = userManager.getUserInfo(uid);

        NotificationsData.Update notifications =
                NotificationsData.updateFromMail(eventData.getEventUserData().getNotifications());

        EventChangesInfo changes = eventChangesFinder.getEventChangesInfo(
                instance, eventData, notifications, Option.of(uid), false, false).onlyPerUserAndEventInstanceChanges();

        val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(userInfo, eventInfo.getEventWithRelations());
        val canEditEvent = authorizer.canEditEvent(userInfo, eventAuthInfo, ActionSource.notTrusted());

        if (changes.wasNonPerUserFieldsChange() && !canEditEvent) {
            changes = changes.onlyPerUserChanges();
        }
        eventRoutines.updateEventCommon(Option.of(uid), instance, changes, UpdateMode.NORMAL, actionInfo);

        EventAndRepetition updatedEvent = eventDbManager.getEventAndRepetitionByIdForUpdate(instance.getEventId());

        if (changes.timeOrRepetitionChanges()) {
            eventDbManager.updateEventLayersAndResourcesIndents(updatedEvent, actionInfo);
        }

        val updated = eventInfoDbLoader.getEventInfosByEventsAndRepetitions(Option.of(userInfo),
                Option.of(updatedEvent), actionInfo.getActionSource()).single();

        eventsLogger.log(EventChangeLogEvents.updated(
                ActorId.user(uid), instance.getEventWithRelations(), instance.getRepetitionInstanceInfo(),
                updated.getEventWithRelations(), updated.getRepetitionInstanceInfo()), actionInfo);

        return updated;
    }

    private Option<EventInfo> findEvent(PassportUid uid, String externalId, boolean forUpdate, ActionSource actionSource) {
        Option<Event> event = eventRoutines.findMasterEventBySubjectIdAndExternalId(
                UidOrResourceId.user(uid), externalId, forUpdate);

        if (event.isPresent()) {
            UserInfo user = userManager.getUserInfo(uid);
            EventInfo eventInfo = loadEventInfo(uid, event.get(), actionSource);
            val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, eventInfo.getEventWithRelations());
            authorizer.ensureCanViewEvent(user, eventAuthInfo, actionSource);
            return Option.of(eventInfo);
        } else {
            return Option.empty();
        }
    }

    private Optional<EventInfo> findEvent(
            PassportUid uid, MainEvent mainEvent, Option<Instant> recurrenceId, ActionSource actionSource)
    {
        val event = eventRoutines.findEventByMainEventIdAndRecurrence(
                mainEvent.getId(), recurrenceId.isPresent()
                        ? RecurrenceIdOrMainEvent.recurrenceId(recurrenceId.get())
                        : RecurrenceIdOrMainEvent.mainEvent()).toOptional();

        return event.map(value -> loadEventInfo(uid, value, actionSource));
    }

    private EventInfo loadEventInfo(PassportUid uid, Event event, ActionSource actionSource) {
        return eventInfoDbLoader
                .getEventInfosByEvents(Option.of(uid), Cf.list(event), actionSource)
                .single();
    }

    private IcsEvent convertIcsEvent(PassportUid uid, IcsCalendar calendar, ActionSource actionSource) {
        val veventGroup = getAcceptableGroup(calendar);
        val vevent = veventGroup.getMasterOrFirstEvent();

        val mainEvent = icsEventImporter.findMainEvent(uid, veventGroup, actionSource);

        val eventTz = mainEvent.map(MainEvent.getTimezoneIdF().andThen(AuxDateTime.getVerifyDateTimeZoneF()))
                .getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));

        val tzs = IcsVTimeZones.cons(calendar.getTimezones(), eventTz, vevent.isRepeatingOrRecurrence());

        val eventData = IcsEventDataConverter.convert(tzs, vevent, true);
        val recurrenceId = eventData.getEvent().getRecurrenceId();

        val deleted = new IcsEvent(calendar, vevent, Optional.empty(), eventData, true);

        if (mainEvent.isPresent()) {
            val event = findEvent(uid, mainEvent.get(), recurrenceId, actionSource);
            if (event.stream().anyMatch(e -> eventInvitationManager.isDeletedYandexUserEventParticipant(e.getEventId(), uid))) {
                return deleted;
            }

            if (recurrenceId.isPresent()) {
                val master = eventRoutines.findMasterEventByMainId(mainEvent.get().getId());
                if (master.isPresent()) {
                    val repetitionWithoutRecurrences =
                            repetitionRoutines.getRepetitionInstanceInfosAmongEvents(master).values().single();

                    if (repetitionWithoutRecurrences.getExdateStarts().containsTs(recurrenceId.get())) {
                        return deleted;
                    }
                    if (!RepetitionUtils.isValidStart(repetitionWithoutRecurrences, recurrenceId.get())) {
                        throw new MailEventException("invalid-occurrence");
                    }
                }
            }
            return new IcsEvent(calendar, vevent, event, eventData, false);
        } else if (icsEventImporter.isMeetingWithDeletedMaster(uid, veventGroup, actionSource)) {
            return deleted;
        }
        return new IcsEvent(calendar, vevent, Optional.empty(), eventData, false);
    }

    public static IcsVEventGroup getAcceptableGroup(IcsCalendar calendar) {
        return getAcceptableGroup(calendar, Set.of(IcsMethod.REQUEST, IcsMethod.PUBLISH, IcsMethod.CANCEL));
    }

    public static IcsVEventGroup getAcceptableGroup(IcsCalendar calendar, Set<IcsMethod> supportedMethods) {
        val methods = calendar.getProperties(Property.METHOD);
        if (!methods.isEmpty() && StreamEx.of(methods)
                .filter(supportedMethods::contains).count() != 1) {
            throw new MailEventException("unsupported-ics-method");
        }

        val groups = calendar.getEventsGroupedByUid();
        if (groups.size() != 1) {
            throw new MailEventException(groups.isEmpty() ? "no-events" : "more-than-one-event");
        }

        val veventGroup = groups.get(0);
        if (!veventGroup.getUid().isPresent()) {
            throw new MailEventException("missing-ics-uid");
        }

        return veventGroup;
    }

    private MailEventInfo getEventInfoForNewEvent(
            PassportUid uid, EventData event, Option<IcsAttributes> icsAttributes,
            Optional<DateTimeZone> clientTz, Optional<Language> langO, ActionInfo actionInfo)
    {
        val lang = langO.orElse(settingsRoutines.getLanguage(uid));

        Event e = event.getEvent();
        RepetitionInstanceInfo repetitionInfo = new RepetitionInstanceInfo(
                new InstantInterval(e.getStartTs(), e.getEndTs()), event.getTimeZone(),
                Option.of(event.getRepetition()).filter(Cf2.f1B(RepetitionRoutines::isNoneRepetition).notF()),
                event.getRdates().filter(Rdate::getIsRdate),
                event.getRdates().filter(Cf2.f1B(Rdate::getIsRdate).notF()), Cf.list());

        val repetition = convertRepetition(repetitionInfo, icsAttributes, lang);

        MailParticipantsInfo participants = mergeParticipants(
                uid, Option.empty(), Option.of(event.getInvData().getParticipantsData()), lang);

        boolean omitConflict = icsAttributes.isPresent();

        Option<EventInstanceInfo> conflict = !omitConflict && repetition.isEmpty() && !event.getEvent().getIsAllDay()
                ? findFirstAcceptedEventInstance(uid, event.getInterval().get(), actionInfo.getActionSource())
                : Option.<EventInstanceInfo>empty();

        boolean isNoRsvp = event.getInvData().getOrganizerEmail().filterMap(userManager::getUidByEmail).isSome(uid);
        boolean isPast = isPast(repetitionInfo, actionInfo);

        Vec2<EventDateTime> startEnd = new Vec2<>(e.getStartTs(), e.getEndTs())
                .map(ts -> EventDateTime.cons(ts, e.getIsAllDay(), event.getTimeZone()));

        val userTz = clientTz.orElse(dateTimeManager.getTimeZoneForUid(uid));
        return new MailEventInfo(Optional.empty(),
                event.getExternalId().get(), event.getEvent().getRecurrenceId(),
                e.getName(), e.getLocation(), e.getFieldValueO(EventFields.DESCRIPTION).getOrElse(""),
                startEnd.get(0).toInstant(userTz), startEnd.get(1).toInstant(userTz), e.getIsAllDay(),
                repetition, participants, MailDecision.UNDECIDED, Option.empty(),
                icsAttributes.map(IcsAttributes::getInstanceKey),
                icsAttributes.map(IcsAttributes::isCancelled).getOrElse(false),
                isNoRsvp, isPast, getEventActions(Decision.UNDECIDED, isNoRsvp, isPast, icsAttributes),
                Option.empty(), Option.empty(),
                conflict.map(MailEventInfo.Conflict.consF()), icsAttributes.flatMapO(IcsAttributes::getProdId), userTz);
    }

    private MailEventInfo getEventInfoForExistingEvent(
            PassportUid uid, EventInfo event, Option<EventData> data, Option<IcsAttributes> icsAttributes,
            Optional<DateTimeZone> clientTz, Optional<Language> langO, ActionInfo actionInfo)
    {
        val lang = langO.orElse(settingsRoutines.getLanguage(uid));

        Option<EventUserWithNotifications> eventUser = event.getEventUserWithNotifications();
        Decision decision = eventUser.map(EventUserWithNotifications.getDecisionF()).getOrElse(Decision.UNDECIDED);

        val repetitionInfo = event.getRepetitionInstanceInfo();
        val repetition = convertRepetition(repetitionInfo, icsAttributes, lang);

        MailParticipantsInfo participants = mergeParticipants(
                uid, Option.of(event.getParticipants()),
                data.map(d -> d.getInvData().getParticipantsData()), lang);

        Option<Notification> notification = eventUser.isPresent()
                ? chooseNotification(eventUser.get())
                : Option.<Notification>empty();

        boolean omitConflict = icsAttributes.isPresent();

        Option<EventInstanceInfo> conflict = !omitConflict && repetition.isEmpty() && decision == Decision.UNDECIDED
                ? findConflictingEvent(uid, event.getEvent(), actionInfo.getActionSource())
                : Option.<EventInstanceInfo>empty();

        boolean isNoRsvp = event.getParticipants().getOrganizerIdSafe()
                .getOrElse(ParticipantId.yandexUid(uid)).isYandexUserWithUid(uid);

        Event e = event.getEvent();
        boolean isPast = isPast(repetitionInfo, actionInfo);

        Vec2<EventDateTime> startEnd = new Vec2<>(e.getStartTs(), e.getEndTs())
                .map(ts -> EventDateTime.cons(ts, e.getIsAllDay(), event.getTimezone()));

        val userTz = clientTz.orElse(dateTimeManager.getTimeZoneForUid(uid));
        return new MailEventInfo(Optional.of(e.getId()),
                event.getMainEvent().getExternalId(), event.getRecurrenceId(),
                e.getName(), e.getLocation(), e.getDescription(),
                startEnd.get(0).toInstant(userTz), startEnd.get(1).toInstant(userTz),
                e.getIsAllDay(), repetition,
                participants, MailDecision.fromDecision(decision), notification,
                icsAttributes.map(IcsAttributes::getInstanceKey),
                icsAttributes.map(IcsAttributes::isCancelled).getOrElse(false),
                isNoRsvp, isPast, getEventActions(decision, isNoRsvp, isPast, icsAttributes),
                icsAttributes.<String>filterMap(IcsAttributes::getCalendarMailType),
                Option.of(svcRoutines.getEventPageUrlPrefixForUid(uid) + e.getId()),
                conflict.map(MailEventInfo.Conflict.consF()), icsAttributes.flatMapO(IcsAttributes::getProdId), userTz);
    }

    // CAL-7026
    private ListF<String> getEventActions(
            Decision decision, boolean isNoRsvp, boolean isPast, Option<IcsAttributes> icsAttributes)
    {
        if (isPast || isNoRsvp || icsAttributes.exists(IcsAttributes::isCancelled)) {
            return Cf.list();
        }
        return Option.when(decision != Decision.YES, "accept")
                .plus(Option.when(decision != Decision.NO, "decline"));
    }

    private boolean isPast(RepetitionInstanceInfo repetitionInfo, ActionInfo actionInfo) {
        return RepetitionUtils.getIntervals(
                repetitionInfo.withoutRecurrences(), actionInfo.getNow(), Option.empty(), true, 1).isEmpty();
    }

    private MailRepetitionInfo convertRepetition(RepetitionInstanceInfo repetitionInfo, Option<IcsAttributes> icsAttributes,
                                                 Language lang) {
        if (icsAttributes.exists(IcsAttributes::isCalendarCancelMail) &&
                icsAttributes.exists(Cf2.f1B(IcsAttributes::isIcsCancel).notF()))  {
            return new MailRepetitionInfo(repetitionInfo, true, lang);
        }
        if (icsAttributes.exists(IcsAttributes::isCancelled)) {
            return new MailRepetitionInfo(repetitionInfo.withDueTs(Option.empty()), lang);
        }
        return new MailRepetitionInfo(repetitionInfo, lang);
    }

    private MailEventInfo getEventInfoForIcsEvent(
            PassportUid uid, IcsEvent event, Optional<Language> lang, ActionInfo actionInfo)
    {
        boolean isCancelled = event.isDeleted()
                || event.getIcs().getMethod().methodIs(IcsMethod.CANCEL)
                || event.getIcs().getXYandexMailType().isSome("event_cancel");

        Option<String> calendarMailType = event.getIcs().getXYandexMailType();

        Md5.Sum key = Md5.A.digest("" + event.getVevent().getUid() + event.getConverted().getEvent().getRecurrenceId());

        IcsAttributes attributes = new IcsAttributes(
                event.getIcs().getMethod(), isCancelled, key.base64(), calendarMailType, event.getIcs().getProdId());

        if (event.getFoundEvent().isEmpty()) {
            return getEventInfoForNewEvent(
                    uid, event.getConverted(), Option.of(attributes), Optional.empty(), lang, actionInfo);
        } else {
            return getEventInfoForExistingEvent(
                    uid, event.getFoundEvent().get(), Option.of(event.getConverted()),
                    Option.of(attributes), Optional.empty(), lang, actionInfo
            );
        }
    }

    private Option<Notification> chooseNotification(EventUserWithNotifications eventUser) {
        if (eventUser.getEventUser().getDecision().goes()) {
            return eventUser.getNotifications().getNotifications()
                    .filter(n -> NotificationsData.mailChannels.containsTs(n.getChannel()))
                    .minByO(Notification::getOffset);
        }
        return Option.empty();
    }

    private Option<EventInstanceInfo> findConflictingEvent(PassportUid uid, Event event, ActionSource actionSource) {
        if (event.getIsAllDay()) {
            return Option.empty();
        } else {
            InstantInterval interval = new InstantInterval(event.getStartTs(), event.getEndTs());
            return findFirstAcceptedEventInstance(uid, interval, actionSource);
        }
    }

    private MailParticipantsInfo mergeParticipants(
            PassportUid uid, Option<Participants> existingParticipants,
            Option<ParticipantsData> participantsData, Language lang)
    {
        Participants oldParticipants = existingParticipants.orElseGet(Participants::notMeeting);
        ParticipantsData newParticipants = participantsData.orElseGet(ParticipantsData::notMeeting);

        Function<Email, ParticipantId> newIdByEmailF = eventInvitationManager
                .getParticipantIdsByEmails(newParticipants.getParticipantEmailsSafe()).toMap()::getOrThrow;

        final MapF<ParticipantId, ParticipantInfo> oldById = oldParticipants
                .getParticipantsSafe().toMapMappingToKey(ParticipantInfo.getIdF());

        final MapF<ParticipantId, ParticipantData> newById = newParticipants
                .getParticipantsSafe().toMapMappingToKey(ParticipantData.getEmailF().andThen(newIdByEmailF));

        ListF<Email> emails = oldParticipants.getParticipantEmailsSafe().plus(newParticipants.getParticipantEmailsSafe());

        final MapF<Email, NameI18n> nameByEmail = Cf2.flatBy2(contactRoutines.getI18NamesByEmails(uid, emails)).toMap();

        SetF<ParticipantId> addedIds = newById.keySet().minus(oldById.keySet());

        final MapF<PassportUid, Settings> addedSettingsByUid = settingsRoutines.getSettingsCommonByUidIfExistsBatch(
                addedIds.filterMap(ParticipantId.getUidIfYandexUserF()));

        final MapF<Long, Resource> addedResourcesById = resourceRoutines.getResourcesByIds(
                addedIds.filterMap(ParticipantId.getResourceIdIfResourceF())).toMapMappingToKey(Resource.getIdF());

        Function<ParticipantId, MailParticipantInfo> infoF = new Function<ParticipantId, MailParticipantInfo>() {
            public MailParticipantInfo apply(ParticipantId id) {
                Option<ParticipantInfo> oldInfo = oldById.getO(id);
                Option<ParticipantData> newData = newById.getO(id);

                ParticipantBasics oldOrNew = Cf.<ParticipantBasics>list().plus(oldInfo).plus(newData).first();
                ParticipantBasics newOrOld = Cf.<ParticipantBasics>list().plus(newData).plus(oldInfo).first();

                MailDecision decision = MailDecision.fromDecision(oldOrNew.getDecision());
                Email email = oldOrNew.getEmail();
                String name = newOrOld.getName(); // new first cause we hide names in public, but ics doesn't

                Option<Resource> resource = id.isResource() && oldInfo.isPresent()
                        ? Option.of(((ResourceParticipantInfo) oldInfo.get()).getResource())
                        : id.getResourceIdIfResource().filterMap(addedResourcesById::getO);

                Option<Settings> settings = id.isYandexUser() && oldInfo.isPresent()
                        ? Option.of(((YandexUserParticipantInfo) oldInfo.get()).getSettingsCommon())
                        : id.getUidIfYandexUser().filterMap(addedSettingsByUid::getO);

                if (resource.isPresent()) {
                    return new MailParticipantInfo.Resource(
                            ResourceRoutines.getNameI18n(resource.get(), lang).getOrElse(name), email, decision,
                            resource.get().getExchangeName().getOrElse(""));
                }
                if (settings.isPresent()) {
                    return new MailParticipantInfo.YandexUser(
                            nameByEmail.getO(email).map(NameI18n.getNameF(lang)).getOrElse(name), email, decision,
                            settings.get().getUserLogin().getOrElse(""));
                }
                return new MailParticipantInfo.ExternalUser(
                        nameByEmail.getO(email).map(NameI18n.getNameF(lang)).getOrElse(name), email, decision);
            }
        };
        Option<ParticipantId> organizerId = existingParticipants.isPresent()
                ? existingParticipants.get().getOrganizerIdSafe()
                : newParticipants.getOrganizerEmailSafe().map(newIdByEmailF);

        ListF<ParticipantId> attendeeIds = existingParticipants.isPresent()
                ? existingParticipants.get().getAllAttendeesButNotOrganizerIdsSafe()
                : newParticipants.getAttendeesButNotOrganizerEmailsSafe().map(newIdByEmailF)
                .filter(ParticipantId.isResourceF().notF()); // CAL-5630

        return new MailParticipantsInfo(organizerId.map(infoF), attendeeIds.map(infoF));
    }

    private Option<EventInstanceInfo> findFirstAcceptedEventInstance(PassportUid uid, InstantInterval interval,
            ActionSource actionSource)
    {
        ListF<EventInstanceInfo> events = eventRoutines.getSortedInstancesIMayView(
                Option.of(uid), interval.getStart(), Option.of(interval.getEnd()),
                LayerIdPredicate.allForUser(uid, false), actionSource);

        Function1B<EventInstanceInfo> isAcceptedF = eventInstanceInfo -> {
            Option<EventUser> eventUser = eventInstanceInfo.getEventUser();
            return eventUser.isPresent() && eventUser.get().getDecision().goes();
        };

        Function1B<EventInstanceInfo> isNotAllDayF =
                Function1B.asFunction1B(EventInstanceInfo.getEventF().andThen(Event.getIsAllDayF())).notF();

        return events.filter(isAcceptedF.andF(isNotAllDayF)).firstO();
    }

    private static class IcsAttributes {
        private final IcsMethod method;
        private final boolean isCancelled;
        private final String instanceKey;
        private final Option<String> calendarMailType;
        private final Option<String> prodId;

        private IcsAttributes(
                IcsMethod method, boolean isCancelled, String instanceKey,
                Option<String> calendarMailType, Option<String> prodId)
        {
            this.method = method;
            this.isCancelled = isCancelled;
            this.instanceKey = instanceKey;
            this.calendarMailType = calendarMailType;
            this.prodId = prodId;
        }

        public boolean isCancelled() {
            return isCancelled;
        }

        public String getInstanceKey() {
            return instanceKey;
        }

        public Option<String> getCalendarMailType() {
            return calendarMailType;
        }

        public boolean isIcsCancel() {
            return method.methodIs(IcsMethod.CANCEL);
        }

        public boolean isCalendarCancelMail() {
            return getCalendarMailType().isSome("event_cancel");
        }

        public Option<String> getProdId() {
            return prodId;
        }
    }
}
