package ru.yandex.calendar.frontend.api.staff;

import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.api.XmlOrJsonApiActionSupport;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.dates.InstantIntervalSet;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author Daniel Brylev
 *
 * @see ru.yandex.calendar.frontend.kiosk.KioskManager
 *
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetRoomsState extends XmlOrJsonApiActionSupport {

    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private DateTimeManager dateTimeManager;

    @RequestParam
    private Email clientEmail;
    @RequestListParam
    private ListF<String> exchangeNames;

    @Override
    protected XmlOrJsonWritable doExecute() {
        final ListF<RoomState> rooms = getRoomsState(clientEmail, exchangeNames);

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                w.startArray("rooms");
                for (RoomState room : rooms) {
                    room.write(w);
                }
                w.endArray();
            }
        };
    }

    private ListF<RoomState> getRoomsState(Email clientEmail, ListF<String> exchangeNames) {
        PassportUid uid = userManager.getUidByEmail(clientEmail).getOrThrow("unknown user " + clientEmail);
        Check.isTrue(uid.isYandexTeamRu(), "not yandex-team user " + clientEmail);

        Function<ResourceInfo, String> exchangeNameF = r -> r.getResource().getExchangeName().get();
        ListF<ResourceInfo> resources = resourceDao.findResourceInfosByExchangeNames(exchangeNames);

        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(uid);
        Instant now = Instant.now();
        Instant endOfDay = new LocalDate(now, tz).plusDays(1).toDateTimeAtStartOfDay().toInstant();

        ListF<ResourceInfo> activeResources = resources.filter(ResourceInfo.isActiveF());

        ListF<RoomState> freeBusy = resourceScheduleManager
                .getResourceBusyIntervals(Option.of(uid), activeResources, now, endOfDay, tz, getActionInfo())
                .map(freeBusyRoomStateF(now, endOfDay));

        ListF<RoomState> notActive = resources
                .filter(ResourceInfo.isActiveF().notF()).map(exchangeNameF.andThen(RoomState.notActiveF()));
        ListF<RoomState> notFound = exchangeNames
                .filter(resources.map(exchangeNameF).containsF().notF()).map(RoomState.notFoundF());

        return exchangeNames.map(freeBusy.plus(notActive).plus(notFound)
                .toMapMappingToKey(RoomState.getExchangeNameF())::getOrThrow);
    }

    private static Function2<ResourceInfo, InstantIntervalSet, RoomState> freeBusyRoomStateF(
            final Instant now, final Instant endOfDay)
    {
        return new Function2<ResourceInfo, InstantIntervalSet, RoomState>() {
            public RoomState apply(ResourceInfo r, InstantIntervalSet is) {
                String exchangeName = r.getResource().getExchangeName().get();
                Duration minGapDuration = Duration.standardMinutes(10);
                Option<InstantInterval> nearest = is.mergeNeighborsCloserThan(minGapDuration).getIntervals().firstO();

                if (!nearest.isPresent()) {
                    return RoomState.free(exchangeName, new Duration(now, endOfDay));
                } else if (nearest.get().getStart().isAfter(now.plus(minGapDuration))) {
                    return RoomState.free(exchangeName, new Duration(now, nearest.get().getStart()));
                } else {
                    return RoomState.busy(exchangeName, new Duration(now, nearest.get().getEnd()));
                }
            }
        };
    }
}
