package ru.yandex.calendar.frontend.api.staff;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.api.XmlOrJsonApiActionSupport;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervals;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.UserOrResourceAvailabilityIntervals;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Check;

/**
 * @link http://st.yandex-team.ru/ISEARCH-1593
 *
 * @author dbrylev
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetUsersFreeBusyState extends XmlOrJsonApiActionSupport {

    @Autowired
    private UserManager userManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;

    @RequestParam
    private Email clientEmail;
    @RequestListParam
    private ListF<Email> emails;

    @Override
    protected XmlOrJsonWritable doExecute() {
        final ListF<UserFreeBusyState> users = getUsersFreeBusyState(clientEmail, emails);

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                w.startArray("users");
                for (UserFreeBusyState user : users) {
                    user.write(w);
                }
                w.endArray();
            }
        };
    }

    private ListF<UserFreeBusyState> getUsersFreeBusyState(Email clientEmail, ListF<Email> emails) {
        PassportUid uid = userManager.getUidByEmail(clientEmail).getOrThrow("unknown user " + clientEmail);
        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(uid);

        Check.isTrue(uid.isYandexTeamRu(), "not yandex-team user " + clientEmail);

        Instant now = Instant.now();
        ListF<Email> usersEmails = Cf2.flatBy2(userManager.getUidsByEmails(emails)).get1();

        AvailabilityRequest request = AvailabilityRequest.interval(now, now)
                .excludeAbsencesEvents().includeEventsResources();

        Tuple2List<Email, Option<PassportUid>> emailUids = userManager.getUidsByEmails(usersEmails);
        MapF<PassportUid, Email> emailByUid = Cf2.flatBy2(emailUids).invert().toMap();
        ListF<PassportUid> knownUids = emailByUid.keys();

        ListF<UserOrResourceAvailabilityIntervals> intervals = availRoutines.getAvailabilityIntervalss(
                uid, knownUids, Cf.<Resource>list(), request, getActionInfo());

        Tuple2List<PassportUid, AvailabilityIntervals> knownIntervals = Cf2.flatBy2(intervals.toTuple2List(
                UserOrResourceAvailabilityIntervals.getUidF(),
                UserOrResourceAvailabilityIntervals.getIntervalsOF()));

        MapF<Email, UserFreeBusyState> knownStates = knownIntervals.zip3With(userFreeBusyStateF(tz, emailByUid))
                .map1(emailByUid::getOrThrow).toMap(Tuple3::get1, Tuple3::get3);

        return emails.map(e -> knownStates.getO(e).getOrElse(() -> UserFreeBusyState.unknown(e)));
    }

    private Function2<PassportUid, AvailabilityIntervals, UserFreeBusyState> userFreeBusyStateF(
            final DateTimeZone tz, final MapF<PassportUid, Email> emailByUid)
    {
        return new Function2<PassportUid, AvailabilityIntervals, UserFreeBusyState>() {
            public UserFreeBusyState apply(PassportUid uid, AvailabilityIntervals is) {
                ListF<AvailabilityInterval> intervals = is.unmerged();

                if (intervals.isEmpty()) return UserFreeBusyState.free(emailByUid.getOrThrow(uid));

                AvailabilityInterval interval = intervals.min(
                        AvailabilityInterval.getAvailabilityF().andThen(Availability.moreBusyFirstComparator())
                                .thenComparing(AvailabilityInterval.getEndF().andThenNaturalComparator().reversed()));

                Option<ResourceInfo> room = resourceRoutines.sortResourcesFromUserOfficeAndCityFirst(
                        uid, interval.getResources()).firstO();
                EventLocation location = new EventLocation(room, interval.getEventLocation());

                Instant busyUntil = interval.getInterval().getEnd();

                return UserFreeBusyState.busy(emailByUid.getOrThrow(uid), busyUntil.toDateTime(tz), location);
            }
        };
    }
}
