package ru.yandex.calendar.frontend.api.todo;

import org.joda.time.DateTimeZone;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.CalendarDataSourceStatus;
import ru.yandex.calendar.frontend.a3.interceptors.WithTlTimeoutByProduct;
import ru.yandex.calendar.frontend.api.XmlOrJsonApiActionSupport;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.update.UpdateLock2;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;

/**
 * @author Daniel Brylev
 *
 */
@WithTlTimeoutByProduct(pub=2000L, yt=2000L)
public abstract class ApiTodoUserAction extends XmlOrJsonApiActionSupport {

    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private UpdateLock2 updateLock2;
    @Autowired
    private CalendarDataSourceStatus dataSourceStatus;

    @RequestParam
    private PassportUid uid;

    private Option<Language> language = Option.empty(); // CAL-4780

    @RequestParam(required = false)
    public void setLocale(String code) {
        this.language = Language.fromValueSafe(code);
    }

    protected Option<Long> findFirstUserCreatedTodoListOrCreateNewIfNotMaya() {
        if (getActionInfo().getActionSource() == ActionSource.WEB_MAYA) {
            return todoRoutines.findFirstCreatedTodoListId(uid);

        } else if (dataSourceStatus.isMasterUnavailable()) {
            return Option.empty();

        } else {
            return Option.of(MasterSlaveContextHolder.withPolicy(
                    MasterSlavePolicy.RW_M, this::findFirstUserCreatedTodoListOrCreateNew));
        }
    }

    protected long findFirstUserCreatedTodoListOrCreateNew() {
        Function0<Long> findOrCreate = () -> todoRoutines.findFirstCreatedListOrCreateNewWithName(
                getUid(), TodoUtils.getDefaultListName(getLanguage()), getActionInfo());

        return todoRoutines.findFirstCreatedTodoListId(uid).getOrElse(() -> updateLock2.isAcquired()
                ? findOrCreate.apply()
                : lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(uid), findOrCreate));
    }

    /**
     * @see ru.yandex.calendar.frontend.a3.interceptors.ActionWithLockAndTransaction
     */
    public ListF<LockResource> getResourcesToLock() {
        return Cf.list(LockResource.todoUser(getUid()));
    }

    public PassportUid getUid() {
        return uid;
    }

    public Option<Language> getLanguage() {
        return language;
    }

    public DateTimeZone getUserTz() {
        return dateTimeManager.getTimeZoneForUid(getUid());
    }

    public void setUid(long uid) {
        this.uid = PassportUid.cons(uid);
    }
}
