package ru.yandex.calendar.frontend.api.todo;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarUtils;
import ru.yandex.calendar.frontend.a3.converters.ConverterToIntColor;
import ru.yandex.calendar.frontend.a3.interceptors.ActionWithLockAndTransaction;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author gutman
 */
public class CreateTodoItemAction extends ApiActionWithIdOrExternalId implements ActionWithLockAndTransaction {
    private static final Logger logger = LoggerFactory.getLogger(CreateTodoItemAction.class);

    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private TodoDao todoDao;

    @RequestParam
    private Option<String> title;

    @RequestParam(customConverter = ConverterToIntColor.class, ignoreEmpty = true)
    private Option<Integer> color;
    @RequestParam
    private Option<Long> timestampPosition = Option.empty();
    @RequestParam
    private Option<String> mailMessageId = Option.empty();
    @RequestParam
    private Option<String> mailSubject = Option.empty();
    @RequestParam(required = false)
    private boolean today = false;
    @RequestParam(required = false)
    private boolean thisWeek = false;
    @RequestParam
    private Option<Boolean> archived = Option.empty();
    @RequestParam
    private Option<Boolean> completed = Option.empty();

    @RequestParam(value = "notificationTs", ignoreEmpty = true)
    private Option<String> notificationTsUnixtime = Option.empty();
    @RequestParam(value = "notificationDateTime", ignoreEmpty = true)
    private Option<String> notificationDateTime = Option.empty();

    @RequestParam(value = "dueTs", ignoreEmpty = true)
    private Option<String> dueTsUnixtime = Option.empty();
    @RequestParam(value = "dueDate", ignoreEmpty = true)
    private Option<String> dueDate = Option.empty();

    @RequestParam(required = false)
    private boolean failIfNoList = false;

    @Override
    protected XmlOrJsonWritable doExecute() {
        TodoItem todo = new TodoItem();

        todo.setTitle(title.getOrElse(TodoUtils.getDefaultItemName(getLanguage())));

        todo.setColor(color);

        Option<Instant> dueTs = TodoTimeConverter.convertOnCreate(dueTsUnixtime, dueDate, getUserTz());
        todo.setDueTs(dueTs);

        todo.setIsAllDay(TodoTimeConverter.isAllDay(dueTsUnixtime, dueDate));

        Validate.V.isFalse(today && thisWeek);
        todo.setIsToday(today);
        todo.setIsThisWeek(thisWeek);

        Option<ListIdOrExternalId> listId = getListIdOrExternalIdO();

        if (listId.isPresent()) {
            Option<Long> todoListId = todoDao.findNotDeletedTodoListIdById(listId.get());

            if (!todoListId.isPresent() && !failIfNoList) {
                logger.info("Todo list " + listId + " already deleted, creating todo in default list");
                todo.setTodoListId(findFirstUserCreatedTodoListOrCreateNew());

            } else if (todoListId.isPresent()) {
                todo.setTodoListId(todoListId.get());

            } else {
                throw CommandRunException.createSituation(
                        "Todo list not found or deleted for uid = " + getUid() + ", list id = " + listId.get(),
                        Situation.TODO_LIST_NOT_FOUND);
            }
        } else {
            todo.setTodoListId(findFirstUserCreatedTodoListOrCreateNew());
        }

        // XXX: synchronize with pos
        if (timestampPosition.isPresent()) {
            todo.setTimestampPosition(timestampPosition.get());
        } else {
            todo.setTimestampPosition(getActionInfo().getNow().getMillis());
        }

        Option<TodoIdOrExternalId> todoIdOrExternalId = getTodoIdOrExternalIdO();
        if (todoIdOrExternalId.isPresent()) {
            Validate.isFalse(todoIdOrExternalId.get().isId());
            todo.setExternalId(todoIdOrExternalId.get().getExternalId());
        } else {
            todo.setExternalId(CalendarUtils.generateExternalId());
        }

        todo.setMailMessageId(mailMessageId);
        todo.setMailSubject(mailSubject);

        if (completed.isPresent() && completed.get()) {
            todo.setCompletionTs(getActionInfo().getNow());
        }

        if (archived.isPresent() && archived.get()) {
            todo.setArchived(true);
        }

        Option<Instant> notificationTs = TodoTimeConverter.convertOnCreate(
                notificationTsUnixtime, notificationDateTime, getUserTz());
        todo.setNotificationTs(notificationTs);

        todo.setCreationTs(getActionInfo().getNow());

        final long todoListId = todo.getTodoListId();
        final long id = todoRoutines.checkPermissionsAndCreateTodoItem(getUid(), todo, getActionInfo());

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                w.startObject("todo-item");
                w.addTextField("id", id);
                w.addTextField("list-id", todoListId);
                w.addTextField("creation-ts", getActionInfo().getNow().getMillis() / 1000);
                w.endObject();
            }
        };
    }
}
