package ru.yandex.calendar.frontend.api.todo;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.converters.ConverterToInstantUnixtime;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoListsAndTodoItems;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;

/**
 * @author gutman
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetAllTodoListsAndTodoItemsAction extends ApiTodoUserAction {

    @Autowired
    private TodoDao todoDao;
    @Autowired
    private TodoRoutines todoRoutines;

    @RequestParam(value = "modifiedSince", customConverter = ConverterToInstantUnixtime.class)
    private Option<Instant> modifiedSinceO = Option.empty();
    @RequestParam(value = "queryId", customConverter = ConverterToInstantUnixtime.class)
    private Option<Instant> queryIdO = Option.empty();
    @RequestParam("skip")
    private Option<Integer> skipO = Option.empty();
    @RequestParam("count")
    private Option<Integer> countO = Option.empty();

    @Override
    protected XmlOrJsonWritable doExecute() {
        long nowTs = System.currentTimeMillis() / 1000;

        Instant lastUpdateTs = todoRoutines.getTodoLastUpdateTs(getUid());

        int skip = this.skipO.getOrElse(0);
        int count = countO.getOrElse(50);

        if (skip > 0) {
            Instant queryId = queryIdO.getOrThrow("Must specify query-id to use skip");

            if (lastUpdateTs.isAfter(queryId)) {
                return XmlOrJsonWritable.textFields("something-changed-restart-query", "", "now", Long.toString(nowTs));
            }
        }

        Option<Long> defaultListId = findFirstUserCreatedTodoListOrCreateNewIfNotMaya();
        long queryId = queryIdO.getOrElse(lastUpdateTs).getMillis() / 1000;

        XmlOrJsonWritable w = XmlOrJsonWritable.textFields(
                "now", Long.toString(nowTs), "query-id", Long.toString(queryId));

        if (modifiedSinceO.isPresent()) {
            return w.chainTo(getModifiedTodoListsAndModifiedTodos(skip, count, modifiedSinceO.get(), defaultListId, getUserTz()));
        } else {
            return w.chainTo(getAllTodoListsAndAllTodos(skip, count, getUid(), defaultListId, getUserTz()));
        }
    }

    private XmlOrJsonWritable getModifiedTodoListsAndModifiedTodos(
            int skip, int count, Instant modifiedSince, Option<Long> defaultListId, final DateTimeZone tz)
    {
        ListF<TodoList> todoLists = todoDao.findNotDeletedUserTodoLists(getUid());

        XmlOrJsonWritable w = XmlOrJsonWritable.empty();

        if (skip == 0) {
            // first response always contains completed todos, deleted todos and deleted lists
            final ListF<TodoItem> completedTodoItems = todoDao.findNotDeletedCompletedTodosByTodoListsAndMinCompletionTs(
                    todoLists.map(TodoListFields.ID.getF()), modifiedSince);

            final ListF<TodoList> deletedTodoLists = todoDao.findDeletedTodoListsByUidAndMinDeletedTs(getUid(), modifiedSince);

            final ListF<TodoItem> deletedTodoItems = todoDao.findDeletedTodoItemsByTodoListIdsAndMinDeletedTs(
                    todoLists.map(TodoListFields.ID.getF()), modifiedSince);

            w = new XmlOrJsonWritable() {
                public void write(XmlOrJsonWriter w) {
                    w.startArray("completed-todo-item-ids");
                    for (TodoItem todo : completedTodoItems) {
                        w.addTextField("id", todo.getId());
                    }
                    w.endArray();

                    w.startArray("deleted-todo-item-ids");
                    for (TodoItem todo : deletedTodoItems) {
                        w.addTextField("id", todo.getId());
                    }
                    w.endArray();

                    w.startArray("deleted-todo-list-ids");
                    for (TodoList list : deletedTodoLists) {
                        w.addTextField("id", list.getId());
                    }
                    w.endArray();
                }
            };
        }
        final TodoListsAndTodoItems todoListsAndTodoItems = todoRoutines
                .getTodoListsAndNotCompletedTodoItems(skip, count, getUid(), Option.of(modifiedSince));

        return w.chainTo(new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                writeTodoListsAndTodoItems(w, todoListsAndTodoItems, defaultListId, tz);
            }
        });
    }

    private XmlOrJsonWritable getAllTodoListsAndAllTodos(
            int skip, int count, PassportUid uid, Option<Long> defaultListId, final DateTimeZone tz)
    {
        final TodoListsAndTodoItems todoListsAndTodoItems = todoRoutines
                .getTodoListsAndNotCompletedTodoItems(skip, count, uid, Option.<Instant>empty());

        return new XmlOrJsonWritable() {
            public void write(XmlOrJsonWriter w) {
                writeTodoListsAndTodoItems(w, todoListsAndTodoItems, defaultListId, tz);
            }
        };
    }


    private void writeTodoListsAndTodoItems(
            XmlOrJsonWriter w, TodoListsAndTodoItems todoListsAndTodoItems, Option<Long> defaultListId, DateTimeZone tz)
    {
        w.startArray("todo-lists-and-todo-items");

        for (TodoList todoList : todoListsAndTodoItems.getTodoLists()) {
            w.startObject("list-or-item");
            w.addTextField("type", "todo-list");
            TodoSerializer.serializeTodoListFields(w, todoList, defaultListId.isSome(todoList.getId()));
            w.endObject();
        }

        for (TodoItem todoItem : todoListsAndTodoItems.getTodoItems()) {
            w.startObject("list-or-item");
            w.addTextField("type", "todo-item");
            TodoSerializer.serializeTodoItemFields(w, todoItem, getOutputMode(), tz);
            w.endObject();
        }

        w.endArray();

        w.addTextField("total", todoListsAndTodoItems.getTotalTodoLists() + todoListsAndTodoItems.getTotalTodoItems());
    }
}
