package ru.yandex.calendar.frontend.api.todo;

import lombok.Data;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.util.dates.ExtendedDateTime;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.db.q.SqlOrderElement;
import ru.yandex.misc.lang.Validate;

/**
 * @author gutman
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetTodoItemsAction extends ApiActionWithIdOrExternalId {

    @Autowired
    private TodoDao todoDao;

    @RequestParam(required = false)
    private boolean today = false;
    @RequestParam(required = false)
    private boolean thisWeek = false;
    @RequestParam(required = false)
    private boolean onlyNotCompleted = false;
    @RequestParam
    private Option<Boolean> onlyNotCompletedUndue;
    @RequestParam
    private Option<String> dueFrom = Option.empty();
    @RequestParam
    private Option<String> dueTo = Option.empty();
    @RequestParam
    private Option<String> iterationKey;
    @RequestParam
    private Option<Boolean> newFirst;
    @RequestParam
    private Option<Boolean> soonestFirst;
    @RequestParam
    private Option<Boolean> undueFirst;
    @RequestParam
    private Option<Integer> count;

    @Override
    protected XmlOrJsonWritable doExecute() {
        Validate.V.isFalse(today && thisWeek);

        DateTimeZone userTz = getUserTz();

        SqlCondition overallCond;

        ExtendedDateTime d = new ExtendedDateTime(getActionInfo().getNow(), userTz);
        if (today) {
            SqlCondition toBeDoneToday = TodoItemFields.DUE_TS.ge(d.toInstantAtStartOfDay())
                    .and(TodoItemFields.DUE_TS.lt(d.toInstantAtEndOfDay()));
            SqlCondition overdue = TodoItemFields.DUE_TS.lt(d.toInstantAtStartOfDay())
                    .and(TodoItemFields.COMPLETION_TS.column().isNull());

            overallCond = SqlCondition.any(toBeDoneToday, overdue);

        } else if (thisWeek) {
            SqlCondition toBeDoneThisWeek = TodoItemFields.DUE_TS.ge(d.toInstantAtStartOfWeek())
                    .and(TodoItemFields.DUE_TS.lt(d.toInstantAtEndOfWeek()));
            SqlCondition overdue = TodoItemFields.DUE_TS.lt(d.toInstantAtStartOfWeek())
                    .and(TodoItemFields.COMPLETION_TS.column().isNull());

            overallCond = SqlCondition.any(toBeDoneThisWeek, overdue);

        } else {
            overallCond = SqlCondition.trueCondition()
                    .and(dueFrom.map(ts -> TodoItemFields.DUE_TS.ge(TodoTimeConverter.parseDateTime(ts, userTz))))
                    .and(dueTo.map(ts -> TodoItemFields.DUE_TS.lt(TodoTimeConverter.parseDateTime(ts, userTz))));
        }

        if (soonestFirst.isPresent() && !undueFirst.isPresent() && overallCond.isConstantTrue()) {
            undueFirst = Option.of(soonestFirst.get());
        }

        SqlCondition dueIsNullCond = TodoItemFields.DUE_TS.column().isNull();

        SqlCondition dueIsNotNullCond = TodoItemFields.DUE_TS.column().isNotNull();

        SqlCondition undueCond = onlyNotCompletedUndue.isSome(true)
                ? dueIsNullCond.and(TodoItemFields.COMPLETION_TS.column().isNull())
                : dueIsNullCond;

        if (undueFirst.isPresent() || onlyNotCompletedUndue.isPresent()) {
            overallCond = overallCond.or(undueCond);
        }
        if (onlyNotCompleted) {
            overallCond = overallCond.and(TodoItemFields.COMPLETION_TS.column().isNull());
        }

        SqlCondition iterationCond = overallCond;

        if (iterationKey.isPresent()) {
            IterationKey key = IterationKey.parse(iterationKey.get());

            SqlCondition idCondition = newFirst.isSome(true)
                    ? TodoItemFields.ID.lt(key.lastItemId)
                    : TodoItemFields.ID.gt(key.lastItemId);

            if (soonestFirst.isSome(true) && key.lastDueTs.isPresent()) {
                idCondition = TodoItemFields.DUE_TS.gt(key.lastDueTs.get())
                        .or(TodoItemFields.DUE_TS.eq(key.lastDueTs.get()).and(idCondition));

            } else if (soonestFirst.isSome(false) && key.lastDueTs.isPresent()) {
                idCondition = TodoItemFields.DUE_TS.lt(key.lastDueTs.get())
                        .or(TodoItemFields.DUE_TS.eq(key.lastDueTs.get()).and(idCondition));
            }

            if (undueFirst.isSome(true)) {
                iterationCond = iterationCond.and(key.lastDueTs.isPresent()
                        ? idCondition.and(dueIsNotNullCond)
                        : idCondition.and(undueCond).or(dueIsNotNullCond));

            } else if (undueFirst.isSome(false)) {
                iterationCond = iterationCond.and(!key.lastDueTs.isPresent()
                        ? idCondition.and(undueCond)
                        : idCondition.and(dueIsNotNullCond).or(undueCond));

            } else {
                iterationCond = iterationCond.and(idCondition);
            }
        }

        ListF<SqlOrderElement> orders = Cf.list();

        if (undueFirst.isPresent()) {
            orders = orders.plus(new SqlOrderElement(dueIsNullCond.sql(), undueFirst.isSome(false)));
        }
        if (soonestFirst.isPresent()) {
            orders = orders.plus(new SqlOrderElement(TodoItemFields.DUE_TS.column().name(), soonestFirst.get()));
        }
        if (count.isPresent() || iterationKey.isPresent() || newFirst.isPresent() || undueFirst.isPresent()) {
            orders = orders.plus(new SqlOrderElement(TodoItemFields.ID.column().name(), !newFirst.isSome(true)));
        }

        SqlLimits limits = count.map(cnt -> SqlLimits.first(cnt + 1)).getOrElse(SqlLimits::all);

        ListF<TodoItem> notDeletedItems = todoDao.findNotDeletedNotArchivedUserTodoItems(
                getUid(), iterationCond, new SqlOrder(orders), limits);
        final MapF<Long, TodoList> todoListsById = todoDao.findNotDeletedUserTodoLists(getUid())
                .toMapMappingToKey(TodoListFields.ID.getF());
        final ListF<TodoItem> foundTodoItems = notDeletedItems.filter(todoItem -> todoListsById.getTs(todoItem.getTodoListId()) != null);

        ListF<TodoItem> todoItems = count.isPresent() ? foundTodoItems.take(count.get()) : foundTodoItems;

        Option<IterationKey> nextIterationKey = Option.when(
                todoItems.size() != foundTodoItems.size(), () -> IterationKey.of(todoItems.last()));

        Option<Long> totalCount;

        if (count.isPresent() || iterationKey.isPresent()) {
            if (!iterationKey.isPresent() && !nextIterationKey.isPresent()) {
                totalCount = Option.of((long) todoItems.size());
            } else {
                totalCount = Option.of(todoDao.findNotDeletedNotArchivedTodoItemsCount(getUid(), overallCond));
            }
        } else {
            totalCount = Option.empty();
        }

        XmlOrJsonWritable commonWriter = w -> {
            nextIterationKey.forEach(k -> w.addTextField("iteration-key", k.serialize()));
            totalCount.forEach(c -> w.addNumberField("total-count", c));

            w.addTextField("tz", userTz.getID());
        };

        if (getOutputMode() == OutputMode.MOBILE) {
            return w -> {
                {
                    w.startArray("todo-items");

                    for (TodoItem todoItem : todoItems) {
                        TodoSerializer.serializeTodoItem(w, todoItem, getOutputMode(), userTz);
                    }
                    w.endArray();
                }
                commonWriter.write(w);
            };
        } else {
            return w -> {
                {
                    w.startArray("todo-items");

                    for (TodoItem todoItem : todoItems) {
                        TodoSerializer.serializeTodoItemWithTodoListInfo(
                                w, todoItem, todoListsById.getTs(todoItem.getTodoListId()), userTz);
                    }
                    w.endArray();
                }
                commonWriter.write(w);
            };
        }
    }

    @Data
    private static class IterationKey {
        private final long lastItemId;
        private final Option<Instant> lastDueTs;

        public static IterationKey of(TodoItem item) {
            return new IterationKey(item.getId(), item.getDueTs());
        }

        public static IterationKey parse(String value) {
            try {
                String[] ps = value.split("-");

                return ps.length > 1
                        ? new IterationKey(Long.parseLong(ps[0], 16), Option.of(new Instant(Long.parseLong(ps[1], 16))))
                        : new IterationKey(Long.parseLong(ps[0], 16), Option.empty());

            } catch (RuntimeException e) {
                throw new IllegalParameterException("iterationKey", ExceptionUtils.getAllMessages(e));
            }
        }

        public String serialize() {
            return Long.toString(lastItemId, 16) + lastDueTs.map(ts -> "-" + Long.toString(ts.getMillis(), 16)).mkString("");
        }
    }
}
