package ru.yandex.calendar.frontend.api.todo;

import java.util.function.Supplier;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Vec3;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.beans.generated.TodoListFields;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.util.dates.ExtendedDateTime;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;

/**
 * @author dbrylev
 */
@WithMasterSlavePolicy(MasterSlavePolicy.R_MS)
public class GetTodoSidebarAction extends ApiTodoUserAction {

    @Autowired
    private TodoDao todoDao;

    @RequestParam
    private Option<Integer> count;
    @RequestParam
    private Option<String> nextPlanned;
    @RequestParam
    private Option<String> nextExpired;
    @RequestParam
    private Option<String> nextCompleted;

    @Override
    protected XmlOrJsonWritable doExecute() {
        count.forEach(c -> ValidateParam.gt("count", c, 0));

        ValidateParam.isTrue("next", nextPlanned.size() + nextExpired.size() + nextCompleted.size() < 2,
                "Only one next parameter expected to be specified at once");

        Vec3<Option<LoadedTab>> items;

        if (nextPlanned.isPresent()) {
            items = new Vec3<>(Option.of(loadTodoItems(TodoType.PLANNED)), Option.empty(), Option.empty());

        } else if (nextExpired.isPresent()) {
            items = new Vec3<>(Option.empty(), Option.of(loadTodoItems(TodoType.EXPIRED)), Option.empty());

        } else if (nextCompleted.isPresent()) {
            items = new Vec3<>(Option.empty(), Option.empty(), Option.of(loadTodoItems(TodoType.COMPLETED)));

        } else {
            items = new Vec3<>(
                    Option.of(loadTodoItems(TodoType.PLANNED)),
                    Option.of(loadTodoItems(TodoType.EXPIRED)),
                    Option.of(loadTodoItems(TodoType.COMPLETED)));
        }

        ListF<Long> moreListIds = items.flatMap(tab -> tab.flatMap(LoadedTab::getListIdsFromItems))
                .filterNot(items.flatMap(tab -> tab.flatMap(LoadedTab::getListIds)).unique()::containsTs);

        ListF<TodoList> lists = items.flatMap(tab -> tab.flatMap(LoadedTab::getLists))
                .plus(todoDao.findTodoLists(TodoListFields.ID.column().inSet(moreListIds)))
                .stableUniqueBy(TodoList::getId);

        DateTimeZone tz = getUserTz();

        return w -> {
            Cf.list("planned-items", "expired-items", "completed-items").zipWithIndex()
                    .forEach((type, idx) -> items.get(idx).forEach(res -> {
                        w.startObject(type);

                        w.startArray("items");
                        res.items.forEach(item -> TodoSerializer.serializeTodoItem(w, item, getOutputMode(), tz));
                        w.endArray();

                        res.lists.forEach(lsts -> {
                            w.startArray("list-ids");
                            lsts.forEach(lst -> w.addTextField("id", lst.getId()));
                            w.endArray();
                        });

                        res.nextKey.forEach(key -> w.addTextField("next-key", key.serialize()));
                        w.endObject();
                    }));

            w.startArray("todo-lists");
            lists.forEach(list -> {
                w.startObject("todo-list");
                TodoSerializer.serializeTodoListFields(w, list, false);
                w.endObject();
            });
            w.endArray();
        };
    }

    private Tuple2<ListF<TodoList>, Option<IterationKey>> loadPlannedTodoLists() {
        SqlCondition cond = TodoListFields.CREATOR_UID.eq(getUid())
                .and(TodoListFields.DELETED.eq(false))
                .and(nextPlanned.map(IterationKey::parse).map(k -> TodoListFields.ID.le(k.curListId)));

        SqlOrder order = SqlOrder.orderByColumnDesc(TodoListFields.ID.column());

        SqlLimits limits = count.map(c -> SqlLimits.first(c + 1)).getOrElse(SqlLimits::all);

        ListF<TodoList> listsFound = todoDao.findTodoLists(cond, order, limits);

        ListF<TodoList> lists = count.map(listsFound::take).getOrElse(listsFound);

        Option<IterationKey> nextKey = count.filter(c -> c < listsFound.size())
                .map(c -> new IterationKey(listsFound.last().getId(), Option.empty(), 0));

        return Tuple2.tuple(lists, nextKey);
    }

    private LoadedTab loadTodoItems(TodoType type) {
        Option<IterationKey> key = Option.empty();
        Option<Tuple2<ListF<TodoList>, Option<IterationKey>>> plannedLists = Option.empty();

        if (type == TodoType.PLANNED) {
            key = nextPlanned.map(k -> IterationKey.parseParameter("plannedKey", k));
            plannedLists = Option.of(loadPlannedTodoLists());

        } else if (type == TodoType.EXPIRED) {
            key = nextExpired.map(k -> IterationKey.parseParameter("expiredKey", k));

        } else if (type == TodoType.COMPLETED) {
            key = nextCompleted.map(k -> IterationKey.parseParameter("completedKey", k));
        }

        ListF<TodoItem> itemsFound = todoDao.findNotDeletedNotArchivedUserTodoItems(
                getUid(), type.condition(key, () -> new DateTime(getActionInfo().getNow(), getUserTz()))
                        .and(plannedLists.map(t -> TodoItemFields.TODO_LIST_ID.column().inSet(t.get1().map(TodoList::getId)))),
                type.order(), count.map(c -> SqlLimits.first(c + 1)).getOrElse(SqlLimits::all));

        ListF<TodoItem> items = count.map(itemsFound::take).getOrElse(itemsFound);

        Option<IterationKey> nextKey = count.filter(c -> c < itemsFound.size())
                .map(c -> new IterationKey(items.last().getTodoListId(), items.last().getDueTs(), items.last().getId()));

        Option<ListF<TodoList>> lists = plannedLists.map(Tuple2::get1);

        if (plannedLists.isPresent()) {
            if (nextKey.isPresent()) {
                Option<Long> lastListId = nextKey.map(k -> k.curListId);

                lists = Option.of(lists.get().takeWhile(l -> l.getId() >= lastListId.get()));
            }
            if (key.exists(k -> k.lastItemId != 0)
                    && lists.get().firstO().exists(l -> !items.firstO().exists(i -> i.getTodoListId() == l.getId())))
            {
                lists = Option.of(lists.get().drop(1));
            }
            nextKey = nextKey.orElse(plannedLists.get().get2());
        }
        return new LoadedTab(items, lists, nextKey);
    }

    private enum TodoType {
        PLANNED,
        EXPIRED,
        COMPLETED,
        ;

        public SqlCondition condition(Option<IterationKey> iterationKey, Supplier<DateTime> now) {
            SqlCondition c = SqlCondition.trueCondition();

            if (this == PLANNED) {
                c = c.and(TodoItemFields.COMPLETION_TS.column().isNull());

            } else if (this == EXPIRED) {
                c = c.and(TodoItemFields.COMPLETION_TS.column().isNull())
                        .and(TodoItemFields.DUE_TS.lt(new ExtendedDateTime(now.get()).toInstantAtStartOfDay()));

            } else if (this == COMPLETED) {
                c = c.and(TodoItemFields.COMPLETION_TS.column().isNotNull());
            }

            return c.and(iterationKey.map(this::iterationCondition));
        }

        public SqlOrder order() {
            SqlOrder order = SqlOrder.orderByColumnDesc(TodoItemFields.TODO_LIST_ID.column().name());

            return this == EXPIRED
                    ? order.andThen(TodoItemFields.DUE_TS.column().name()).andThen(TodoItemFields.ID.column().name())
                    : order.andThen(TodoItemFields.ID.column().name());
        }

        private SqlCondition iterationCondition(IterationKey key) {
            return TodoItemFields.TODO_LIST_ID.lt(key.curListId)
                    .or(TodoItemFields.TODO_LIST_ID.eq(key.curListId).and(this == EXPIRED
                            ? TodoItemFields.DUE_TS.gt(key.lastDueTs)
                                    .or(TodoItemFields.DUE_TS.eq(key.lastDueTs).and(TodoItemFields.ID.gt(key.lastItemId)))
                            : TodoItemFields.ID.gt(key.lastItemId)));
        }
    }

    private static class LoadedTab {
        public final ListF<TodoItem> items;
        public final Option<ListF<TodoList>> lists;
        public final Option<IterationKey> nextKey;

        public LoadedTab(
                ListF<TodoItem> items,
                Option<ListF<TodoList>> lists,
                Option<IterationKey> nextKey)
        {
            this.items = items;
            this.lists = lists;
            this.nextKey = nextKey;
        }

        public ListF<TodoList> getLists() {
            return lists.getOrElse(Cf.list());
        }

        public ListF<Long> getListIds() {
            return lists.getOrElse(Cf.list()).map(TodoList::getId);
        }

        public ListF<Long> getListIdsFromItems() {
            return items.map(TodoItem::getTodoListId);
        }
    }

    private static class IterationKey {
        public final long curListId;
        public final Instant lastDueTs;
        public final long lastItemId;

        public IterationKey(long curListId, Option<Instant> lastDueTs, long lastItemId) {
            this(curListId, lastDueTs.getOrElse(() -> new Instant(0)), lastItemId);
        }

        public IterationKey(long curListId, Instant lastDueTs, long lastItemId) {
            this.curListId = curListId;
            this.lastDueTs = lastDueTs;
            this.lastItemId = lastItemId;
        }

        public String serialize() {
            return curListId + "-" + lastDueTs.getMillis() + "-" + lastItemId;
        }

        public static IterationKey parse(String value) {
            String[] ps = value.split("-");

            return new IterationKey(Long.parseLong(ps[0]), new Instant(Long.parseLong(ps[1])), Long.parseLong(ps[2]));
        }

        public static IterationKey parseParameter(String parameter, String value) {
            try {
                return parse(value);

            } catch (RuntimeException e) {
                throw new IllegalParameterException(parameter, ExceptionUtils.getAllMessages(e));
            }
        }
    }
}
