package ru.yandex.calendar.frontend.api.todo;

import java.util.Locale;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.util.color.ColorUtils;
import ru.yandex.calendar.util.xmlorjson.XmlOrJsonWriter;

/**
 * @author gutman
 */
public class TodoSerializer {

    static void serializeTodoItem(XmlOrJsonWriter w, TodoItem todo, OutputMode outputMode, DateTimeZone tz) {
        if (outputMode == OutputMode.MOBILE) {
            serializeTodoItemMobile(w, todo, tz);
        } else {
            serializeTodoItemFull(w, todo, tz);
        }
    }

    static void serializeTodoItemFields(XmlOrJsonWriter w, TodoItem todo, OutputMode outputMode, DateTimeZone tz) {
        if (outputMode == OutputMode.MOBILE) {
            serializeTodoItemFieldsMobile(w, todo, tz);
        } else {
            serializeTodoItemFieldsFull(w, todo, tz);
        }
    }

    private static void serializeTodoItemFull(XmlOrJsonWriter w, TodoItem todo, DateTimeZone tz) {
        w.startObject("todo-item");

        serializeTodoItemFieldsFull(w, todo, tz);

        w.endObject();
    }

    private static void serializeTodoItemMobile(XmlOrJsonWriter w, TodoItem todo, DateTimeZone tz) {
        w.startObject("todo-item");

        serializeTodoItemFieldsMobile(w, todo, tz);

        w.endObject();
    }

    private static void serializeTodoItemFieldsFull(XmlOrJsonWriter w, TodoItem todo, DateTimeZone tz) {
        w.addTextField("id", todo.getId());
        w.addTextField("external-id", todo.getExternalId());
        w.addTextField("position", todo.getPos());
        w.addTextField("title", todo.getTitle());
        if (todo.getColor().isPresent()) {
            w.addTextField("color", ColorUtils.formatColor(todo.getColor().get()));
        }
        w.addTextField("completed", Boolean.toString(todo.getCompletionTs().isPresent()));
        w.addTextField("creation-ts", todo.getCreationTs().getMillis() / 1000);
        if (todo.getDueTs().isPresent()) {
            w.addTextField("due-ts", todo.getDueTs().get().getMillis() / 1000);
            w.addTextField("due-date", TodoTimeConverter.printNotificationDate(todo.getDueTs().get(), tz));

            if (!todo.getIsAllDay()) {
                w.addTextField("due-date-time", TodoTimeConverter.printNotificationDateTime(todo.getDueTs().get(), tz));
            }
        }

        // deprecated - use timestamp
        w.addTextField("last-update-ts", todo.getLastUpdateTs().getMillis() / 1000);

        w.addTextField("timestamp-position", todo.getTimestampPosition());
        if (todo.getMailMessageId().isPresent()) {
            w.addTextField("mail-message-id", todo.getMailMessageId().get());
        }
        if (todo.getMailSubject().isPresent()) {
            w.addTextField("mail-subject", todo.getMailSubject().get());
        }
        w.addTextField("list-id", todo.getTodoListId());
        if (todo.getNotificationTs().isPresent()) {
            w.addTextField("notification-ts", todo.getNotificationTs().get().getMillis() / 1000);
            w.addTextField("notification-date-time",
                    TodoTimeConverter.printNotificationDateTime(todo.getNotificationTs().get(), tz));
        }

        w.addTextField("timestamp", todo.getLastUpdateTs().getMillis() / 1000);
    }

    private static void serializeTodoItemFieldsMobile(XmlOrJsonWriter w, TodoItem todo, DateTimeZone tz) {
        w.addTextField("id", todo.getId());
        w.addTextField("title", todo.getTitle());
        if (todo.getColor().isPresent()) {
            w.addTextField("color", ColorUtils.formatColor(todo.getColor().get()));
        }
        if (todo.getCompletionTs().isPresent()) {
            w.addTextField("done", "");
        }
        if (todo.getIsToday()) {
            w.addTextField("is-today", "");
        }
        if (todo.getIsThisWeek()) {
            w.addTextField("is-this-week", "");
        }
        if (todo.getDueTs().isPresent()) {
            w.addTextField("due-ts", todo.getDueTs().get().getMillis() / 1000);
            w.addTextField("due-date", TodoTimeConverter.printNotificationDate(todo.getDueTs().get(), tz));
        }
        w.addTextField("pos", todo.getTimestampPosition());
        if (todo.getMailMessageId().isPresent()) {
            w.addTextField("mail-message-id", todo.getMailMessageId().get());
        }
        if (todo.getMailSubject().isPresent()) {
            w.addTextField("mail-subject", todo.getMailSubject().get());
        }
        w.addTextField("external-id", todo.getExternalId());
        w.addTextField("list-id", todo.getTodoListId());
        if (todo.getNotificationTs().isPresent()) {
            w.addTextField("notification-ts", todo.getNotificationTs().get().getMillis() / 1000);
            w.addTextField("notification-date-time",
                    TodoTimeConverter.printNotificationDateTime(todo.getNotificationTs().get(), tz));
        }
        w.addTextField("timestamp", todo.getLastUpdateTs().getMillis() / 1000);
    }

    static void serializeTodoItemsGroupedByMonthAndDay(XmlOrJsonWriter w, ListF<TodoItem> todoItems,
            final DateTimeZone tz, OutputMode outputMode, final Function<TodoItem, Instant> groupBy)
    {
        MapF<LocalDate, ListF<TodoItem>> byMonth = todoItems.groupBy(new Function<TodoItem, LocalDate>() {
            public LocalDate apply(TodoItem todoItem) {
                return groupBy.apply(todoItem).toDateTime(tz).withDayOfMonth(1).toLocalDate();
            }
        });

        w.startArray("months");

        for (Tuple2<LocalDate, ListF<TodoItem>> month : byMonth.entries().sortedBy1()) {
            w.startObject("month");

            w.addTextField("value", month._1.toString("YYYY-MM"));
            w.addTextField("value-rus", month._1.toString("MMMM YYYY", new Locale("RU")));

            writeMonth(month._2, tz, w, outputMode, groupBy);

            w.endObject();
        }

        w.endArray();
    }

    private static void writeMonth(ListF<TodoItem> todoItems, final DateTimeZone tz, XmlOrJsonWriter w,
            OutputMode outputMode, final Function<TodoItem, Instant> groupBy)
    {
        MapF<LocalDate, ListF<TodoItem>> byDay = todoItems.groupBy(new Function<TodoItem, LocalDate>() {
            public LocalDate apply(TodoItem todoItem) {
                return groupBy.apply(todoItem).toDateTime(tz).toLocalDate();
            }
        });

        w.startArray("days");
        for (Tuple2<LocalDate, ListF<TodoItem>> day : byDay.entries().sortedBy1()) {
            w.startObject("day");

            w.addTextField("value", day._1.toString("YYYY-MM-dd"));

            w.startArray("todo-items");
            for (TodoItem todo : day._2) {
                serializeTodoItem(w, todo, outputMode, tz);
            }
            w.endArray();

            w.endObject();
        }
        w.endArray();
    }

    static void serializeTodoListFields(XmlOrJsonWriter w, TodoList todoList, boolean isDefault) {
        w.addTextField("title", todoList.getTitle());
        w.addTextField("description", todoList.getDescription());
        w.addTextField("id", todoList.getId());
        w.addTextField("default", Boolean.toString(isDefault));
        w.addTextField("external-id", todoList.getExternalId());
        w.addTextField("timestamp", todoList.getLastUpdateTs().getMillis() / 1000);
        w.addTextField("creation-ts", todoList.getCreationTs().getMillis() / 1000);
        if (todoList.getColor().isPresent()) {
            w.addTextField("color", ColorUtils.formatColor(todoList.getColor().get()));
        }
    }

    static void serializeTodoItemWithTodoListInfo(XmlOrJsonWriter w, TodoItem todoItem, TodoList todoList, DateTimeZone tz) {
        w.startObject("todo-item");

        serializeTodoItemFieldsFull(w, todoItem, tz);

        w.addTextField("list-title", todoList.getTitle());
        if (todoList.getColor().isPresent()) {
            w.addTextField("list-color", ColorUtils.formatColor(todoList.getColor().get()));
        }

        w.endObject();
    }

    public static void conflictingChange(XmlOrJsonWriter w) {
        w.addTextField("conflicting-change", "");
    }

}
