package ru.yandex.calendar.frontend.api.todo;

import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.converters.ConverterToInstantUnixtime;
import ru.yandex.calendar.frontend.a3.converters.ConverterToIntColor;
import ru.yandex.calendar.frontend.a3.interceptors.ActionWithLockAndTransaction;
import ru.yandex.calendar.frontend.api.XmlOrJsonWritable;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.todo.TodoSynchStatus;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.lang.Validate;

/**
 * @author gutman
 */
public class UpdateTodoItemAction extends ApiActionWithIdOrExternalId implements ActionWithLockAndTransaction {

    @Autowired
    private TodoRoutines todoRoutines;

    @RequestParam
    private Option<String> title = Option.empty();
    @RequestParam(customConverter = ConverterToIntColor.class)
    private Option<Integer> color = Option.empty();
    @RequestParam
    private Option<Boolean> completed = Option.empty();
    @RequestParam
    private Option<Boolean> archived = Option.empty();
    @RequestParam
    private Option<Long> timestampPosition = Option.empty();
    @RequestParam
    private Option<Boolean> today = Option.empty();
    @RequestParam
    private Option<Boolean> thisWeek = Option.empty();
    @RequestParam
    private Option<Boolean> notificationShownInWeb = Option.empty();

    @RequestParam(customConverter = ConverterToInstantUnixtime.class)
    private Option<Instant> timestamp = Option.empty();

    @RequestParam("notificationTs")
    private Option<String> notificationTsUnixtime = Option.empty();
    @RequestParam
    private Option<String> notificationDateTime = Option.empty();

    @RequestParam("dueTs")
    private Option<String> dueTsUnixtime = Option.empty();
    @RequestParam
    private Option<String> dueDate = Option.empty();

    // XXX: ordering
    @Override
    protected XmlOrJsonWritable doExecute() {
        TodoItem todoItem = new TodoItem();
        if (title.isPresent()) todoItem.setTitle(title.get());

        Option<ListIdOrExternalId> listId = getListIdOrExternalIdO();
        if (listId.isPresent()) {
            todoItem.setTodoListId(todoRoutines.getTodoListId(listId.get()));
        }

        if (color.isPresent()) todoItem.setColor(color.get());

        Option<Option<Instant>> dueTs = TodoTimeConverter.convertOnUpdate(dueTsUnixtime, dueDate, getUserTz());
        if (dueTs.isPresent()) {
            if (!dueTs.get().isPresent()) {
                todoItem.setDueTsNull();
            } else {
                todoItem.setDueTs(dueTs.get().get());
                todoItem.setIsAllDay(TodoTimeConverter.isAllDay(dueTsUnixtime, dueDate));
            }
        }

        Option<Option<Instant>> notificationTsForUpdate = TodoTimeConverter.convertOnUpdate(
                notificationTsUnixtime, notificationDateTime, getUserTz());
        if (notificationTsForUpdate.isPresent()) {
            if (!notificationTsForUpdate.get().isPresent()) {
                todoItem.setNotificationTsNull();
            } else {
                todoItem.setNotificationTs(notificationTsForUpdate.get().get());
            }
        }

        if (today.isPresent() && thisWeek.isPresent()) {
            Validate.V.isFalse(today.get() && thisWeek.get());
        }
        if (today.isPresent()) {
            todoItem.setIsToday(today.get());
            todoItem.setIsThisWeek(!today.get());
        }
        if (thisWeek.isPresent()) {
            todoItem.setIsThisWeek(thisWeek.get());
            todoItem.setIsToday(!thisWeek.get());
        }
        if (notificationShownInWeb.isPresent()) {
            todoItem.setNotificationShownInWeb(notificationShownInWeb.get());
        }

        // XXX: synchronize with pos
        if (timestampPosition.isPresent()) todoItem.setTimestampPosition(timestampPosition.get());

        if (todoItem.isEmpty() && !completed.isPresent()) return XmlOrJsonWritable.empty();

        ActionInfo actionInfo = getActionInfo();

        TodoIdOrExternalId todoItemId = getTodoIdOrExternalId();

        TodoSynchStatus synchStatus = todoRoutines.getTodoSynchStatus(
                 getUid(), todoItem, todoItemId, completed, archived, timestamp);

        if (!synchStatus.isFound()) {
            throw CommandRunException.createSituation(
                    "Todo item not found by id " + todoItemId, Situation.TODO_ITEM_NOT_FOUND);
        }

        if (synchStatus.found().wasConflictingChange()) {
            return XmlOrJsonWritable.textField("conflicting-change", "");

        } else if (synchStatus.found().wasChange()) {
            long todoId = synchStatus.found().getTodoItemId();

            final Instant newLastUpdateTs = todoRoutines.checkPermissionsAndUpdateTodoItem(
                    getUid(), todoItem, todoId, actionInfo);


            if (synchStatus.found().completedChanged()) {
                todoRoutines.updateTodoItemSetCompleted(
                        getUid(), todoItemId, synchStatus.found().getNowCompleted(), actionInfo);
            }
            if (synchStatus.found().archivedChanged()) {
                todoRoutines.updateTodoItemSetArchived(
                        todoItemId, synchStatus.found().getNowArchived(), actionInfo);
            }
            if (todoItem.isFieldSet(TodoItemFields.NOTIFICATION_TS)) {
                // XXX notification-ts already updated
                todoRoutines.updateNotificationTs(todoId,
                        todoItem.getFieldValueNullAsNone(TodoItemFields.NOTIFICATION_TS), actionInfo);
            }

            return XmlOrJsonWritable.textField("new-timestamp", newLastUpdateTs.getMillis() / 1000);

        } else {
            return XmlOrJsonWritable.empty();
        }
    }
}
