package ru.yandex.calendar.frontend.caldav.ban;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.worker.StoppableThread;

/**
 * @author gutman
 */
public class CaldavBanManager {
    private static final Logger logger = LoggerFactory.getLogger(CaldavBanManager.class);

    private final Duration banUserInterval;
    private final Duration requestsHistoryInterval;
    private final Duration sleepBetweenBanChecks;

    private final long tooManyRequestsPerIntervalTotal;
    private final long tooManyRequestsPerIntervalPerUser;

    private final QueueWithExpiration<PassportUid> userRequests = new QueueWithExpiration<PassportUid>();
    private final ConcurrentHashMap<PassportUid, UserBan> userBans = new ConcurrentHashMap<PassportUid, UserBan>();

    private final BanUnbanThread banUnbanThread = new BanUnbanThread();

    public CaldavBanManager(Duration banUserInterval, Duration requestsHistoryInterval, Duration sleepBetweenBanChecks,
            long tooManyRequestsPerIntervalTotal, long tooManyRequestsPerIntervalPerUser) {
        this.banUserInterval = banUserInterval;
        this.requestsHistoryInterval = requestsHistoryInterval;
        this.sleepBetweenBanChecks = sleepBetweenBanChecks;
        this.tooManyRequestsPerIntervalTotal = tooManyRequestsPerIntervalTotal;
        this.tooManyRequestsPerIntervalPerUser = tooManyRequestsPerIntervalPerUser;
    }

    @PostConstruct
    public void init() {
        userRequests.init();
        banUnbanThread.start();
    }

    @PreDestroy
    public void destroy() {
        userRequests.destroy();
        banUnbanThread.stopGracefully();
    }

    public UserBanStatus handleUserRequest(PassportUid uid) {
        if (userBanned(uid)) {
            return UserBanStatus.BANNED;
        } else {
            userRequests.add(uid, requestsHistoryInterval);
            return UserBanStatus.NOT_BANNED;
        }
    }

    private boolean userBanned(PassportUid uid) {
        return userBans.containsKey(uid);
    }

    private static class UserBan {
        private final PassportUid uid;
        private final Instant bannedAt;

        private UserBan(PassportUid uid, Instant bannedAt) {
            this.uid = uid;
            this.bannedAt = bannedAt;
        }
    }


    private final class BanUnbanThread extends StoppableThread {
        public BanUnbanThread() {
            super("ban-unbun-thread");
        }

        protected void run1() throws Exception {
            while (!timeToDie) {
                try {
                    banNewUsersIfNeeded();
                    unbanOldUsersIfNeeded();
                    ThreadUtils.sleep(sleepBetweenBanChecks);
                } catch (Throwable t) {
                    if (!timeToDie) {
                        logger.error(t, t);
                        ThreadUtils.sleep(1000); // jic
                    }
                }
            }
        }

        private void banNewUsersIfNeeded() {
            if (userRequests.size() < tooManyRequestsPerIntervalTotal) {
                return;
            }
            PassportUid[] passportUids = userRequests.toArray(new PassportUid[0]);
            MapF<PassportUid, ListF<PassportUid>> requests =
                    Cf.x(passportUids).groupBy(Function.<PassportUid>identityF());

            Instant now = new Instant();

            for (Map.Entry<PassportUid, ListF<PassportUid>> userRequest : requests.entrySet()) {
                if (userRequest.getValue().length() >= tooManyRequestsPerIntervalPerUser) {
                    PassportUid key = userRequest.getKey();
                    userBans.put(key, new UserBan(key, now));
                }
            }
        }

        private void unbanOldUsersIfNeeded() {
            for (UserBan userBan : userBans.values()) {
                if (userBan.bannedAt.plus(banUserInterval).isBefore(new Instant())) {
                    userBans.remove(userBan.uid);
                }
            }
        }
    }

}
