package ru.yandex.calendar.frontend.caldav.ban;

import java.util.Arrays;
import java.util.PriorityQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import org.joda.time.Duration;

import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;
import ru.yandex.misc.worker.StoppableThread;

/**
 * @author gutman
 */
public class QueueWithExpiration<E> {
    private static final Logger logger = LoggerFactory.getLogger(QueueWithExpiration.class);

    private final PriorityQueue<Expiring<E>> queue = new PriorityQueue<Expiring<E>>();

    private final ReentrantLock lock = new ReentrantLock();
    private final Condition newHeadAvailable = lock.newCondition();

    private final ExpirationThread expirationThread = new ExpirationThread();

    public void init() {
        expirationThread.start();
    }

    public void destroy() {
        expirationThread.stopGracefully();
    }

    public boolean add(E e, Duration timeout) {
        long expiresAtMs = System.currentTimeMillis() + timeout.getMillis();
        Expiring<E> ee = new Expiring<E>(e, expiresAtMs);
        lock.lock();
        try {
            Expiring<E> head = queue.peek();
            queue.add(ee);
            if (head == null || ee.expiresAtMs < head.expiresAtMs) {
                newHeadAvailable.signalAll();
            }
            return true;
        } finally {
            lock.unlock();
        }
    }

    private void waitForExpirationAndRemove() throws InterruptedException {
        lock.lockInterruptibly();
        try {
            for (;;) {
                Expiring<E> first = queue.peek();
                if (first == null) {
                    newHeadAvailable.await();
                } else {
                    long expiresInMs = first.expiresAtMs - System.currentTimeMillis();
                    if (expiresInMs > 0) {
                        newHeadAvailable.await(expiresInMs, TimeUnit.MILLISECONDS);
                    } else {
                        queue.remove();
                        return;
                    }
                }
            }
        } finally {
            lock.unlock();
        }
    }

    @SuppressWarnings("unchecked")
    public E[] toArray(E[] es) {
        lock.lock();
        try {
            Expiring<E>[] ees = queue.toArray(new Expiring[0]);
            es = (E[]) Arrays.copyOf(es, ees.length, es.getClass());
            for (int i = 0; i < ees.length; i++) {
                es[i] = ees[i].getValue();
            }
            return es;
        } finally {
            lock.unlock();
        }
    }

    public int size() {
        lock.lock();
        try {
            return queue.size();
        } finally {
            lock.unlock();
        }
    }

    private static class Expiring<E> implements Comparable<Expiring<E>> {
        private final long expiresAtMs;
        private final E e;

        private Expiring(E e, long expiresAtMs) {
            this.e = e;
            this.expiresAtMs = expiresAtMs;
        }

        private E getValue() {
            return e;
        }

        public int compareTo(Expiring<E> that) {
            return (this.expiresAtMs < that.expiresAtMs) ? -1 : ((this.expiresAtMs == that.expiresAtMs) ? 0 : 1);
        }
    }

    private final class ExpirationThread extends StoppableThread {
        public ExpirationThread() {
            super("expiration-thread");
        }

        protected void run1() throws Exception {
            while (!timeToDie) {
                try {
                    waitForExpirationAndRemove();
                } catch (InterruptedException ie) {
                    if (!timeToDie) logger.error(ie, ie);
                } catch (Throwable t) {
                    logger.error(t, t);
                    ThreadUtils.sleep(1000); // jic
                }
            }
        }
    }

}
