package ru.yandex.calendar.frontend.caldav.impl;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Locale;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.CalendarRequestHandle;
import ru.yandex.calendar.RemoteInfo;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.tree.CalendarUrls;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeout.Handle;
import ru.yandex.misc.web.servlet.AccessLogFilter;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author Stepan Koltsov
 *
 * @see AccessLogFilter
 */
public class CaldavFilter implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(CaldavFilter.class);

    private final boolean printDebugInfo;

    public CaldavFilter(boolean printDebugInfo) {
        this.printDebugInfo = printDebugInfo;
    }

    /**
     * @see CaldavJackrabbitServlet#service(HttpServletRequest, HttpServletResponse)
     */
    @SuppressWarnings("deprecation")
    @Override
    public void doFilter(ServletRequest request0, ServletResponse response0, FilterChain chain) throws IOException, ServletException {
        if (!(request0 instanceof HttpServletRequest) || !(response0 instanceof HttpServletResponse)) {
            throw new IllegalArgumentException("unsupported");
        }

        HttpServletRequestX request = HttpServletRequestX.wrap((HttpServletRequest) request0);
        HttpServletResponse response = (HttpServletResponse) response0;

        String rl = request.getRequestLine();

        long start = System.currentTimeMillis();
        boolean ok = false;
        // XXX: too long timeout, see https://jira.yandex-team.ru/browse/CAL-2432
        Handle tltHandle = ThreadLocalTimeout.push(Duration.standardSeconds(60), "CaldavFilter");

        String tag = CalendarUrls.covertToTagForLog(DavHref.fromEncoded(request.getRequestURI()));
        String action = request.getMethod().toLowerCase(Locale.ROOT) + StringUtils.capitalize(tag);

        CalendarRequestHandle handle = CalendarRequest.push(
                new RemoteInfo(request.getXRealIp(), Option.empty()), ActionSource.CALDAV, rl, action);
        try {
            response.setHeader("X-Request-Id", handle.getActionInfo().getRequestIdWithHostId());
            if (printDebugInfo) {
                // XXX print host abbreviations anyway
                response.setHeader("X-Server-Host", HostnameUtils.localHostname());
            }
            chain.doFilter(request, response);
            ok = true;
        } catch (Throwable e) {
            logger.error(rl + " failed; client " + ClientHolder.getSafe() + " using " + request.getUserAgent().getOrElse("?") + ": " + e, e);
            response.setStatus(500, "Internal Server Error; rid=" + RequestIdStack.current().getOrElse("?"));
            response.setContentType("text/plain");

            PrintWriter out;
            try {
                out = new PrintWriter(response.getWriter());
            } catch (IllegalStateException getOutputStreamAlreadyCalled) {
                out = new PrintWriter(new OutputStreamWriter(response.getOutputStream(), "utf-8"));
            }

            out.println("500");
            out.println("rid=" + handle.getActionInfo().getRequestIdWithHostId());
            out.println();

            if (printDebugInfo) {
                e.printStackTrace(out);
            }
        } finally {
            handle.popSafely();
            tltHandle.popSafely();
        }
    }

    public void init(FilterConfig filterConfig) throws ServletException {
    }

    public void destroy() {
    }

}
