package ru.yandex.calendar.frontend.caldav.impl;

import java.util.Arrays;
import java.util.List;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;

@SuppressWarnings("serial")
@Slf4j
public class CaldavJackrabbitServlet extends HttpServlet {
    private static final List<String> AVAILABLE_METHODS = Arrays.asList("get", "delete", "post", "propfind", "put", "proppatch", "report", "options", "mkcalendar");
    private static final List<String> AVAILABLE_URI_PREFIXES = Arrays.asList("", "/", "calendar/dav", "addressbook", "calendars", "principals", "directory");

    @Autowired
    private MeterRegistry registry;
    @Autowired
    private CaldavService caldavService;

    /**
     * @see CaldavFilter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
     */
    @Override
    @SneakyThrows
    protected void service(HttpServletRequest request, HttpServletResponse response) {
        if (AVAILABLE_URI_PREFIXES.contains(normalizeUriForSignal(request)) && AVAILABLE_METHODS.contains(request.getMethod().toLowerCase())) {
            try {
                registry.timer(getRequestSignal("application.request.time.caldav", request))
                        .record(() -> caldavService.execute(request, response));
            } catch (Exception e) {
                log.error("Error on: " + request.getRequestURI(), e);
                response.setStatus(500);
                response.sendError(JackrabbitUtils.translateToStatusCode(e));
            } finally {
                val signalName = getRequestSignal("application.request.caldav", request) + "." + response.getStatus();
                registry.counter(signalName).increment();
            }
        } else {
            response.setStatus(404);
        }
    }

    private static String normalizeUriForSignal(HttpServletRequest request) {
        String uri = request.getRequestURI();
        if (uri.startsWith("/")){
            uri = uri.substring(1);
        }
        if (uri.endsWith("/")) {
            uri = uri.substring(0, uri.length() - 1);
        }
        val slashIndex = uri.indexOf('/');
        if (slashIndex != -1) {
            uri = uri.substring(0, slashIndex);
        }
        return uri;
    }

    public static String getRequestSignal(String signalPrefix, HttpServletRequest request) {
        val method = request.getMethod().toLowerCase();
        val uri = normalizeUriForSignal(request);
        return StreamEx.of(signalPrefix, method, uri).joining(".");
    }
}
