package ru.yandex.calendar.frontend.caldav.impl;

import java.util.Optional;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.SneakyThrows;
import lombok.val;
import org.apache.jackrabbit.webdav.DavLocatorFactory;
import org.apache.jackrabbit.webdav.DavSessionProvider;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.caldav.proto.AbstractCaldavServlet;
import ru.yandex.calendar.frontend.caldav.proto.CaldavLocatorFactory;
import ru.yandex.calendar.frontend.caldav.proto.CaldavSessionProvider;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.facade.AuthenticationFailedException;
import ru.yandex.calendar.frontend.caldav.proto.facade.CcCalendarFacade;
import ru.yandex.calendar.frontend.caldav.proto.facade.CcUserInfo;
import ru.yandex.calendar.frontend.caldav.proto.tree.CaldavResourceFactory;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.log.requests.RequestsLogger;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.inside.passport.oauth.OAuthUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

public class CaldavServiceImpl extends AbstractCaldavServlet implements CaldavService  {
    private static final Logger logger = LoggerFactory.getLogger(CaldavServiceImpl.class);

    @Autowired
    private CcCalendarFacade ccCalendarFacade;
    @Autowired
    private CaldavResourceFactory caldavResourceFactory;
    @Autowired
    private CaldavLocatorFactory davLocatorFactory;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    public CaldavServiceImpl() {
        JackrabbitHacks.configureXmlParserForSecurity();
    }

    @Override
    public DavSessionProvider getDavSessionProvider() {
        return new CaldavSessionProvider();
    }

    @Override
    public DavLocatorFactory getLocatorFactory() {
        return davLocatorFactory;
    }

    @Override
    public CaldavResourceFactory getResourceFactory() {
        return caldavResourceFactory;
    }

    @SneakyThrows
    public void execute(HttpServletRequest req, HttpServletResponse response) {
        val request = HttpServletRequestX.wrap(req);

        val authO = request.getHeaderO("Authorization");
        if (!authO.isPresent()) {
            response.addHeader("WWW-Authenticate", "Basic realm=\"CalDAV\"");
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "please authenticate");
            logger.warn("please authenticate for req:" + req.getRequestURI());
            return;
        }
        CcUserInfo userInfo;
        try {
            val auth = authO.get();
            val userIp = request.getXRealIpOrRemoteAddr();
            val userAgent = request.getUserAgent().getOrElse("?");

            if (auth.startsWith(HttpBasicAuthentication.BASIC_PREFIX)) {
                userInfo = handleBasicAuth(auth, request.getServerName(), userIp, userAgent);
            } else if (auth.startsWith(OAuthUtils.AUTH_HEADER_START)) {
                userInfo = handleOAuth(auth, userIp, userAgent);
            } else {
                response.addHeader("WWW-Authenticate", "Basic realm=\"CalDAV\"");
                response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "please authenticate");
                logger.warn("something wrong with authentication with req:" + req.getRequestURI());
                return;
            }
        } catch (AuthenticationFailedException e) {
            logger.warn("please authenticate for req:" + req.getRequestURI(), e);
            if (EnvironmentType.PRODUCTION != EnvironmentType.getActive()) {
                logger.error(e, e);
            } else {
                logger.error(e);
            }
            response.addHeader("WWW-Authenticate", "Basic realm=\"CalDAV\"");
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "please authenticate (incorrect login or password)");
            return;
        }

        val email = userInfo.getEmail();

        ClientHolder.holder.set(userInfo);
        Ndc.Handle sdcHandle = Ndc.push("l=" + LoginShortener.shortenLoginForLog(Emails.getUnicoded(email)));
        try {
            super.service(request, response);
            RequestsLogger.log(request, userInfo.getUid());

        } finally {
            sdcHandle.popSafely();
            ClientHolder.holder.remove();
        }
    }

    private CcUserInfo handleBasicAuth(String auth, String serverName, IpAddress userIp, String userAgent) {
        val loginPassword = HttpBasicAuthentication.parseBasic(auth);
        val email = emailFromLogin(loginPassword._1, serverName);
        val password = loginPassword._2;
        return ccCalendarFacade.checkPassword(email.getEmail(), password, userIp, userAgent);
    }

    private CcUserInfo handleOAuth(String auth, IpAddress userIp, String userAgent) {
        val token = auth.substring(OAuthUtils.AUTH_HEADER_START.length());
        return ccCalendarFacade.checkOAuthToken(token, userIp, userAgent);
    }

    private Email emailFromLogin(String login, String serverName) {
        val defaultDomain = defaultDomain(serverName);
        if (!login.contains("@")) {
            if (login.endsWith(".yandex-team.ru")) {
                // Address Book does not understand "@" in login
                login = login.replaceFirst("\\.yandex-team\\.ru$", "@yandex-team.ru");
            } else if (login.endsWith(".yandex.ru")) {
                login = login.replaceFirst("\\.yandex\\.ru$", "@yandex.ru");
            } else if (defaultDomain.isPresent()) {
                login += "@" + defaultDomain.get();
            } else {
                throw new IllegalStateException("cannot guess domain for login: " + login + ", server name: " + serverName);
            }
        }
        return new Email(login);
    }

    private Optional<String> defaultDomain(String serverName) {
        Optional<String> defaultDomain;
        if (serverName.toLowerCase().endsWith(".yandex.ru")) {
            defaultDomain = Optional.of("yandex.ru");
        } else if (serverName.toLowerCase().endsWith(".yandex-team.ru")) {
            defaultDomain = Optional.of("yandex-team.ru");
        } else if (serverName.toLowerCase().contains("carddav")) {
            defaultDomain = Optional.of("yandex-team.ru");
        } else {
            defaultDomain = passportAuthDomainsHolder.getDefaultDomain().map(d -> d.getDomain().getDomain()).toOptional();
        }
        return defaultDomain;
    }
}
