package ru.yandex.calendar.frontend.caldav.impl;

import java.util.Optional;
import java.util.regex.Pattern;

import lombok.experimental.UtilityClass;
import lombok.val;
import org.joda.time.Instant;

import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncInfo;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.logic.beans.generated.Layer;

@UtilityClass
public class LayerSyncToken {
    private static final String OLD_TOKEN_PREFIX = "data:,"; // This prefix is left for backward compatibility only
    private static final String SYNC_TOKEN_PREFIX = "sync-token:";
    private static final Pattern PATTERN = Pattern.compile("^(" + OLD_TOKEN_PREFIX + "|" + SYNC_TOKEN_PREFIX + "\\d+ )(\\d+)(\\s(.+))?");
    private static final int CURRENT_VERSION = 1; // For backward compatibility in case we decide to change drastically the semantics of the token

    private String getTokenValue(long millis, Optional<String> id) {
        val idPart = id.map(x -> " " + x).orElse("");
        return String.format("%s%d %d%s", SYNC_TOKEN_PREFIX, CURRENT_VERSION, millis, idPart);
    }

    public DavSyncToken getSyncToken(DavSyncInfo info) {
        return new DavSyncToken(getTokenValue(info.getMillis(), info.getExternalId()));
    }

    public DavSyncToken layerCollLastUpdateToSyncToken(Layer layer) {
        return new DavSyncToken(getTokenValue(layer.getCollLastUpdateTs().getMillis(), Optional.empty()));
    }

    public DavSyncToken lastUpdateTsToSyncToken(Optional<Instant> instant) {
        val millis = instant.orElseGet(Instant::now).getMillis();
        return new DavSyncToken(getTokenValue(millis, Optional.empty()));
    }

    public DavSyncToken lastUpdateTsToSyncToken(Instant instant) {
        return lastUpdateTsToSyncToken(Optional.of(instant));
    }

    public DavSyncToken getSyncToken(Instant instant, String id) {
        return new DavSyncToken(getTokenValue(instant.getMillis(), Optional.of(id)));
    }

    public Instant getInstantFromSyncToken(DavSyncToken syncToken) {
        return parseSyncToken(syncToken).getInstant();
    }

    public DavSyncInfo parseSyncToken(DavSyncToken syncToken) {
        val matcher = PATTERN.matcher(syncToken.getValue());
        if (!matcher.matches()) {
            throw new IllegalArgumentException("The token provided does not match any of the prefixes");
        }

        final long millis;
        try {
            millis = Long.parseLong(matcher.group(2));
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException("An error has occurred while parsing the token", e);
        }

        val externalId = Optional.ofNullable(matcher.group(4));

        return new DavSyncInfo(millis, externalId);
    }
}
