package ru.yandex.calendar.frontend.caldav.proto;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.NotImplementedException;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavLocatorFactory;
import org.apache.jackrabbit.webdav.DavMethods;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceFactory;
import org.apache.jackrabbit.webdav.DavResourceLocator;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.DavSessionProvider;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavRequestImpl;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.WebdavResponseImpl;
import org.apache.jackrabbit.webdav.server.AbstractWebdavServlet;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavRequest;
import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;
import ru.yandex.calendar.frontend.caldav.proto.tree.CaldavResource;
import ru.yandex.calendar.frontend.caldav.proto.tree.WebdavResource;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatus2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusUtils;
import ru.yandex.calendar.monitoring.EwsMonitoring;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 */
@SuppressWarnings("serial")
@Slf4j
public abstract class AbstractCaldavServlet extends AbstractWebdavServlet {
    @Autowired
    private EwsMonitoring ewsMonitoring;
    @Autowired
    private MeterRegistry registry;

    @Override
    public String getAuthenticateHeaderValue() {
        return "Basic realm=\"CalDAV\"";
    }


    @Override
    public void setDavSessionProvider(DavSessionProvider provider) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void setLocatorFactory(DavLocatorFactory factory) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void setResourceFactory(DavResourceFactory factory) {
        throw new UnsupportedOperationException();
    }



    @Override
    protected boolean isPreconditionValid(WebdavRequest req, DavResource res) {
        return true;
    }



    @Override
    protected boolean execute(WebdavRequest request, WebdavResponse response, int method, DavResource resource)
            throws ServletException, IOException, DavException
    {
        // default toOptions implementation also adds header
        if (method != DavMethods.DAV_OPTIONS) {
            response.addHeader(DavConstants.HEADER_DAV, resource.getComplianceClass());
        }
        return super.execute(request, response, method, resource);
    }


    @Override
    protected void service(HttpServletRequest request, HttpServletResponse response) throws IOException {
        try {
            log.info("method: " + request.getMethod() + ", service: " + request.getRequestURI());
            if (request.getMethod().equals("MKCALENDAR")) {
                WebdavRequest webdavRequest = new WebdavRequestImpl(request, getLocatorFactory());
                WebdavResponse webdavResponse = new WebdavResponseImpl(response, true);

                DavResource resource = getResourceFactory().createResource(webdavRequest.getRequestLocator(), webdavRequest, webdavResponse);
                doMkCalendar(webdavRequest, webdavResponse, (CaldavResource) resource);
            } else {
                super.service(request, response);
            }
        } catch (Exception e) {
            log.error("method: " + request.getMethod() + ", service: " + request.getRequestURI(), e);
            ewsMonitoring.reportIfEwsException(e);
            response.sendError(JackrabbitUtils.translateToStatusCode(e));
        }
    }


    protected void doMkCalendar(WebdavRequest request, WebdavResponse response,
            CaldavResource resource) throws DavException, IOException
    {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource.getCollection());

        resource.mkCalendar(new CaldavRequest(request).getMkCalendarProperties());
        response.setStatus(DavServletResponse.SC_CREATED);
    }

    @Override
    protected void doGet(WebdavRequest request, WebdavResponse response, DavResource resource) throws IOException, DavException {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);

        if (((WebdavResource) resource).hasCustomGet()) {
            ((WebdavResource) resource).get(request, response);
        } else {
            // XXX: add custom exception translation
            super.doGet(request, response, resource);
        }
    }

    @Override
    protected void doPost(WebdavRequest request, WebdavResponse response,
            DavResource resource) throws DavException, IOException
    {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);

        // XXX: add custom exception translation
        ((WebdavResource) resource).post(request, response);
    }

    @Override
    protected void doPut(WebdavRequest request, WebdavResponse response, DavResource resource)
            throws IOException, DavException
    {
        DavResource parentResource = resource.getCollection();
        if (parentResource == null || !parentResource.exists()) {
            // parent does not exist
            response.sendError(DavServletResponse.SC_CONFLICT);
            return;
        }
        checkResourceExistsAndAccessibleToClient((WebdavResource) parentResource);

        PutResponse result = ((WebdavResource) parentResource).put(
                (WebdavResource) resource, getInputContext(request, request.getInputStream()));

        response.setStatus(result.getStatusCode());
        if (result.getEtag().isPresent()) {
            response.setHeader(DavConstants.HEADER_ETAG, result.getEtag().get().getValue());
        }
    }

    @Override
    protected void doOptions(WebdavRequest request, WebdavResponse response, DavResource resource) throws IOException, DavException {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);

        super.doOptions(request, response, resource);
    }

    @Override
    protected void doReport(WebdavRequest request, WebdavResponse response, DavResource resource) throws DavException, IOException {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);

        response.addHeader("ETag", "\"None\""); // mimicking calendar server
        // XXX: Depth header
        // XXX: add custom exception translation
        ((WebdavResource) resource).report(request, response);
    }

    @Override
    protected void doPropFind(WebdavRequest request, WebdavResponse response, DavResource resource) throws IOException, DavException {
        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);

        WebdavResource webdavResource = (WebdavResource) resource;

        int depth = request.getDepth(DEPTH_INFINITY);

        PropertiesRequest propertiesRequest;
        switch (request.getPropFindType()) {
        case PROPFIND_ALL_PROP:
            propertiesRequest = PropertiesRequest.all();
            break;
        case PROPFIND_ALL_PROP_INCLUDE:
            throw new NotImplementedException("all-include is not supported");
        case PROPFIND_BY_PROPERTY:
            propertiesRequest = PropertiesRequest.set(request.getPropFindProperties());
            break;
        case PROPFIND_PROPERTY_NAMES:
            propertiesRequest = PropertiesRequest.names();
            break;
        default:
            throw new IllegalStateException("unknown propfind type: " + request.getPropFindType());
        }

        log.info("PROPFIND " + request.getRequestURI() + " depth: " + depth + ", properties: " + propertiesRequest);

        MultiStatus2 multiStatus = webdavResource.propFind(propertiesRequest, depth);

        // XXX: dump response summary

        MultiStatusUtils.sendMultiStatus(multiStatus, request, response, registry);
    }

    @Override
    protected void doMove(WebdavRequest request, WebdavResponse response, DavResource resource) throws DavException {
        DavResource destResource;
        try {
            URI uri = new URI(request.getHeader(HEADER_DESTINATION));
            String path = uri.getRawPath();
            if (uri.getAuthority() == null && (path.startsWith("//") || !path.startsWith("/"))) {
                throw new DavException(DavServletResponse.SC_BAD_REQUEST);
            }
            DavResourceLocator locator = getLocatorFactory().createResourceLocator(
                    request.getScheme() + "://" + request.getHeader("Host"), path);
            destResource = this.getResourceFactory().createResource(locator, request, response);
        } catch (URISyntaxException e) {
            throw new DavException(DavServletResponse.SC_BAD_REQUEST);
        }

        checkResourceExistsAndAccessibleToClient((WebdavResource) resource);
        checkResourceExistsAndAccessibleToClient((WebdavResource) destResource.getCollection());

        /**
         * XXX: wrong in case of overwrite
         * @see org.apache.jackrabbit.webdav.DavServletRequest#isOverwrite()
         */
        Validate.V.isFalse(destResource.exists());

        Validate.V.notEquals(resource.getHref(), destResource.getHref());

        // XXX: add custom exception translation
        resource.move(destResource);
        response.setStatus(DavServletResponse.SC_NO_CONTENT);
    }

    private void checkResourceExistsAndAccessibleToClient(WebdavResource resource) throws DavException {
        if (!resource.exists() || !resource.isAccessibleTo(ClientHolder.getSafe())) {
            throw new DavException(DavServletResponse.SC_NOT_FOUND,
                    "resource does not exist or not accessible to user " + resource.getHref());
        }
    }

} //~
