package ru.yandex.calendar.frontend.caldav.proto.caldav;

import lombok.extern.slf4j.Slf4j;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.xml.DomUtil;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.regex.Pattern2;
import ru.yandex.misc.web.servlet.HttpServletRequestX;
import ru.yandex.misc.xml.dom.DomUtils;

/**
 * @author Stepan Koltsov
 * @author shinderuk
 */
@Slf4j
public class CaldavRequest {
    private final WebdavRequest webdavRequest;

    public CaldavRequest(WebdavRequest webdavRequest) {
        this.webdavRequest = webdavRequest;
    }

    public WebdavRequest getWebdavRequest() {
        return webdavRequest;
    }

    public HttpServletRequestX getHttpServletRequest() {
        return HttpServletRequestX.wrap(webdavRequest);
    }

    public ListF<String> getHeaders(String name) {
        return getHttpServletRequest().getHeadersAsList(name);
    }

    public Option<String> getHeader(String name) {
        return getHttpServletRequest().getHeaderO(name);
    }

    /**
     * <pre>
     * The Originator request header value is a URI which specifies the
     * calendar user address of the originator of the scheduling message.
     * Note that the absoluteURI rule is defined in [RFC3986].
     * </pre>
     *
     * @url http://tools.ietf.org/html/draft-desruisseaux-ischedule-01#section-8.4
     * @url http://wiki.yandex-team.ru/calendar/caldav/1/google/outbox-post-attendee
     */
    public Option<Email> getOriginator() {
        return getHeader("Originator").map(Email.parseMailtoF());
    }

    /**
     * <pre>
     * Recipient      = "Recipient" ":" OWS Recipient-v
     * Recipient-v    = Recipient-elem *( OWS "," OWS Recipient-elem )
     * Recipient-elem = absoluteURI
     * </pre>
     *
     * @url http://tools.ietf.org/html/draft-desruisseaux-ischedule-01#section-8.5
     * @url http://wiki.yandex-team.ru/calendar/caldav/1/google/outbox-post-attendee
     */
    public ListF<Email> getRecipients() {
        ListF<String> headers = getHeaders("Recipient");
        try {
            return headers.flatMap(Pattern2.compile(", *")::split).map(Email.parseMailtoF());
        } catch (Exception e) {
            throw new CaldavException("failed to parse Recipient headers: " + headers, e);
        }
    }

    public ListF<DavProperty<?>> getMkCalendarProperties() {
        byte[] requestBody = new HttpServletRequestX(webdavRequest).getInputStreamX().readBytes();
        if (requestBody.length == 0) {
            return Cf.list();
        }
        log.debug(new String(requestBody));
        Element rootEl = DomUtils.I.readRootElement(requestBody);
        Validate.isTrue(DomUtil.matches(rootEl, "mkcalendar", CaldavConstants.CALDAV_NS));

        Option<Element> davSetEl = DomUtils.childElements(rootEl).singleO();
        if (!davSetEl.isPresent()) {
            return Cf.list();
        }
        Validate.isTrue(DomUtil.matches(davSetEl.get(), DavConstants.XML_SET, WebdavConstants.DAV_NS));

        Element davPropEl = DomUtils.childElements(davSetEl.get()).single();
        Validate.isTrue(DomUtil.matches(davPropEl, DavConstants.XML_PROP, WebdavConstants.DAV_NS));

        ListF<DavProperty<?>> propList = Cf.arrayList();
        for (Element propertyEl : DomUtils.childElements(davPropEl)) {
            DavPropertyName prop = CaldavConstants.CALDAV_SUPPORTED_CALENDAR_COMPONENT_SET_PROP;
            if (DomUtil.matches(propertyEl, prop.getName(), prop.getNamespace())) {
                propList.add(CaldavSupportedCalendarComponentSet.createFromXml(propertyEl));
            } else {
                propList.add(DefaultDavProperty.createFromXml(propertyEl));
            }
        }
        return propList;
    }

} //~
