package ru.yandex.calendar.frontend.caldav.proto.caldav.report;

import java.util.Optional;

import lombok.val;
import one.util.streamex.StreamEx;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncLevel;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequestParser;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.WebdavXmlizers;
import ru.yandex.misc.xml.dom.DomUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @url http://tools.ietf.org/html/rfc6578#section-6.1
 *
 * <!ELEMENT sync-collection (sync-token, sync-level, limit?, prop)>
 */
public class ReportRequestSyncCollectionParser {

    public static ReportRequestSyncCollection parseSyncCollection(Element element) {
        DomUtils.validateName(element, Dom4jUtils.toJavaxXmlNamespaceQName(WebdavConstants.DAV_SYNC_COLLECTION_QNAME));

        PropertiesRequest propertiesRequest = PropertiesRequest.all();
        DavSyncToken syncToken = null;
        // http://tools.ietf.org/html/rfc6578#appendix-A
        DavSyncLevel syncLevel = DavSyncLevel.ONE;
        Optional<Integer> limit = Optional.empty();

        for (Element child : DomUtils.childElements(element)) {
            Option<PropertiesRequest> prop = PropertiesRequestParser.parseProp(child);
            if (prop.isPresent()) {
                propertiesRequest = prop.get();
            } else if (DomUtils.nameIs(child, Dom4jUtils.toJavaxXmlNamespaceQName(
                    WebdavConstants.DAV_SYNC_TOKEN_QNAME))) {
                syncToken = WebdavXmlizers.syncTokenXmlizer.parseXml(child);
            } else if (DomUtils.nameIs(child, Dom4jUtils.toJavaxXmlNamespaceQName(
                    WebdavConstants.DAV_SYNC_LEVEL_QNAME))) {
                syncLevel = DavSyncLevel.fromValue(child.getTextContent());
            } else if (DomUtils.nameIs(child, Dom4jUtils.toJavaxXmlNamespaceQName(
                        WebdavConstants.DAV_LIMIT_QNAME))) {
                    // ignore
                    // http://tools.ietf.org/html/rfc5323#section-5.17
                val children = DomUtils.childElements(child, Dom4jUtils.toJavaxXmlNamespaceQName(WebdavConstants.DAV_NRESULTS_QNAME));
                limit = StreamEx.of(children)
                    .map(Node::getTextContent)
                    .map(Integer::valueOf)
                    .findFirst();
            } else {
                throw new IllegalArgumentException("unknown element: " + DomUtils.qname(child));
            }
        }

        if (syncToken == null) {
            throw new IllegalArgumentException("sync token not defined in sync-collection report request");
        }
        return new ReportRequestSyncCollection(propertiesRequest, syncToken, syncLevel, limit);
    }

}
