package ru.yandex.calendar.frontend.caldav.proto.facade;

import java.util.List;
import java.util.Optional;

import lombok.val;
import net.fortuna.ical4j.model.Component;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.NotImplementedException;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.caldav.impl.LayerCtag;
import ru.yandex.calendar.frontend.caldav.impl.LayerSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentConditions;
import ru.yandex.calendar.frontend.caldav.proto.tree.CollectionId;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.exp.IcsTodoExporter;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * For Lightning client to use /events-default calendar link for easily access default layer and all todos
 */
public class CaldavCalendarEventsPlusTodosProvider implements CaldavCalendarFacadeProvider {
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private UserManager userManager;
    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private IcsTodoExporter icsTodoExporter;
    @Autowired
    private CaldavCalendarFacadeProvider caldavCalendarEventsProvider;

    private void ensureUserIsCollectionOwner(UserInfo userInfo, CollectionId collectionId) {
        Validate.isTrue(collectionId.isEvents());
        Validate.isTrue(collectionId.getId().equals("default"));

        if (!userInfo.getUid().sameAs(parseUser(collectionId.getUser()))) {
            throw new PermissionDeniedUserException("User is not collection owner");
        }
    }

    private long getLayerId(CollectionId collectionId) {
        Validate.isTrue(collectionId.isEvents());
        Validate.isTrue(collectionId.getId().equals("default"));

        return layerRoutines.getOrCreateDefaultLayer(parseUser(collectionId.getUser()));
    }

    private CollectionId toEventsCollectionId(CollectionId collectionId) {
        return CollectionId.events(collectionId.getUser(), Long.toString(getLayerId(collectionId)),
                collectionId.getPassportUid());
    }

    private PassportUid parseUser(String user) {
        return userManager.getUidByEmail(Emails.punycode(user)).get();
    }

    private Instant calculateLastUpdateTs(long layerId) {
        Layer layer = layerRoutines.getLayerById(layerId);
        Instant layerLastUpdate = layer.getCollLastUpdateTs();
        Instant todoMaxLastUpdate = todoRoutines.getTodoLastUpdateTs(layer.getCreatorUid());

        return Cf.list(todoMaxLastUpdate, layerLastUpdate).max();
    }

    private ActionInfo getActionInfo() {
        return new ActionInfo(ActionSource.CALDAV, RequestIdStack.current().get(), Instant.now());
    }


    @Override
    public boolean existsCalendarWithId(CollectionId collectionId) {
        return collectionId.isEvents() && collectionId.getId().equals("default")
                && userManager.getUidsByEmails(Email.parseSafe(collectionId.getUser())).single().get2().isPresent();
    }

    @Override
    public void makeCalendarIfUserHasNoOne(UserInfo userInfo) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public ListF<CalendarDescription> getOwnCalendars(UserInfo userInfo) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public ListF<CalendarDescription> getCalendarsSharedByAnotherUser(UserInfo userInfo, PassportUid owner) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public ListF<CalendarDescription> getVisibleExternalCalendars(UserInfo userInfo) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public CalendarDescription getCalendarById(UserInfo userInfo, CollectionId collectionId) {
        ensureUserIsCollectionOwner(userInfo, collectionId);
        val layerId = getLayerId(collectionId);

        val layerUser = layerRoutines.getLayerUsersWithRelationsByUid(userInfo.getUid(), Option.empty())
                .find(lu -> lu.layerIdIs(layerId)).get();

        val lastUpdateTs = calculateLastUpdateTs(layerId);
        val evaluatedName = layerUser.getEvaluatedLayerName();
        val evaluatedColor = layerUser.getEvaluatedColor();

        final boolean writable;
        if (layerUser.getLayer().getType() == LayerType.USER) {
            val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layerUser.getLayer());
            writable = authorizer.canPerformLayerAction(userInfo, layerPermInfo, Optional.empty(), LayerAction.CREATE_EVENT,
                ActionSource.CALDAV);
        } else {
            writable = false;
        }

        return new CalendarDescription(
                Option.of(collectionId.getUser()), collectionId, evaluatedName, new IcalColor(evaluatedColor),
                writable, Cf.list(Component.VEVENT, Component.VTODO),
                LayerCtag.lastUpdateTsToCTag(lastUpdateTs),
                Option.of(LayerSyncToken.lastUpdateTsToSyncToken(lastUpdateTs)));
    }

    @Override
    public DavSyncToken getCalendarSyncToken(CollectionId collectionId) {
        return LayerSyncToken.lastUpdateTsToSyncToken(calculateLastUpdateTs(getLayerId(collectionId)));
    }

    @Override
    public void makeCalendar(UserInfo userInfo, String id, CalendarProperties properties) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public void changeCalendar(UserInfo userInfo, CollectionId collectionId, CalendarProperties properties) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public void removeCalendar(UserInfo userInfo, CollectionId collectionId) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public void putCalendarEntry(
            UserInfo userInfo, IcsCalendar icsCalendar, CollectionId collectionId, Option<Instant> notModifiedSince)
    {
        ensureUserIsCollectionOwner(userInfo, collectionId);
        val layerId = getLayerId(collectionId);

        icsImporter.importIcsStuff(userInfo.getUid(), icsCalendar, IcsImportMode.caldavPut(layerId, notModifiedSince));
    }

    @Override
    public void moveCalendarEntry(UserInfo userInfo, String fileName, CollectionId from, CollectionId to) {
        throw new NotImplementedException("not needed");
    }

    @Override
    public void removeCalendarEntry(UserInfo userInfo, String fileName, CollectionId collectionId) {
        ensureUserIsCollectionOwner(userInfo, collectionId);

        val externalId = IcsNameUtils.fileNameToExternalId(fileName);

        caldavCalendarEventsProvider.removeCalendarEntry(userInfo, externalId, toEventsCollectionId(collectionId));
        try {
            todoRoutines.deleteTodoItemById(userInfo.getUid(), TodoIdOrExternalId.externalId(externalId), getActionInfo());

        } catch (CommandRunException e) {
            if (!e.getSituation().isSome(Situation.TODO_LIST_NOT_FOUND)) throw e; // CAL-6766
        }
    }

    @Override
    public ListF<CalendarComponent> getCalendarEntries(UserInfo userInfo, CollectionId collectionId,
            CalendarComponentConditions eventConditions, CalendarComponentConditions todoConditions,
            ExportOptions options)
    {
        ensureUserIsCollectionOwner(userInfo, collectionId);

        val events = caldavCalendarEventsProvider.getCalendarEntries(
                userInfo, toEventsCollectionId(collectionId), eventConditions, todoConditions, options);
        val todos = icsTodoExporter.exportTodoItemsByUidForCaldav(userInfo.getUid())
                .map(CalendarComponent.fromTodoExportF(options));

        return events.plus(todos);
    }

    @Override
    public ListF<ComponentGetResult> getCalendarEntries(
            UserInfo userInfo, CollectionId collectionId, ListF<String> fileNames, ExportOptions options)
    {
        ensureUserIsCollectionOwner(userInfo, collectionId);

        val foundEvents = caldavCalendarEventsProvider
                .getCalendarEntries(userInfo, toEventsCollectionId(collectionId), fileNames, options)
                .filter(ComponentGetResult::isFound);

        val exports = icsTodoExporter.exportTodoItemsByExternalIdForCaldav(
                userInfo.getUid(), fileNames.map(IcsNameUtils::fileNameToExternalId));

        val foundTodos = exports
                .map(CalendarComponent.fromTodoExportF(options).andThen(ComponentModified::found));

        val foundByFileName = foundEvents.plus(foundTodos)
                .toMapMappingToKey(ComponentGetResult::getFileName);

        return fileNames.map(n -> foundByFileName.getO(n).getOrElse(() -> ComponentGetResult.notFound(n)));
    }

    @Override
    public ListF<ComponentModified> getCalendarEntriesCreatedOrModifiedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since, ExportOptions options)
    {
        ensureUserIsCollectionOwner(userInfo, collectionId);

        val events = caldavCalendarEventsProvider.getCalendarEntriesCreatedOrModifiedSince(
                userInfo, toEventsCollectionId(collectionId), since, options);
        val todos = icsTodoExporter.exportTodoItemsByUidForCaldavCreatedOrModifiedSince(
                userInfo.getUid(), since).map(CalendarComponent.fromTodoExportF(options));

        return events.plus(todos.map(ComponentModified::found));
    }

    @Override
    public List<ComponentModified> getCalendarEntriesDeletedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since)
    {
        ensureUserIsCollectionOwner(userInfo, collectionId);

        val todoNameHandler = TodoItem.getExternalIdF().andThen(IcsNameUtils::externalIdToFileName);
        val eventIds = caldavCalendarEventsProvider.getCalendarEntriesDeletedSince(
                userInfo, toEventsCollectionId(collectionId), since);
        final var todoIds = todoDao.findDeletedTodoItemsByUidAndMinDeletedTs(userInfo.getUid(), since)
                .map(todo -> ComponentModified.deleted(todoNameHandler.apply(todo), todo.getDeletionTs().get()));

        return StreamEx.of(eventIds)
                .append(todoIds)
                .sortedBy(ComponentGetResult::getFileName)
                .distinct(ComponentGetResult::getFileName)
                .toImmutableList();
    }
}
