package ru.yandex.calendar.frontend.caldav.proto.facade;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import lombok.val;
import net.fortuna.ical4j.model.Component;
import one.util.streamex.StreamEx;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.impl.LayerCtag;
import ru.yandex.calendar.frontend.caldav.impl.LayerSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentConditions;
import ru.yandex.calendar.frontend.caldav.proto.tree.CollectionId;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.beans.generated.DeletedEvent;
import ru.yandex.calendar.logic.beans.generated.DeletedEventLayer;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.ExternalId;
import ru.yandex.calendar.logic.event.archive.DeletedEventDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.ics.exp.IcsEventExporter;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserCssClassHack;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.reqid.RequestIdStack;

public class CaldavCalendarEventsProvider implements CaldavCalendarFacadeProvider {
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private IcsEventExporter icsEventExporter;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private DeletedEventDao deletedEventDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;

    private long getLayerId(CollectionId collectionId) {
        Validate.isTrue(collectionId.isEvents());
        return Long.parseLong(collectionId.getId());
    }

    private void ensureCanPerformLayerAction(UserInfo userInfo, long layerId, LayerAction action) {
        val source = ActionSource.CALDAV;
        val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layerRoutines.getLayerById(layerId));

        if (action != LayerAction.LIST) {
            authorizer.ensureLayerType(layerPermInfo, LayerType.USER);
        }
        authorizer.ensureCanPerformLayerAction(userInfo, layerPermInfo, Optional.empty(), action, source);
    }

    private boolean canPerformLayerAction(UserInfo userInfo, long layerId, LayerAction action) {
        try {
            ensureCanPerformLayerAction(userInfo, layerId, action);
            return true;
        } catch (PermissionDeniedUserException e) {
            return false;
        }
    }

    private PassportUid parseUser(String user) {
        return userManager.getUidByEmail(Emails.punycode(user)).get();
    }

    private ActionInfo getActionInfo() {
        return new ActionInfo(ActionSource.CALDAV, RequestIdStack.current().get(), Instant.now());
    }


    @Override
    public boolean existsCalendarWithId(CollectionId collectionId) {
        if (!collectionId.isEvents()) return false;

        Option<Long> id = Cf.Long.parseSafe(collectionId.getId());
        Option<PassportUid> uid = userManager.getUidsByEmails(Email.parseSafe(collectionId.getUser())).single().get2();

        return id.isPresent() && uid.isPresent() && layerRoutines.getLayerUser(id.get(), uid.get()).isPresent();
    }

    @Override
    public void makeCalendarIfUserHasNoOne(UserInfo userInfo) {
        layerRoutines.getOrCreateDefaultLayer(userInfo.getUid());
    }

    @Override
    public ListF<CalendarDescription> getOwnCalendars(UserInfo userInfo) {
        return layerRoutines.getLayerUsersWithRelationsByUid(userInfo.getUid(), Option.empty())
                .filter(lu -> lu.layerCreatorIs(userInfo.getUid()))
                .map(toCalendarDescriptionF(userInfo));
    }

    @Override
    public ListF<CalendarDescription> getCalendarsSharedByAnotherUser(UserInfo userInfo, PassportUid owner) {
        return layerRoutines.getLayerUsersWithRelationsByUid(userInfo.getUid(), Option.empty())
                .filter(lu -> lu.layerCreatorIs(owner))
                .map(toCalendarDescriptionF(userInfo));
    }

    @Override
    public ListF<CalendarDescription> getVisibleExternalCalendars(UserInfo userInfo) {
        return layerRoutines.getLayerUsersWithRelationsByUid(userInfo.getUid(), Option.empty())
                .filterNot(lu -> lu.layerCreatorIs(userInfo.getUid()))
                .map(toCalendarDescriptionF(userInfo));
    }

    @Override
    public void makeCalendar(UserInfo userInfo, String id, CalendarProperties properties) {
        layerRoutines.createUserLayer(userInfo.getUid(), extractLayerData(properties), extractLayerUserData(properties));
    }

    @Override
    public CalendarDescription getCalendarById(UserInfo userInfo, CollectionId collectionId) {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.LIST);

        Option<CalendarDescription> calendar = layerRoutines
                .getLayerUsersWithRelationsByLayerIds(Cf.list(layerId))
                .find(lu -> lu.getLayerUserUid().sameAs(userInfo.getUid()))
                .map(toCalendarDescriptionF(userInfo));

        if (!calendar.isPresent()) throw new PermissionDeniedUserException();
        return calendar.get();
    }

    @Override
    public void changeCalendar(UserInfo userInfo, CollectionId collectionId, CalendarProperties properties) {
        long layerId = getLayerId(collectionId);
        ActionInfo actionInfo = getActionInfo();

        if (canPerformLayerAction(userInfo, layerId, LayerAction.EDIT)) {
            Layer data = extractLayerData(properties);
            if (data.cardinality() > 0) {
                data.setId(layerId);
                layerRoutines.updateLayer(data, userInfo.getUid(), actionInfo);
            }
        }
        Option<LayerUser> layerUser = layerRoutines.getLayerUser(layerId, userInfo.getUid());
        if (layerUser.isPresent()) {
            LayerUser data = extractLayerUserData(properties);
            if (data.cardinality() > 0) {
                data.setId(layerUser.get().getId());
                layerRoutines.updateLayerUser(layerId, userInfo.getUid(), data, actionInfo);
            }
        }
    }

    @Override
    public DavSyncToken getCalendarSyncToken(CollectionId collectionId) {
        return LayerSyncToken.layerCollLastUpdateToSyncToken(layerRoutines.getLayerById(getLayerId(collectionId)));
    }

    @Override
    public void removeCalendar(UserInfo userInfo, CollectionId collectionId) {
        if (userInfo.getUid().isYandexTeamRu()) {
            throw new PermissionDeniedUserException("Not allowed to delete calendars in yandex-team");
        }
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.DELETE);

        layerRoutines.deleteLayer(userInfo, layerId, getActionInfo());
    }

    @Override
    public void putCalendarEntry(
            UserInfo userInfo, IcsCalendar icsCalendar, CollectionId collectionId, Option<Instant> notModifiedSince)
    {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.CREATE_EVENT);

        PassportUid performerUid = canPerformLayerAction(userInfo, layerId, LayerAction.EDIT) ?
                parseUser(collectionId.getUser()) : userInfo.getUid();

        icsImporter.importIcsStuff(performerUid, icsCalendar, IcsImportMode.caldavPut(layerId, notModifiedSince));
    }

    @Override
    public void moveCalendarEntry(UserInfo userInfo, String fileName, CollectionId from, CollectionId to) {
        long fromId = getLayerId(from);
        long toId = getLayerId(to);

        ensureCanPerformLayerAction(userInfo, fromId, LayerAction.DETACH_EVENT);
        ensureCanPerformLayerAction(userInfo, toId, LayerAction.CREATE_EVENT);

        ActionInfo actionInfo = getActionInfo();
        eventRoutines.updateEventLayerByEventExternalId(
                userInfo, IcsNameUtils.fileNameToExternalId(fileName), fromId, toId, actionInfo);
        xivaNotificationManager.notifyLayersUsersAboutEventsChange(Cf.list(fromId, toId), actionInfo);
    }

    @Override
    public void removeCalendarEntry(UserInfo userInfo, String fileName, CollectionId collectionId) {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.DETACH_EVENT);

        ActionInfo actionInfo = getActionInfo();
        eventRoutines.deleteOrDetachEventsByExternalId(
                userInfo, layerId, IcsNameUtils.fileNameToExternalId(fileName), actionInfo);
        xivaNotificationManager.notifyLayersUsersAboutEventsChange(Cf.list(layerId), actionInfo);
    }

    // XXX: condition unused
    @Override
    public ListF<CalendarComponent> getCalendarEntries(UserInfo userInfo, CollectionId collectionId,
            CalendarComponentConditions eventConditions, CalendarComponentConditions todoConditions,
            ExportOptions options)
    {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.LIST);

        if (eventConditions.isFalse()) return Cf.list();

        val eventGroups = icsEventExporter.exportEventsOnLayerForCaldav(userInfo.getUid(), layerId, options.toIcsOptions(),
                        eventConditions.getTimeRangeCondition().getTimeRange(), Instant.now());

        return Cf.toList(eventGroups).map(CalendarComponent.fromEventGroupExportF(options));
    }

    @Override
    public ListF<ComponentGetResult> getCalendarEntries(
            UserInfo userInfo, CollectionId collectionId, ListF<String> fileNames, ExportOptions options)
    {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.LIST);

        val externalIds = fileNames.map(IcsNameUtils::fileNameToExternalId).map(ExternalId::new);

        val exports = icsEventExporter.exportEventsByExternalIdsForCaldav(
                userInfo.getUid(), layerId, externalIds, options.toIcsOptions(), Instant.now());

        ListF<ComponentGetResult> found = Cf.toList(exports).map(
                g -> ComponentModified.found(CalendarComponent.fromEventGroupExport(g, options)));

        val foundByFileName = found.toMapMappingToKey(ComponentGetResult::getFileName);

        return fileNames.map(n -> foundByFileName.getO(n).getOrElse(() -> ComponentGetResult.notFound(n)));
    }

    @Override
    public ListF<ComponentModified> getCalendarEntriesCreatedOrModifiedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since, ExportOptions options)
    {
        val layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.LIST);

        val eventGroups =
                icsEventExporter.exportEventsOnLayerForCaldavCreatedOrModifiedSince(
                        userInfo.getUid(), layerId, since, options.toIcsOptions(), Instant.now());

        return eventGroups.map(e -> e.getExport().isPresent()
                ? ComponentModified.found(CalendarComponent.fromEventGroupExport(e.getExport().get(), options))
                : ComponentModified.deleted(IcsNameUtils.externalIdToFileName(e.getExternalId().getRaw()), e.getDate()));
    }

    private static <T extends Comparable> Comparator<Map.Entry<T, Instant>> compareByKeyAndValueDescending() {
        return Comparator.comparing(Map.Entry<T, Instant>::getKey)
                .thenComparing(Comparator.comparing(Map.Entry<T, Instant>::getValue))
                .reversed();
    }

    @Override
    public List<ComponentModified> getCalendarEntriesDeletedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since)
    {
        long layerId = getLayerId(collectionId);
        ensureCanPerformLayerAction(userInfo, layerId, LayerAction.LIST);

        val deletedEventLayerEvents = deletedEventDao
                .findDeletedEventLayerByLayerIdDeletedSince(layerId, since);

        val deletedEventLayerEventIds = deletedEventLayerEvents
                .map(DeletedEventLayer::getEventId);
        val mapEventsToDeletionDates = StreamEx.of(deletedEventLayerEvents)
                .mapToEntry(DeletedEventLayer::getEventId, DeletedEventLayer::getDeletionTs)
                .sorted(compareByKeyAndValueDescending())
                .distinctKeys()
                .toImmutableMap();

        val deletedEvents = StreamEx.of(deletedEventDao.findDeletedEventsByIds(deletedEventLayerEventIds))
                .mapToEntry(DeletedEvent::getExternalId, DeletedEvent::getDeletionTs);
        val movedOrRejectedEvents = mainEventDao.mapEventIdsToExternalIds(deletedEventLayerEventIds);
        val movedOrRejectedComponents = StreamEx.of(deletedEventLayerEventIds)
                .filter(movedOrRejectedEvents::containsKey)
                .mapToEntry(movedOrRejectedEvents::get, mapEventsToDeletionDates::get)
                .sorted(compareByKeyAndValueDescending())
                .distinctKeys()
                .toImmutableMap();

        // We don't need sort here theoretically, but just to be safe - because our data may be corrupted
        return deletedEvents
                .append(movedOrRejectedComponents)
                .mapKeys(IcsNameUtils::externalIdToFileName)
                .sorted(compareByKeyAndValueDescending())
                .distinctKeys()
                .mapKeyValue(ComponentModified::deleted)
                .toImmutableList();
    }

    private Layer extractLayerData(CalendarProperties properties) {
        Layer data = new Layer();
        if (properties.getDisplayName().isPresent()) {
            data.setName(properties.getDisplayName().get());
        }
        return data;
    }

    private LayerUser extractLayerUserData(CalendarProperties properties) {
        LayerUser data = new LayerUser();
        if (properties.getColor().isPresent()) {
            Color color = properties.getColor().get().getColor();
            data.setHeadBgColor(color.getRgb());
            data.setCssClass(LayerUserCssClassHack.getBestMatchingCssClassByColor(color));
        }
        if (properties.getAffectsAvailability().isPresent()) {
            data.setAffectsAvailability(properties.getAffectsAvailability().get());
        }
        return data;
    }

    private CalendarDescription toCalendarDescription(UserInfo userInfo, LayerUserWithRelations layerUser) {
        Validate.equals(userInfo.getUid(), layerUser.getLayerUser().getUid());

        Layer layer = layerUser.getLayer();

        String evaluatedName = layerUser.getEvaluatedLayerName();
        Color evaluatedColor = layerUser.getEvaluatedColor();

        val owner = userManager.getEmailByUid(layer.getCreatorUid()).map(Email::getEmail);
        String user = userManager.getEmailByUid(userInfo.getUid()).get().getEmail();

        CollectionId collectionId = CollectionId.events(user, Long.toString(layer.getId()), userInfo.getUid());

        boolean writable = canPerformLayerAction(userInfo, layer.getId(), LayerAction.CREATE_EVENT);

        // Don't archive feed layer events, so can't use sync (see IcsFeedManager.removeEventsDeletedFromFeed)
        Option<DavSyncToken> syncTokenO = layer.getType() != LayerType.FEED
                ? Option.of(LayerSyncToken.layerCollLastUpdateToSyncToken(layer))
                : Option.empty();

        return new CalendarDescription(
                Option.x(owner), collectionId, evaluatedName, new IcalColor(evaluatedColor),
                writable, Cf.list(Component.VEVENT),
                LayerCtag.layerCollLastUpdateToCTag(layer), syncTokenO);
    }

    private Function<LayerUserWithRelations, CalendarDescription> toCalendarDescriptionF(final UserInfo userInfo) {
        return layerUser -> toCalendarDescription(userInfo, layerUser);
    }
}
