package ru.yandex.calendar.frontend.caldav.proto.facade;

import java.io.ByteArrayInputStream;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import net.fortuna.ical4j.model.Calendar;
import net.fortuna.ical4j.model.Component;
import net.fortuna.ical4j.model.PeriodList;
import net.fortuna.ical4j.model.component.VFreeBusy;
import net.fortuna.ical4j.model.property.DtEnd;
import net.fortuna.ical4j.model.property.DtStart;
import net.fortuna.ical4j.model.property.FreeBusy;
import net.fortuna.ical4j.model.property.Method;
import net.fortuna.ical4j.model.property.Uid;
import net.fortuna.ical4j.model.property.Version;
import one.util.streamex.StreamEx;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarDataSourceStatus;
import ru.yandex.calendar.frontend.caldav.impl.EventEtag;
import ru.yandex.calendar.frontend.caldav.impl.LayerSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.caldav.itip.StatusCode;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentConditions;
import ru.yandex.calendar.frontend.caldav.proto.caldav.schedule.RequestStatus;
import ru.yandex.calendar.frontend.caldav.proto.caldav.schedule.ScheduleResponse;
import ru.yandex.calendar.frontend.caldav.proto.caldav.schedule.ScheduleResponseItem;
import ru.yandex.calendar.frontend.caldav.proto.caldav.schedule.outbox.OutboxRequestVEvent;
import ru.yandex.calendar.frontend.caldav.proto.caldav.schedule.outbox.OutboxRequestVFreeBusy;
import ru.yandex.calendar.frontend.caldav.proto.tree.CollectionId;
import ru.yandex.calendar.frontend.caldav.proto.tree.ResourceNotFoundException;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncInfo;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.caldav.userAgent.UserAgentType;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervalsOrRefusal;
import ru.yandex.calendar.logic.event.avail.AvailabilityQueryRefusalReason;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.random.Random2;

/**
 * XXX: do permission checks
 * @see CarddavCalendarFacadeImpl
 */
@Slf4j
public class CaldavCalendarFacadeImpl implements CaldavCalendarFacade {
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private CaldavCalendarFacadeProvider caldavCalendarEventsProvider;
    @Autowired
    private CaldavCalendarFacadeProvider caldavCalendarTodosProvider;
    @Autowired
    private CaldavCalendarFacadeProvider caldavCalendarEventsPlusTodosProvider;
    @Autowired
    private CalendarDataSourceStatus calendarDataSourceStatus;

    private final DynamicProperty<ListF<String>> etcGmtForTimezones =
            new DynamicProperty<>("caldavEtcGmtForTimezones", Cf.list());

    private List<CaldavCalendarFacadeProvider> getAllProviders() {
        return List.of(caldavCalendarEventsProvider, caldavCalendarTodosProvider, caldavCalendarEventsPlusTodosProvider);
    }

    private CaldavCalendarFacadeProvider chooseProvider(CollectionId collectionId) {
        for (val m : getAllProviders()) {
            if (m.existsCalendarWithId(collectionId)) return m;
        }
        throw new ResourceNotFoundException("Collection not found: " + collectionId);
    }

    private PassportUid parseUser(String user) throws ResourceNotFoundException {
        try {
            return userManager.getUidByEmail(Emails.punycode(user)).get();
        } catch (Exception e) {
            throw new ResourceNotFoundException("User not found: " + user, e);
        }
    }

    private UserInfo obtainUserInfo(PassportUid passportUid) {
        return userManager.getUserInfo(passportUid);
    }

    private void createDefaultCalendarsIfNotExist(UserInfo userInfo) {
        if (calendarDataSourceStatus.isMasterUnavailable()) return;

        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () -> {
            caldavCalendarEventsProvider.makeCalendarIfUserHasNoOne(userInfo);
            caldavCalendarTodosProvider.makeCalendarIfUserHasNoOne(userInfo);
        });
    }

    @Override
    public ListF<CalendarDescription> getUserOwnCalendars(PassportUid passportUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val userInfo = obtainUserInfo(passportUid);
            createDefaultCalendarsIfNotExist(userInfo);

            return caldavCalendarEventsProvider.getOwnCalendars(userInfo).plus(
                    caldavCalendarTodosProvider.getOwnCalendars(userInfo));
        });
    }

    @Override
    public ListF<CalendarDescription> getUserOwnCalendarsSharedToClient(PassportUid ownerUid, PassportUid clientUid) {
        Validate.notEquals(ownerUid, clientUid);

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val clientInfo = obtainUserInfo(clientUid);
            return caldavCalendarEventsProvider.getCalendarsSharedByAnotherUser(clientInfo, ownerUid).plus(
                    caldavCalendarTodosProvider.getCalendarsSharedByAnotherUser(clientInfo, ownerUid));
        });
    }

    @Override
    public ListF<CalendarDescription> getUserVisibleExternalCalendars(PassportUid passportUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val userInfo = obtainUserInfo(passportUid);
            return caldavCalendarEventsProvider.getVisibleExternalCalendars(userInfo).plus(
                    caldavCalendarTodosProvider.getVisibleExternalCalendars(userInfo));
        });
    }

    @Override
    public Email getUserEmail(String user) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                userManager.getUserByEmail(Emails.punycode(user)).get().getEmail().get());
    }

    @Override
    public void changeCalendar(PassportUid clientPassportUid, CollectionId collectionId, CalendarProperties properties) {
        if (calendarDataSourceStatus.isMasterUnavailable()) return;

        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () ->
                chooseProvider(collectionId).changeCalendar(obtainUserInfo(clientPassportUid), collectionId, properties));
    }

    @Override
    public void makeCalendar(PassportUid userUid, String id, ListF<String> componentNames, CalendarProperties properties) {
        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () -> {
            if (componentNames.containsTs(Component.VTODO)
                    && !componentNames.containsTs(Component.VEVENT))
            {
                caldavCalendarTodosProvider.makeCalendar(obtainUserInfo(userUid), id, properties);
            } else {
                caldavCalendarEventsProvider.makeCalendar(obtainUserInfo(userUid), id, properties);
            }
        });
    }

    @Override
    public void removeCalendar(PassportUid clientPassportUid, CollectionId collectionId) {
        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () ->
                chooseProvider(collectionId).removeCalendar(obtainUserInfo(clientPassportUid), collectionId));
    }

    @Override
    public CalendarDescription getUserCalendar(PassportUid clientPassportUid, CollectionId collectionId) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                chooseProvider(collectionId).getCalendarById(obtainUserInfo(clientPassportUid), collectionId));
    }

    @Override
    public ListF<CalendarComponent> getUserCalendarEvents(PassportUid clientUid, CollectionId collectionId,
            CalendarComponentConditions veventConditions, CalendarComponentConditions vtodoConditions,
            boolean includeIcs, UserAgentType userAgent)
    {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                chooseProvider(collectionId).getCalendarEntries(
                        obtainUserInfo(clientUid), collectionId, veventConditions, vtodoConditions,
                        exportOptions(clientUid, includeIcs, userAgent)));
    }

    @Override
    public Option<Long> getDefaultUserLayerId(PassportUid userUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> layerRoutines.getDefaultLayerId(userUid));
    }

    @Override
    public void moveUserCalendarEvent(PassportUid clientPassportUid,
            CollectionId sourceCollectionId, CollectionId destinationCollectionId, String fileName)
    {
        Validate.isTrue(fileName.endsWith(".ics"), "file name must end with .ics: ", fileName);

        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () ->
                chooseProvider(sourceCollectionId).moveCalendarEntry(
                        obtainUserInfo(clientPassportUid), fileName, sourceCollectionId, destinationCollectionId));
    }

    @Override
    public ListF<ComponentGetResult> getUserCalendarEvents(
            PassportUid clientUid, CollectionId collectionId, ListF<String> fileNames,
            boolean includeIcs, UserAgentType userAgent)
    {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                chooseProvider(collectionId).getCalendarEntries(
                        obtainUserInfo(clientUid), collectionId, fileNames, exportOptions(clientUid, includeIcs, userAgent)));
    }

    @Override
    public boolean existsCalendarWithId(CollectionId collectionId) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> existsCalendarWithIdUnsafe(collectionId));
    }

    private boolean existsCalendarWithIdUnsafe(CollectionId collectionId) {
        return StreamEx.of(getAllProviders())
            .anyMatch(provider -> provider.existsCalendarWithId(collectionId));
    }

    @Override
    public Option<CalendarComponent> getUserCalendarEvent(
            PassportUid clientUid, CollectionId collectionId, String fileName,
            boolean includeIcs, UserAgentType userAgent)
    {
        val componentGetResult = MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                getUserCalendarEvents(clientUid, collectionId, Cf.list(fileName), includeIcs, userAgent).single());
        return Option.x(componentGetResult.getEventDescription());
    }

    @Override
    public void putUserCalendarEvent(PassportUid clientUid, CollectionId collectionId,
            String fileName, Option<ETag> eTag, byte[] calendarData)
    {
        final var calendar = IcsUtils.parse(new ByteArrayInputStream(calendarData));

        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () ->
                chooseProvider(collectionId).putCalendarEntry(
                        obtainUserInfo(clientUid), calendar, collectionId, eTag.map(EventEtag::lastModifiedFromETag)));
    }

    @Override
    public void removeUserCalendarEvent(PassportUid clientUid, CollectionId collectionId, String fileName) {
        Validate.isTrue(fileName.endsWith(".ics"), "file name must end with .ics: ", fileName);

        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M, () ->
                chooseProvider(collectionId).removeCalendarEntry(obtainUserInfo(clientUid), fileName, collectionId));
    }

    @Override
    public ScheduleResponse notifyAttendees(PassportUid user, OutboxRequestVEvent req, PassportUid client, ListF<Email> recipients) {
        checkUserIsSameAsClient(user, client);

        return new ScheduleResponse(
            req.getAttendees()
                .map(ScheduleResponseItem.consF()
                    .bind2(new RequestStatus(StatusCode.SUCCESS))
                    .bind2(Option.empty())
                )
        );
    }

    private static void checkUserIsSameAsClient(PassportUid user, PassportUid client) {
        if (!user.equalsTs(client)) {
            throw new PermissionDeniedUserException();
        }
    }

    @Override
    public ScheduleResponse getFreeBusy(PassportUid userId, OutboxRequestVFreeBusy request, PassportUid clientUid) {
        checkUserIsSameAsClient(userId, clientUid);

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val emails = request.getEmails();

            Function<RequestStatus, ScheduleResponse> errorResponseF =
                    status -> new ScheduleResponse(emails.map(e -> new ScheduleResponseItem(e, status, Option.empty())));

            if (!request.getStart().isPresent() || !request.getEnd().isPresent()) {
                return errorResponseF.apply(new RequestStatus(
                        StatusCode.REQUEST_NOT_SUPPORTED, "Both DTSTART and DTEND must be specified"));
            }
            if (request.getStart().get().isAfter(request.getEnd().get())) {
                return errorResponseF.apply(new RequestStatus(
                        StatusCode.REQUEST_NOT_SUPPORTED, "DTSTART not expected to be after DTEND"));
            }
            try {
                val avails = availRoutines.getAvailabilityIntervalssByEmails(userId, emails,
                        AvailabilityRequest.interval(request.getStart().get(), request.getEnd().get()),
                        new ActionInfo(ActionSource.CALDAV, RequestIdStack.current().get(), Instant.now()));
                return new ScheduleResponse(
                        avails.map(t -> toFreeBusy(t.get1(), t.get2(), request.getStart().get(), request.getEnd().get())));
            } catch (Exception e) {
                return errorResponseF.apply(new RequestStatus(StatusCode.SERVICE_UNAVAILABLE));
            }
        });
    }

    private ScheduleResponseItem toFreeBusy(
            Email attendee, AvailabilityIntervalsOrRefusal availability, Instant start, Instant end)
    {
        try {
            if (availability.isRefusal()) {
                AvailabilityQueryRefusalReason reason = availability.getRefusalReason();
                return new ScheduleResponseItem(attendee, new RequestStatus(reason.getStatusCode()), Option.empty());
            }

            val intervals = availability.getIntervals().bounded().merged();
            val calendar = new Calendar();
            calendar.getProperties().add(Version.VERSION_2_0);
            calendar.getProperties().add(Method.REPLY);

            val vfreebusy = new VFreeBusy();

            vfreebusy.getProperties().add(new Uid(Random2.R.nextLetters(20)));

            for (AvailabilityInterval availInterval : intervals) {
                if (availInterval.getAvailability() == Availability.AVAILABLE)
                    continue;

                vfreebusy.getProperties().add(toFreeBusy(availInterval));
            }

            //vfreebusy.getProperties().add(new Organizer(new ))
            vfreebusy.getProperties().add(IcsUtils.attendee(attendee));

            vfreebusy.getProperties().add(new DtStart(IcsUtils.toDateTime(start)));
            vfreebusy.getProperties().add(new DtEnd(IcsUtils.toDateTime(end)));

            calendar.getComponents().add(vfreebusy);

            return new ScheduleResponseItem(attendee, new RequestStatus(StatusCode.SUCCESS), Option.of(calendar));
        } catch (Exception e) {
            log.error("Failed to create ScheduleResponseItem", e);
            return new ScheduleResponseItem(attendee, new RequestStatus(StatusCode.SERVICE_UNAVAILABLE), Option.empty());
        }
    }

    private static FreeBusy toFreeBusy(AvailabilityInterval availInterval) {
        val period = IcsUtils.period(
                availInterval.getInterval().getStart().toInstant(),
                availInterval.getInterval().getEnd().toInstant());
        val periodList = new PeriodList();
        periodList.add(period);
        val freeBusy = new FreeBusy(periodList);
        freeBusy.getParameters().add(availInterval.getAvailability().toFbType());
        return freeBusy;
    }

    @Override
    public DavSyncToken getUserSyncToken(PassportUid userUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val layersMax = layerDao.findLayersByLayerUserUid(userUid).map(Layer.getCollLastUpdateTsF()).maxO();
            val todosMax = todoDao.findMaxTodoListLastUpdateTs(userUid);

            return LayerSyncToken.lastUpdateTsToSyncToken(layersMax.plus(todosMax).maxO().toOptional());
        });
    }

    @Override
    public List<ComponentModified> getUserCalendarModifiedEvents(
            PassportUid clientUid, CollectionId collectionId, DavSyncToken syncToken,
            boolean includeIcs, UserAgentType userAgent)
    {
        val sync = LayerSyncToken.parseSyncToken(syncToken);
        val since = sync.getInstant();

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            val user = obtainUserInfo(clientUid);
            val provider = chooseProvider(collectionId);
            val options = exportOptions(clientUid, includeIcs, userAgent);

            val modifiedAndMissed =
                    provider.getCalendarEntriesCreatedOrModifiedSince(user, collectionId, since, options)
                            .partition(c -> c.getEventDescription().isPresent());

            val modified = modifiedAndMissed.get1().map(ComponentGetResult::getFileName).unique();

            val deleted = StreamEx.of(provider.getCalendarEntriesDeletedSince(user, collectionId, since))
                    .append(modifiedAndMissed.get2())
                    .filter(c -> !modified.containsTs(c.getFileName()))
                    .toImmutableList();

            val components = modifiedAndMissed.get1().plus(deleted);

            return filterBySyncInfo(components, sync);
        });
    }

    private List<ComponentModified> filterBySyncInfo(List<ComponentModified> components, DavSyncInfo sync) {
        return StreamEx.of(components)
                .filter(c -> match(c, sync))
                .toImmutableList();
    }

    private static boolean match(ComponentModified component, DavSyncInfo sync) {
        val componentUpdate = component.getDate();
        val syncUpdate = sync.getInstant();
        if (componentUpdate.isAfter(syncUpdate)) {
            return true;
        }
        if (componentUpdate.isBefore(syncUpdate)) {
            return false;
        }
        val result = sync.getExternalId().stream().noneMatch(extId -> extId.compareTo(component.getFileName()) >= 0);
        log.debug("Component {} has the same instant as sync {}, try to filter by id if there is any. Match: {}", component.getFileName(), sync, result);
        return result;

    }

    @Override
    public DavSyncToken getCalendarSyncToken(CollectionId collectionId) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () ->
                chooseProvider(collectionId).getCalendarSyncToken(collectionId));
    }

    @Override
    public PassportUid checkAndGetPassportUid(String user) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> parseUser(user));
    }

    private ExportOptions exportOptions(PassportUid clientUid, boolean includeIcs, UserAgentType userAgent) {
        val etcGmtTzs = new HashSet(etcGmtForTimezones.get());

        if (withRussianEtcGmtTimezones(userAgent)) {
            val users = userManager.getYtUserByUid(clientUid);
            final var uid = StreamEx.of(users)
                .findFirst()
                .map(u -> PassportUid.cons(u.getUid().getValue()));
            val settings = settingsRoutines.getSettingsByUidIfExistsBatch(Option.x(uid)).values().singleO();

            if (settings.exists(s -> s.getCommon().getHackCaldavTimezones())) {
                etcGmtTzs.addAll(AuxDateTime.RUSSIAN_TZ_IDS);
            }
        }
        val includeDeclined = userAgent == UserAgentType.ICAL
                || userAgent == UserAgentType.IOS_8_0_AND_LOWER
                || userAgent == UserAgentType.IOS_8_1_AND_HIGHER;

        return new ExportOptions(includeIcs, includeDeclined, withFullTimezones(userAgent), etcGmtTzs);
    }

    private static boolean withFullTimezones(UserAgentType userAgent) {
        return userAgent != UserAgentType.WINDOWS_PHONE; // CAL-6702
    }

    private static final Set<UserAgentType> RUSSIAN_ETC_GMT_USER_AGENTS = Set.of(
            UserAgentType.IOS_8_0_AND_LOWER,
            UserAgentType.CALDAVSYNC_ANDROID,
            UserAgentType.CALDAVSYNCADAPTER_ANDROID,
            UserAgentType.DAVDROID,
            UserAgentType.YANDEX_KIT);

    private static boolean withRussianEtcGmtTimezones(UserAgentType userAgent) {
        return RUSSIAN_ETC_GMT_USER_AGENTS.contains(userAgent);
    }
}
