package ru.yandex.calendar.frontend.caldav.proto.facade;

import java.util.List;

import net.fortuna.ical4j.model.Component;
import one.util.streamex.StreamEx;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.impl.LayerCtag;
import ru.yandex.calendar.frontend.caldav.impl.LayerSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavWithHttpStatusCodeException;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentConditions;
import ru.yandex.calendar.frontend.caldav.proto.tree.CollectionId;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.logic.beans.generated.TodoItem;
import ru.yandex.calendar.logic.beans.generated.TodoList;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.exp.IcsSingleTodoItemExportData;
import ru.yandex.calendar.logic.ics.exp.IcsTodoExporter;
import ru.yandex.calendar.logic.ics.imp.IcsTodoImporter;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoRoutines;
import ru.yandex.calendar.logic.todo.id.ListIdOrExternalId;
import ru.yandex.calendar.logic.todo.id.TodoIdOrExternalId;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.reqid.RequestIdStack;

public class CaldavCalendarTodosProvider implements CaldavCalendarFacadeProvider {
    @Autowired
    private TodoRoutines todoRoutines;
    @Autowired
    private IcsTodoImporter icsTodoImporter;
    @Autowired
    private IcsTodoExporter icsTodoExporter;
    @Autowired
    private TodoDao todoDao;
    @Autowired
    private UserManager userManager;
    @Autowired
    private LockTransactionManager lockTransactionManager;


    private long getTodoListId(CollectionId collectionId) {
        if (collectionId.isTodos()) {
            return Long.parseLong(collectionId.getId());
        } else if (collectionId.isUnknown()) {
            PassportUid uid = userManager.getUidByEmail(new Email(collectionId.getUser())).get();
            return todoDao.findNotDeletedTodoListIdByCaldavCollId(uid, collectionId.getId()).get();
        }
        throw new IllegalArgumentException("Invalid collection id: " + collectionId);
    }

    private void ensureTodoListCreatedByUser(UserInfo userInfo, long todoListId) {
        if (!todoDao.existsAndNotDeletedTodoListWithIdCreatedByUid(todoListId, userInfo.getUid())) {
            throw new PermissionDeniedUserException("Todo list not found or created by another user");
        }
    }

    private ActionInfo getActionInfo() {
        return new ActionInfo(ActionSource.CALDAV, RequestIdStack.current().get(), Instant.now());
    }


    @Override
    public boolean existsCalendarWithId(CollectionId collectionId) {
        Option<PassportUid> uid = userManager.getUidsByEmails(Email.parseSafe(collectionId.getUser())).single().get2();

        if (collectionId.isTodos() && uid.isPresent()) {
            Option<Long> id = Cf.Long.parseSafe(collectionId.getId());
            return id.isPresent() && todoDao.existsAndNotDeletedTodoListWithIdCreatedByUid(id.get(), uid.get());
        } else if (collectionId.isUnknown() && uid.isPresent()) {
            return todoDao.findNotDeletedTodoListIdByCaldavCollId(uid.get(), collectionId.getId()).isPresent();
        }
        return false;
    }

    @Override
    public void makeCalendarIfUserHasNoOne(UserInfo userInfo) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            todoRoutines.createTodoListIfUserHasNoLists(userInfo.getUid(), getActionInfo());
        });
    }

    @Override
    public ListF<CalendarDescription> getOwnCalendars(UserInfo userInfo) {
        return todoRoutines.getNonDeletedTodoLists(userInfo.getUid()).map(toCalendarDescriptionF());
    }

    @Override
    public ListF<CalendarDescription> getCalendarsSharedByAnotherUser(UserInfo userInfo, PassportUid owner) {
        return Cf.list();
    }

    @Override
    public ListF<CalendarDescription> getVisibleExternalCalendars(UserInfo userInfo) {
        return Cf.list();
    }

    @Override
    public CalendarDescription getCalendarById(UserInfo userInfo, CollectionId collectionId) {
        long todoListId = getTodoListId(collectionId);
        ensureTodoListCreatedByUser(userInfo, todoListId);

        return toDescription(todoRoutines.getTodoLists(userInfo.getUid(), Cf.list(todoListId)).single());
    }

    @Override
    public void makeCalendar(UserInfo userInfo, String id, CalendarProperties properties) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            if (todoDao.findNotDeletedTodoListIdByCaldavCollId(userInfo.getUid(), id).isPresent()) { // CAL-7232
                throw new CaldavWithHttpStatusCodeException(
                        HttpStatus.SC_409_CONFLICT, "todo list with id " + id + " already exists");
            }
            TodoList data = extractTodoListData(properties);
            data.setCaldavCollId(id);
            todoRoutines.createTodoList(userInfo.getUid(), data, getActionInfo());

        });
    }

    @Override
    public void changeCalendar(UserInfo userInfo, CollectionId collectionId, CalendarProperties properties) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            long todoListId = getTodoListId(collectionId);
            ensureTodoListCreatedByUser(userInfo, todoListId);
            TodoList data = extractTodoListData(properties);

            if (data.cardinality() > 0) {
                todoRoutines.updateTodoListById(userInfo.getUid(), data, ListIdOrExternalId.id(todoListId), getActionInfo());
            }
        });
    }

    @Override
    public DavSyncToken getCalendarSyncToken(CollectionId collectionId) {
        long todoListId = getTodoListId(collectionId);
        return LayerSyncToken.lastUpdateTsToSyncToken(todoDao.findTodoListLastUpdateTs(todoListId).toOptional());
    }

    @Override
    public void removeCalendar(UserInfo userInfo, CollectionId collectionId) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            long todoListId = getTodoListId(collectionId);
            ensureTodoListCreatedByUser(userInfo, todoListId);

            todoRoutines.deleteTodoList(userInfo.getUid(), ListIdOrExternalId.id(todoListId), getActionInfo());
        });
    }

    @Override
    public void putCalendarEntry(
            UserInfo userInfo, IcsCalendar icsCalendar, CollectionId collectionId, Option<Instant> notModofiedSince)
    {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            long todoListId = getTodoListId(collectionId);
            ensureTodoListCreatedByUser(userInfo, todoListId);

            icsTodoImporter.importTodos(userInfo.getUid(), icsCalendar, getActionInfo(), todoListId);
        });
    }

    @Override
    public void moveCalendarEntry(UserInfo userInfo, String fileName, CollectionId from, CollectionId to) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            long fromId = getTodoListId(from);
            long toId = getTodoListId(to);

            ensureTodoListCreatedByUser(userInfo, fromId);
            ensureTodoListCreatedByUser(userInfo, toId);

            todoRoutines.moveTodoItemByExternalId(
                    userInfo.getUid(), IcsNameUtils.fileNameToExternalId(fileName), fromId, toId, getActionInfo());
        });
    }

    @Override
    public void removeCalendarEntry(UserInfo userInfo, String fileName, CollectionId collectionId) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.todoUser(userInfo.getUid()), () -> {
            long todoListId = getTodoListId(collectionId);
            ensureTodoListCreatedByUser(userInfo, todoListId);

            todoRoutines.deleteTodoItemById(userInfo.getUid(),
                    TodoIdOrExternalId.externalId(IcsNameUtils.fileNameToExternalId(fileName)), getActionInfo());
        });
    }

    // XXX: condition unused
    @Override
    public ListF<CalendarComponent> getCalendarEntries(
            UserInfo userInfo, CollectionId collectionId,
            CalendarComponentConditions eventConditions, CalendarComponentConditions todoConditions,
            ExportOptions options)
    {
        long todoListId = getTodoListId(collectionId);
        ensureTodoListCreatedByUser(userInfo, todoListId);

        if (todoConditions.isFalse()) return Cf.list();

        return icsTodoExporter
                .exportTodoItemsByListIdForCaldav(todoListId, userInfo.getUid())
                .map(CalendarComponent.fromTodoExportF(options));
    }

    @Override
    public ListF<ComponentGetResult> getCalendarEntries(
            UserInfo userInfo, CollectionId collectionId, ListF<String> fileNames, ExportOptions options)
    {
        long todoListId = getTodoListId(collectionId);
        ensureTodoListCreatedByUser(userInfo, todoListId);

        ListF<IcsSingleTodoItemExportData> exports = icsTodoExporter.exportTodoItemsByExternalIdAndTodoListIdForCaldav(
                userInfo.getUid(), fileNames.map(IcsNameUtils::fileNameToExternalId), todoListId);

        ListF<ComponentGetResult> found = exports
                .map(CalendarComponent.fromTodoExportF(options).andThen(ComponentModified::found));

        MapF<String, ComponentGetResult> foundByFileName = found.toMapMappingToKey(ComponentGetResult::getFileName);

        return fileNames.map(n -> foundByFileName.getO(n).getOrElse(() -> ComponentGetResult.notFound(n)));
    }

    @Override
    public ListF<ComponentModified> getCalendarEntriesCreatedOrModifiedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since, ExportOptions options)
    {
        long todoListId = getTodoListId(collectionId);
        ensureTodoListCreatedByUser(userInfo, todoListId);

        return icsTodoExporter
                .exportTodoItemsByListIdForCaldavCreatedOrModifiedSince(todoListId, userInfo.getUid(), since)
                .map(e -> ComponentModified.found(CalendarComponent.fromTodoExport(e, options)));
    }

    @Override
    public List<ComponentModified> getCalendarEntriesDeletedSince(
            UserInfo userInfo, CollectionId collectionId, Instant since)
    {
        long todoListId = getTodoListId(collectionId);
        ensureTodoListCreatedByUser(userInfo, todoListId);

        return StreamEx.of(todoDao.findDeletedTodoItemsByTodoListIdAndMinDeletedTs(todoListId, since))
            .mapToEntry(TodoItem::getExternalId, TodoItem::getDeletionTs)
            .flatMapValues(Option::stream)
            .mapKeys(IcsNameUtils::externalIdToFileName)
            .mapKeyValue(ComponentModified::deleted)
            .toImmutableList();
    }


    private TodoList extractTodoListData(CalendarProperties properties) {
        TodoList data = new TodoList();
        if (properties.getColor().isPresent()) {
            data.setColor(properties.getColor().get().getColor().getRgb());
        }
        if (properties.getDisplayName().isPresent()) {
            data.setTitle(properties.getDisplayName().get());
        }
        return data;
    }

    private CalendarDescription toDescription(TodoList list) {
        Color color = list.getColor().isPresent() ? Color.fromRgb(list.getColor().get()) : TodoRoutines.DEFAULT_LIST_COLOR;

        String user = userManager.getEmailByUid(list.getCreatorUid()).get().getEmail();
        boolean writable = true;

        CollectionId collectionId;
        if (!list.getCaldavCollId().isPresent()) {
            collectionId = CollectionId.todos(user, Long.toString(list.getId()), list.getCreatorUid());
        } else {
            collectionId = CollectionId.unknown(user, list.getCaldavCollId().get(), list.getCreatorUid());
        }
        Option<Instant> collLastUpdate = Option.of(list.getLastUpdateTs());

        return new CalendarDescription(
                Option.of(user), collectionId, list.getTitle(), new IcalColor(color),
                writable, Cf.list(Component.VTODO),
                LayerCtag.lastUpdateTsToCTag(collLastUpdate),
                Option.of(LayerSyncToken.lastUpdateTsToSyncToken(collLastUpdate.toOptional())));
    }

    private Function<TodoList, CalendarDescription> toCalendarDescriptionF() {
        return this::toDescription;
    }

}
