package ru.yandex.calendar.frontend.caldav.proto.facade;

import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.ResourceType;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.WebdavUtils;
import ru.yandex.calendar.frontend.caldav.impl.EventEtag;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.tree.CaldavResourceUserCalendarEntryBase;
import ru.yandex.calendar.frontend.caldav.proto.tree.PropertiesCollection;
import ru.yandex.calendar.frontend.caldav.proto.tree.PropertiesCollectionBase;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.logic.ics.exp.IcsEventGroupExportData;
import ru.yandex.calendar.logic.ics.exp.IcsSingleTodoItemExportData;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;

public class CalendarComponent extends PropertiesCollectionBase implements PropertiesCollection {
    private final String fileName;
    private final Option<String> ical;
    private final ETag etag;
    private final CalendarItemType type;

    public CalendarComponent(String fileName, Option<String> ical, ETag etag, CalendarItemType type) {
        this.fileName = fileName;
        this.ical = ical;
        this.etag = etag;
        this.type = type;
    }

    public String getFileName() {
        return fileName;
    }

    public Option<String> getIcal() {
        return ical;
    }

    public ETag getEtag() {
        return etag;
    }

    public CalendarItemType getType() {
        return type;
    }

    @Override
    public DavPropertyName[] getPropertyNames() {
        return CaldavResourceUserCalendarEntryBase.propertyNames();
    }

    /**
     * @see CaldavResourceUserCalendarEntryBase#getProperty(DavPropertyName)
     */
    @Override
    public DavProperty<?> getProperty(DavPropertyName name) {
        if (name.equals(WebdavConstants.DAV_RESOURCETYPE_PROP)) {
            return new ResourceType(new int[0]);
        } else if (name.equals(WebdavConstants.DAV_GETETAG_PROP)) {
            return new DefaultDavProperty<>(WebdavConstants.DAV_GETETAG_PROP, etag.getValue());
        } else if (name.equals(CaldavConstants.CALDAV_CALENDAR_DATA_PROP)) {
            return new DefaultDavProperty<>(CaldavConstants.CALDAV_CALENDAR_DATA_PROP, ical.getOrElse(""));
        } else if (name.equals(WebdavConstants.DAV_GETCONTENTTYPE_PROP)) {
            return new DefaultDavProperty<>(WebdavConstants.DAV_GETCONTENTTYPE_PROP, "text/calendar");
        } else {
            return null;
        }
    }

    @Override
    public DavPropertySet getProperties() {
        return WebdavUtils.getPropertiesFromNames(this);
    }

    public static Function<CalendarComponent, String> getIcalF() {
        return d -> d.getIcal().get();
    }

    public static Function<CalendarComponent, String> getFileNameF() {
        return CalendarComponent::getFileName;
    }

    public static CalendarComponent fromEventGroupExport(IcsEventGroupExportData group, ExportOptions options) {
        return new CalendarComponent(
                IcsNameUtils.externalIdToFileName(group.getExternalId().getRaw()),
                group.toCalendarO(options).map(IcsCalendar.serializeToStringF()),
                EventEtag.lastModifiedToEtag(group.getLastModified()),
                CalendarItemType.VEVENT);
    }

    public static CalendarComponent fromTodoExport(IcsSingleTodoItemExportData todo, ExportOptions options) {
        return new CalendarComponent(
                IcsNameUtils.externalIdToFileName(todo.getExternalId()),
                Option.of(todo.toCalendar(options).serializeToString()),
                EventEtag.lastModifiedToEtag(todo.getLastModified()),
                CalendarItemType.VTODO);
    }

    public static Function<IcsEventGroupExportData, CalendarComponent> fromEventGroupExportF(ExportOptions options) {
        return group -> fromEventGroupExport(group, options);
    }

    public static Function<IcsSingleTodoItemExportData, CalendarComponent> fromTodoExportF(ExportOptions options) {
        return todo -> fromTodoExport(todo, options);
    }

}
