package ru.yandex.calendar.frontend.caldav.proto.facade;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.PutResponse;
import ru.yandex.calendar.frontend.caldav.proto.carddav.DirectoryCardData;
import ru.yandex.calendar.frontend.caldav.proto.carddav.report.CardComponentFilter;
import ru.yandex.calendar.frontend.caldav.proto.carddav.report.PropFilter;
import ru.yandex.calendar.frontend.caldav.proto.carddav.report.PropFilterPredicate;
import ru.yandex.calendar.frontend.caldav.proto.tree.ResourceNotFoundException;
import ru.yandex.calendar.logic.contact.addressbook.AddressBook;
import ru.yandex.calendar.logic.contact.directory.DirectoryEntry;
import ru.yandex.calendar.logic.contact.directory.DirectoryManager;
import ru.yandex.calendar.logic.contact.directory.DirectoryType;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchField;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchFieldOperator;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchPredicate;
import ru.yandex.calendar.logic.ics.iv5j.vcard.VcfVCard;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfEmail;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfFn;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfN;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfOrg;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfPhoto;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfTel;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfTitle;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfUid;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfVersion;
import ru.yandex.calendar.logic.ics.iv5j.vcard.property.VcfXProperty;
import ru.yandex.calendar.logic.user.Avatar;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @see CaldavCalendarFacadeImpl
 */
public class CarddavCalendarFacadeImpl implements CarddavCalendarFacade {
    private static final Logger logger = LoggerFactory.getLogger(CarddavCalendarFacadeImpl.class);

    @Autowired
    private UserManager userManager;
    @Autowired
    private AddressBook addressBook;
    @Autowired
    private DirectoryManager directoryManager;

    @Override
    public ListF<ContactEtag> getUserAddressbookContactEtags(PassportUid userUid, PassportUid clientUid) {
        Validate.equals(clientUid, userUid);
        return addressBook.getCarddavContactEtags(clientUid);
    }

    @Override
    public Tuple2List<String, Option<ContactVcard>> getUserAddressbookContactsByFileNames(PassportUid userUid,
                                                                                          ListF<String> fileNames,
                                                                                          PassportUid clientUid)
    {
        Validate.equals(clientUid, userUid);
        return addressBook.getCarddavContactsByFileNames(clientUid, fileNames);
    }

    @Override
    public Option<ContactVcard> getUserAddressbookContact(PassportUid userUid, String fileName, PassportUid clientUid) {
        return getUserAddressbookContactsByFileNames(userUid, Cf.list(fileName), clientUid).single()._2;
    }

    @Override
    public PutResponse putUserAddressbookContact(
            PassportUid userUid, String fileName, Option<ETag> ifMatchEtag, byte[] vcf, PassportUid clientUid)
    {
        Validate.equals(clientUid, userUid);

        VcfVCard vcard = VcfVCard.parse(vcf);
        return addressBook.putCarddavContact(clientUid, fileName, ifMatchEtag.toOptional(), vcard);
    }

    @Override
    public void removeUserAddressbookCard(PassportUid userUid, String fileName, PassportUid clientUid) {
        Validate.equals(clientUid, userUid);
        addressBook.deleteCarddavContact(clientUid, fileName);
    }

    private Option<DirectorySearchField> convertField(String vcfFieldName) {
        switch (vcfFieldName) {
            case "NICKNAME":
                return Option.of(DirectorySearchField.NICKNAME);
            case VcfFn.FN:
                return Option.of(DirectorySearchField.DISPLAY_NAME);
            case VcfEmail.EMAIL:
                return Option.of(DirectorySearchField.EMAIL);
            default:
                return Option.empty();
        }
    }

    private DirectorySearchFieldOperator convertOperator(PropFilterPredicate.TextMatch.MatchType matchType) {
        switch (matchType) {
            case CONTAINS:
                return DirectorySearchFieldOperator.CONTAINS;
            case STARTS_WITH:
                return DirectorySearchFieldOperator.STARTS_WITH;
            default:
                logger.warn("Unsupported operator, using CONTAINS: " + matchType);
                return DirectorySearchFieldOperator.CONTAINS;
        }
    }

    private DirectorySearchPredicate convertPropFilter(final PropFilter propFilter) {
        ListF<DirectorySearchPredicate> predicates = propFilter.getPredicates().map(propFilterPredicate -> {

            if (propFilterPredicate instanceof PropFilterPredicate.TextMatch) {
                PropFilterPredicate.TextMatch textMatch = (PropFilterPredicate.TextMatch) propFilterPredicate;
                Option<DirectorySearchField> field = convertField(propFilter.getName());
                if (!field.isPresent()) {
                    return DirectorySearchPredicate.falsePredicate();
                } else {
                    DirectorySearchFieldOperator operator = convertOperator(textMatch.getMatchType());
                    return DirectorySearchPredicate.fieldPredicate(field.get(), operator, textMatch.getText());
                }
            } else {
                return DirectorySearchPredicate.falsePredicate();
            }
        });

        switch (propFilter.getOperator()) {
            case ALLOF:
                return DirectorySearchPredicate.allPredicate(predicates);
            case ANYOF:
                return DirectorySearchPredicate.anyPredicate(predicates);
            default:
                throw new IllegalArgumentException("Unexpected operator " + propFilter.getOperator());
        }
    }

    private Function<PropFilter, DirectorySearchPredicate> convertPropFilterF() {
        return this::convertPropFilter;
    }

    private DirectorySearchPredicate convertFilter(CardComponentFilter cardFilter) {
        ListF<DirectorySearchPredicate> predicates = cardFilter.getPropFilters().map(convertPropFilterF());

        switch (cardFilter.getOperator()) {
            case ALLOF:
                return DirectorySearchPredicate.allPredicate(predicates);
            case ANYOF:
                return DirectorySearchPredicate.anyPredicate(predicates);
            default:
                throw new IllegalArgumentException("Unexpected operator " + cardFilter.getOperator());
        }
    }

    @Override
    public ListF<DirectoryCardData> getUserDirectoryContacts(CardComponentFilter cardComponentFilter, PassportUid clientUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            DirectorySearchPredicate directorySearchPredicate = convertFilter(cardComponentFilter);

            ListF<String> queries = directorySearchPredicate.getLeavesFieldPredicatesValues();
            ListF<DirectorySearchPredicate> carNumberPredicates = queries.map(DirectorySearchPredicate.fieldPredicateF(
                    DirectorySearchField.CAR_NUMBER, DirectorySearchFieldOperator.STARTS_WITH));

            directorySearchPredicate = DirectorySearchPredicate.anyPredicate(
                    carNumberPredicates.plus(directorySearchPredicate)); // CAL-4447

            ListF<DirectoryEntry> entries = directoryManager.findContacts(clientUid, directorySearchPredicate, true, 8);

            return entries.flatMap(directoryEntry -> {
                try {
                    return Option.of(makeDirectoryCardData(directoryEntry));
                } catch (Exception e) {
                    logger.warn(e, e);
                    return Option.empty();
                }
            });
        });
    }

    @Override
    public Option<DirectoryCardData> getUserDirectoryContact(String id, PassportUid clientUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            Option<DirectoryEntry> directoryEntry = directoryManager.getEntry(clientUid, id, true);
            if (!directoryEntry.isPresent()) {
                return Option.empty();
            }
            return Option.of(makeDirectoryCardData(directoryEntry.get()));
        });
    }

    @Override
    public Option<DirectoryType> getUserDirectory(PassportUid clientUid) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            return directoryManager.getDirectory(clientUid);
        });
    }

    private DirectoryCardData makeDirectoryCardData(DirectoryEntry directoryEntry) {
        VcfVCard vcf = new VcfVCard();
        vcf = vcf.addProperty(VcfVersion.VERSION_3_0);
        vcf = vcf.addProperty(new VcfEmail(directoryEntry.getEmail()));

        if (directoryEntry.getJabber().isPresent()) {
            vcf = vcf.addProperty(new VcfXProperty("X-JABBER", directoryEntry.getJabber().get().getEmail(), Cf.list()));
        }

        ListF<String> cellPhones = directoryEntry.getCellPhones();
        if (cellPhones.isNotEmpty()) {
            vcf = vcf.addProperty(new VcfTel(cellPhones.first(), "cell"));

            if (cellPhones.size() > 1) {
                vcf = vcf.addProperty(new VcfTel(cellPhones.get(1), "main"));
            }
            // iPhone directory client does not show phones with TYPE=other but we try
            for (String phone : cellPhones.drop(2)) {
                vcf = vcf.addProperty(new VcfTel(phone, "other"));
            }
        }
        vcf = vcf.addPropertyIfNotBlank(new VcfTel(directoryEntry.getWorkPhone(), "work"));

        String familyName = directoryEntry.getLastName();
        String givenName = directoryEntry.getFirstName();

        if (StringUtils.isNotBlank(familyName.concat(givenName))) {
            vcf = vcf.addProperty(new VcfN(familyName, givenName, "", "", ""));
        }
        vcf = vcf.addPropertyIfNotBlank(new VcfFn(directoryEntry.getDisplayName()));

        vcf = vcf.addPropertyIfNotBlank(new VcfTitle(directoryEntry.getJobTitle()));
        vcf = vcf.addPropertyIfNotBlank(new VcfOrg(directoryEntry.getJobOrganization()));

        vcf = vcf.addProperty(new VcfUid(directoryEntry.getExternalId()));

        Option<Avatar> avatar = directoryEntry.getAvatar();
        if (avatar.isPresent() && avatar.get().getContentType().isPresent()) {
            vcf = vcf.addProperty(new VcfPhoto(avatar.get().getContentType().get().getSubtype(),
                    FastBase64Coder.encode(avatar.get().getImageData())));
        }
        return new DirectoryCardData(directoryEntry.getExternalId(), ETag.random(), vcf);
    }

    private PassportUid parseUser(String user) throws ResourceNotFoundException {
        MasterSlaveContextHolder.PolicyHandle h = MasterSlaveContextHolder.push(MasterSlavePolicy.R_MS);
        try {
            return userManager.getUidByEmail(Emails.punycode(user)).get();
        } catch (Exception e) {
            throw new ResourceNotFoundException("User not found: " + user, e);
        } finally {
            h.popSafely();
        }
    }
}
