package ru.yandex.calendar.frontend.caldav.proto.facade;

import java.util.NoSuchElementException;
import java.util.Optional;

import lombok.val;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.HrefProperty;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.caldav.EmailProperty;
import ru.yandex.calendar.frontend.caldav.proto.ccdav.AddressbookSearchOperator;
import ru.yandex.calendar.frontend.caldav.proto.ccdav.CcDavUtils;
import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;
import ru.yandex.calendar.frontend.caldav.proto.tree.CalendarUrls;
import ru.yandex.calendar.frontend.caldav.proto.tree.PropertiesCollection;
import ru.yandex.calendar.frontend.caldav.proto.tree.PropertiesCollectionImpl;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertySearch;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequestPrincipalPropertySearch;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusResponse2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.PropStat;
import ru.yandex.calendar.frontend.caldav.userAgent.UserAgentType;
import ru.yandex.calendar.log.LogMarker;
import ru.yandex.calendar.logic.contact.directory.DirectoryEntry;
import ru.yandex.calendar.logic.contact.directory.DirectoryManager;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchField;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchFieldOperator;
import ru.yandex.calendar.logic.contact.directory.search.DirectorySearchPredicate;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.ics.iv5j.ical.parameter.IcsCuType;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.calendar.util.idlent.YandexUser;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.YandexPassport;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthStatus;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.ip.IpAddress;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.calendar.logic.user.UserManager.ALLOW_BY_PASSPORT_ATTRIB_VALUE;

/**
 * @see CaldavCalendarFacade
 * @see CarddavCalendarFacade
 */
public class CcCalendarFacadeImpl implements CcCalendarFacade {
    private static final Logger logger = LoggerFactory.getLogger(CcCalendarFacadeImpl.class);

    @Autowired
    private UserManager userManager;
    @Autowired
    private DirectoryManager directoryManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private YandexPassport yandexPassport;

    @Override
    public CcUserInfo checkPassword(String user, String password, IpAddress userIp, String userAgent) {
        Email email;
        try {
            email = Emails.punycode(user);
        } catch (IllegalArgumentException e) {
            logger.error("Illegal email received for user" + user);
            throw new AuthenticationFailedException(e);
        }
        if (StringUtils.isEmpty(password)) {
            logger.error("Empty password received for " + user);
            throw new AuthenticationFailedException("empty password");
        }
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            Option<PassportUid> uidO = userManager.getUidByEmail(email);

            if (!uidO.isPresent()) {
                logger.error("User " + user + " not found by e-mail");
                throw new AuthenticationFailedException("user not found by e-mail " + user);
            }
            PassportUid uid = uidO.get();

            try {
                logger.debug("Checking password " + LogMarker.currentUser(uid, email));
                BlackboxAbstractResponse bbResponse = userManager.checkPasswordByEmail(email, password, userIp);
                if (!bbResponse.getO().isPresent()) {
                    // GREG-898: Проверять oauth аутентификацию в корповом Caldav
                    logger.warn("Invalid password, checking password as OAuth token for "
                            + LogMarker.currentUser(uid, email) + " "
                            + logPassword(user, password, userIp, userAgent));
                    return checkOAuthToken(password, userIp, userAgent);
                }
                // GREG-907
                BlackboxCorrectResponse blackboxCorrectResponse = bbResponse.getO().get();
                Optional<String> allowByPassportO =
                        blackboxCorrectResponse.getAttributes().getOptional(ALLOW_BY_PASSPORT_ATTRIB_VALUE);
                if (blackboxCorrectResponse.getStatus() == BlackboxOAuthStatus.INVALID.getId() && (allowByPassportO.isEmpty() || allowByPassportO.get().equals("0"))) {
                    // GREG-907: Если status=INVALID и не "Можно в carddav и caldav при помощи обычного пароля", в описании ошибки рекламирует включение паролей приложений
                    logger.error("Allow application passwords for user  " + user + ", see ticker GREG-907, status: " + blackboxCorrectResponse.getStatus());
                }
                if (blackboxCorrectResponse.getStatus() == BlackboxOAuthStatus.VALID.getId() && allowByPassportO.isPresent() && allowByPassportO.get().equals("1")) {
                    // GREG-907: Если status=VALID и "Можно в carddav и caldav при помощи обычного пароля", рекламирует включение паролей приложений другими средствами.
                    logger.error("Allow application passwords for user  " + user + ", see ticker GREG-907, status: " + blackboxCorrectResponse.getStatus());
                }
            } catch (Exception e) {
                logger.error("Authentication process failed for " + logPassword(user, password, userIp, userAgent), e);
                throw e;
            }

            return new CcUserInfo(uid, email, CcClientScope.ALL);
        });
    }

    @NotNull
    private String logPassword(String user, String password, IpAddress userIp, String userAgent) {
        return "user = " + user + ", password = " + secure(password) + ", userIp = " + userIp + ", " +
                "userAgent = " + userAgent;
    }

    @Override
    public CcUserInfo checkOAuthToken(String token, IpAddress userIp, String userAgent)
            throws AuthenticationFailedException
    {
        PassportAuthDomain authDomain = passportAuthDomainsHolder.contains(PassportAuthDomain.PUBLIC)
                ? PassportAuthDomain.PUBLIC
                : PassportAuthDomain.YANDEX_TEAM_RU;

        YandexUser user;
        CcClientScope scope;
        try {
            Tuple2<Option<YandexUser>, ListF<String>> userAndScopes =
                    userManager.getUserAndScopesByOAuthToken(authDomain, token, userIp);

            user = userAndScopes.get1().getOrThrow("Yandex-user not found by oauth!");

            if (userAndScopes.get2().containsTs("mobile:all")) {
                scope = CcClientScope.ALL;
            } else if (userAndScopes.get2().containsTs("calendar:all")) {
                scope = CcClientScope.CALENDAR;
            } else {
                logger.error("Authentication failed: Oauth token doesn't have acceptable scopes for " + loggingInfo(token, userIp, userAgent));
                throw new AuthenticationFailedException("Oauth token doesn't have acceptable scopes");
            }

        } catch (BlackboxOAuthException e) {
            logger.error("Authentication failed for " + loggingInfo(token, userIp, userAgent), e);
            throw new AuthenticationFailedException(e);
        } catch (Exception e) {
            logger.error("Authentication process failed", e);
            throw e;
        }

        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            PassportUid uid = user.getUid();
            Email email;

            if (!user.getEmail().isPresent()) { // CAL-6641
                yandexPassport.admSubscribe(uid.getUid(), "mail");
                email = userManager.getEmailByUid(uid)
                    .orElseThrow(() -> new NoSuchElementException("cannot get email by uid " + uid));
            } else {
                email = user.getEmail().get();
            }

            return new CcUserInfo(uid, email, scope);
        });
    }

    @NotNull
    private String loggingInfo(String token, IpAddress userIp, String userAgent) {
        return "token = " + secure(token) + ", userIp = " + userIp + ", userAgent = " + userAgent;
    }

    private static String secure(String token) {
        if (token == null) {
            return "-";
        }
        if (token.length() >= 1) {
            return token.charAt(0) + "***" + token.length();
        }
        return "***";
    }

    @Override
    public String getUserName(String user) {
        MasterSlaveContextHolder.PolicyHandle h = MasterSlaveContextHolder.push(MasterSlavePolicy.R_MS);
        try {
            return userManager.getUserNameByEmail(Emails.punycode(user)).getOrElse(user);
        } catch (Exception e) {
            logger.error(e);
            return user;
        } finally {
            h.popSafely();
        }
    }

    @Override
    public ListF<MultiStatusResponse2> findUsers(final ReportRequestPrincipalPropertySearch principalPropertySearch,
            PassportUid client, final UserAgentType userAgent)
    {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            DirectorySearchPredicate predicate = predicate(principalPropertySearch);
            ListF<DirectoryEntry> contacts =
                    directoryManager.findContacts(client, predicate, false, 50); // XXX: proper limit

            ListF<DavPropertyName> propertyNames = principalPropertySearch.getProp().get().getPropertyNames();

            return contacts.map(directoryEntry -> makeResponse(directoryEntry, propertyNames, userAgent));
        });
    }

    @Override
    public boolean isUserResourcesAccessibleToClient(String user, String client) {
        return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_MS, () -> {
            ListF<Email> emails = Cf.list(user, client).filterMap(Email.parseSafeF());
            ListF<PassportUid> uids = Cf2.flatBy2(userManager.getUidsByEmails(emails)).get2();

            if (passportAuthDomainsHolder.containsYandexTeamRu()) {
                return uids.isNotEmpty()
                        && (uids.first().sameAs(uids.last()) || !userManager.isExternalYtUser(uids.last()));
            } else {
                return uids.unique().size() == 1;
            }
        });
    }

    private DirectorySearchPredicate predicate(ReportRequestPrincipalPropertySearch principalPropertySearch) {
        val predicates = principalPropertySearch.getPropertySearches().filterMap(this::propertyPredicate);
        if (principalPropertySearch.getOperator() == AddressbookSearchOperator.ALLOF) {
            return DirectorySearchPredicate.allPredicate(predicates);
        } else if (principalPropertySearch.getOperator() == AddressbookSearchOperator.ANYOF) {
            return DirectorySearchPredicate.anyPredicate(predicates);
        } else {
            throw new IllegalArgumentException("unknown operator: " + principalPropertySearch.getOperator());
        }
    }

    private Option<DirectorySearchPredicate> propertyPredicate(PropertySearch propertySearch) {
        DavPropertyName propertyName = (DavPropertyName) propertySearch.getProp().getProperties().single();
        Option<DirectorySearchField> field = field(propertyName);
        if (!field.isPresent()) {
            logger.warn("Unknown property for search: " + propertySearch.getProp());
            return Option.empty();
        }
        DirectorySearchFieldOperator operator = operator(propertySearch.getMatch().getMatchType());
        String text = propertySearch.getMatch().getText();
        return Option.of(DirectorySearchPredicate.fieldPredicate(field.get(), operator, text));
    }

    private Option<DirectorySearchField> field(DavPropertyName propertyName) {
        if (propertyName.equals(CaldavConstants.CALENDARSERVER_EMAIL_ADDRESS_SET_PROP)) {
            return Option.of(DirectorySearchField.EMAIL);
        } else if (propertyName.equals(WebdavConstants.DAV_DISPLAYNAME_PROP)) {
            return Option.of(DirectorySearchField.DISPLAY_NAME);
        } else if (propertyName.equals(CaldavConstants.CALENDARSERVER_FIRST_NAME_PROP)) {
            return Option.of(DirectorySearchField.FIRST_NAME);
        } else if (propertyName.equals(CaldavConstants.CALENDARSERVER_LAST_NAME_PROP)) {
            return Option.of(DirectorySearchField.LAST_NAME);
        } else if (propertyName.equals(CaldavConstants.CALDAV_CALENDAR_USER_TYPE_PROP)) {
            return Option.of(DirectorySearchField.CALENDAR_USER_TYPE);
        } else if (propertyName.equals(CaldavConstants.CALDAV_CALENDAR_USER_ADDRESS_SET_PROP)) {
            return Option.of(DirectorySearchField.CALENDAR_USER_ADDRESS_SET);
        } else {
            return Option.empty();
        }
    }

    private DirectorySearchFieldOperator operator(Option<String> matchTypeO) {
        String matchType = matchTypeO.getOrElse("contains");
        return DirectorySearchFieldOperator.R.valueOfO(matchType)
            .getOrElse(DirectorySearchFieldOperator.CONTAINS);
    }

    private MultiStatusResponse2 makeResponse(DirectoryEntry directoryEntry, ListF<DavPropertyName> propNames, UserAgentType userAgent) {
        PropertiesCollection propertiesCollection = propertiesCollection(directoryEntry, userAgent);
        ListF<PropStat> propStats = JackrabbitUtils.getPropertiesByNames(propertiesCollection, propNames);
        return MultiStatusResponse2.propStatResponse(CalendarUrls.addressbooksUser(directoryEntry.getEmail().getEmail()), propStats);
    }

    private PropertiesCollection propertiesCollection(DirectoryEntry directoryEntry, UserAgentType userAgent) {
        DavPropertySet ps = new DavPropertySet();
        ps.add(new EmailProperty(CaldavConstants.CALENDARSERVER_EMAIL_ADDRESS_SET_PROP, directoryEntry.getEmail()));
        ps.add(new DefaultDavProperty<>(WebdavConstants.DAV_DISPLAYNAME_PROP, directoryEntry.getDisplayName()));
        ps.add(new DefaultDavProperty<>(CaldavConstants.CALENDARSERVER_FIRST_NAME_PROP, directoryEntry.getFirstName()));
        ps.add(new DefaultDavProperty<>(CaldavConstants.CALENDARSERVER_LAST_NAME_PROP, directoryEntry.getLastName()));

        IcsCuType icsCuType = CcDavUtils.directoryEntryTypeToIcsCuType(directoryEntry, userAgent);
        ps.add(new DefaultDavProperty<>(CaldavConstants.CALDAV_CALENDAR_USER_TYPE_PROP, icsCuType.getValue()));

        ps.add(new HrefProperty(WebdavConstants.DAV_PRINCIPAL_URL_PROP, CalendarUrls.principals(directoryEntry.getEmail().getEmail()).getEncoded(), false));
        ps.add(new HrefProperty(CaldavConstants.CALDAV_CALENDAR_USER_ADDRESS_SET_PROP, Emails.getUnicodedMailto(directoryEntry.getEmail()), false));

        ps.add(new DefaultDavProperty<>(
            CaldavConstants.CALENDARSERVER_RECORD_TYPE_PROP, directoryEntry.getType().getRecordType().getValue()));

        return new PropertiesCollectionImpl(ps);
    }
}
