package ru.yandex.calendar.frontend.caldav.proto.jackrabbit;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.MultiStatus;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertyNameSet;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.xml.DomUtil;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavWithHttpStatusCodeException;
import ru.yandex.calendar.frontend.caldav.proto.tree.PropertiesCollection;
import ru.yandex.calendar.frontend.caldav.proto.tree.ResourceNotFoundException;
import ru.yandex.calendar.frontend.caldav.proto.tree.WebdavResource;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropFindType;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatus2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusResponse2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.Prop;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.PropStat;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.Status;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.PermissionDeniedUserException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.http.HttpStatus;

/**
 * @author Stepan Koltsov
 */
public class JackrabbitUtils {

    public static ListF<DavPropertyName> getPropNames(DavPropertyNameSet propertyNameSet) {
        return Cf.x(propertyNameSet.iterator()).toList();
    }

    /**
     * XXX: rewrite using
     * @see MultiStatus2
     */
    public static MultiStatus makeMultiStatus(ListF<DavResource> eventResources, PropertiesRequest propertiesRequest) {
        MultiStatus ms = new MultiStatus();

        for (DavResource eventResource : eventResources) {
            DavPropertyNameSet davPropertyNameSet = propertiesRequest.getDavPropertyNameSetO().getOrElse(new DavPropertyNameSet());
            ms.addResourceProperties(eventResource, davPropertyNameSet, propertiesRequest.getType().getJackrabbitValue(), 0);
        }

        return ms;
    }

    public static DocumentBuilder documentBuilder() {
        try {
            return DomUtil.BUILDER_FACTORY.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public static DavException supportedReportError() {
        return error(HttpStatus.SC_403_FORBIDDEN,
                documentBuilder().newDocument().createElementNS("DAV:", "supported-report"));
    }

    public static DavException error(int statusCode, Element element) {
        return new DavException(statusCode, null, null, element);
    }

    public static int translateToStatusCode(Throwable e) {
        if (e instanceof PermissionDeniedUserException) {
            return HttpStatus.SC_403_FORBIDDEN;
        } else if (e instanceof CaldavWithHttpStatusCodeException) {
            return ((CaldavWithHttpStatusCodeException) e).getStatusCode();
        } else if (e instanceof CommandRunException) {
            CommandRunException c = (CommandRunException) e;
            if (c.isSituation(Situation.BUSY_OVERLAP)
                || c.isSituation(Situation.EXCHANGE_EVENT_CONFLICT)
                || c.isSituation(Situation.TOO_LONG_EVENT)
                || c.isSituation(Situation.MASSAGE_DENIED))
            {
                return HttpStatus.SC_409_CONFLICT;
            }
            if (c.isSituation(Situation.EVENT_MODIFIED)) {
                return HttpStatus.SC_412_PRECONDITION_FAILED;
            }
            if (c.isSituation(Situation.BAD_KARMA)) {
                return HttpStatus.SC_403_FORBIDDEN;
            }
            return HttpStatus.SC_500_INTERNAL_SERVER_ERROR;
        } else if (e instanceof ResourceNotFoundException) {
            return HttpStatus.SC_404_NOT_FOUND;
        } else if (e instanceof DavException) {
            return ((DavException) e).getErrorCode();
        } else {
            return HttpStatus.SC_500_INTERNAL_SERVER_ERROR;
        }
    }

    public static ListF<PropStat> getProperties(PropertiesCollection resource, PropertiesRequest request) {
        if (request.getType() == PropFindType.ALL_PROP || request.getType() == PropFindType.ALL_PROP_INCLUDE) {
            DavPropertySet propertySet = resource.getProperties();
            return Cf.list(new PropStat(Status.STATUS_200_OK, new Prop(propertySet.getContent())));
        } else if (request.getType() == PropFindType.BY_PROPERTY) {
            ListF<DavPropertyName> propNames = getPropNames(request.getDavPropertyNameSetO().get());
            return getPropertiesByNames(resource, propNames);
        } else if (request.getType() == PropFindType.PROPERTY_NAMES) {
            return Cf.list(new PropStat(Status.STATUS_200_OK, new Prop(resource.getPropertyNamesList())));
        } else {
            throw new IllegalStateException("unknown enum: " + request.getType());
        }
    }

    public static MultiStatusResponse2 getPropertiesResponse(WebdavResource webdavResource, PropertiesRequest request) {
        return MultiStatusResponse2.propStatResponse(DavHref.fromEncoded(webdavResource.getHref()), getProperties(webdavResource, request));
    }

    public static ListF<PropStat> getPropertiesByNames(PropertiesCollection resource, ListF<DavPropertyName> propNames) {
        Function<DavProperty<?>, DavPropertyName> getNameF = new Function<DavProperty<?>, DavPropertyName>() {
            public DavPropertyName apply(DavProperty<?> property) {
                return property.getName();
            }
        };

        ListF<DavProperty<?>> properties = Cf.x(resource.getProperties().iterator()).toList();

        ListF<DavPropertyName> status404Properties = propNames.filter(properties.map(getNameF).containsF().notF());
        ListF<DavProperty<?>> status200Properties = properties.filter(getNameF.andThen(propNames.containsF()));

        ListF<PropStat> r = Cf.arrayList();
        if (status200Properties.isNotEmpty()) {
            r.add(new PropStat(Status.STATUS_200_OK, new Prop(status200Properties)));
        }
        if (status404Properties.isNotEmpty()) {
            r.add(new PropStat(Status.STATUS_404_NOT_FOUND, new Prop(status404Properties)));
        }
        return r;
    }

} //~
