package ru.yandex.calendar.frontend.caldav.proto.tree;

import java.io.IOException;

import io.micrometer.core.instrument.MeterRegistry;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.MultiStatus;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.HrefProperty;
import org.apache.jackrabbit.webdav.property.ResourceType;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.caldav.proto.CaldavLocatorFactory;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavRequest;
import ru.yandex.calendar.frontend.caldav.proto.carddav.CarddavConstants;
import ru.yandex.calendar.frontend.caldav.proto.ccdav.CcDavUtils;
import ru.yandex.calendar.frontend.caldav.proto.facade.CaldavCalendarFacade;
import ru.yandex.calendar.frontend.caldav.proto.facade.CalendarDescription;
import ru.yandex.calendar.frontend.caldav.proto.facade.CarddavCalendarFacade;
import ru.yandex.calendar.frontend.caldav.proto.facade.CcCalendarFacade;
import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequestExpandProperty;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequestExpandPropertyProperty;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequestParser;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.SupportedReportSetProperty;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusUtils;
import ru.yandex.calendar.frontend.caldav.userAgent.UserAgentType;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;

import static java.util.Collections.emptyList;

public class CaldavContext {
    private static final int DEFAULT_TRUNCATION_LIMIT = 100;
    @Autowired
    private CaldavCalendarFacade caldavCalendarFacade;
    @Autowired
    private CarddavCalendarFacade carddavCalendarFacade;
    @Autowired
    private CcCalendarFacade ccCalendarFacade;
    @Autowired
    private CaldavResourceFactory caldavResourceFactory;
    @Autowired
    private CaldavLocatorFactory caldavLocatorFactory;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private MeterRegistry registry;

    private final DynamicProperty<CaldavTruncationConfig> caldavTruncationConfig =
            new DynamicProperty<>("caldav.truncation.config", new CaldavTruncationConfig(DEFAULT_TRUNCATION_LIMIT, TruncationMode.REQUEST_ONLY, emptyList()));

    public CaldavCalendarFacade getCaldavCalendarFacade() {
        return caldavCalendarFacade;
    }

    public CarddavCalendarFacade getCarddavCalendarFacade() {
        return carddavCalendarFacade;
    }

    public CcCalendarFacade getCcCalendarFacade() {
        return ccCalendarFacade;
    }

    public CaldavResourceFactory getCaldavResourceFactory() {
        return caldavResourceFactory;
    }

    public CaldavLocatorFactory getCaldavLocatorFactory() {
        return caldavLocatorFactory;
    }

    public DavProperty<?> getDavPrincipalUrlProperty(String user) {
        return new HrefProperty(WebdavConstants.DAV_PRINCIPAL_URL_PROP, CalendarUrls.principals(user).getEncoded(), true);
    }

    public DavPropertySet getCaldavUrlProperties(String user) {
        DavPropertySet ps = new DavPropertySet();
        ps.add(new HrefProperty(CaldavConstants.CALDAV_HOME_SET_PROP, CalendarUrls.calendarsRoot(user).getEncoded(), false));
        ps.add(new HrefProperty(CaldavConstants.CALDAV_USER_ADDRESS_SET_PROP, new String[] { Emails.getUnicodedMailto(caldavCalendarFacade.getUserEmail(user)), CalendarUrls.principals(user).getEncoded() }, false));
        ps.add(new HrefProperty(CaldavConstants.CALDAV_SCHEDULE_INBOX_URL_PROP, CalendarUrls.inbox(user).getEncoded(), false));
        ps.add(new HrefProperty(CaldavConstants.CALDAV_SCHEDULE_OUTBOX_URL_PROP, CalendarUrls.outbox(user).getEncoded(), false));
        ps.add(new HrefProperty(CaldavConstants.CALENDARSERVER_EMAIL_ADDRESS_SET_PROP, new String[] { Emails.getUnicodedMailto(caldavCalendarFacade.getUserEmail(user)) }, false));
        return ps;
    }

    public DavPropertySet getCarddavUrlProperties(String user) {
        DavPropertySet ps = new DavPropertySet();
        ps.add(new HrefProperty(CarddavConstants.CARDDAV_ADDRESSBOOK_HOME_SET_PROP, CalendarUrls.addressbooksUser(user).getEncoded(), true));
        return ps;
    }

    public DavPropertySet getGlobalProperties() {
        DavPropertySet ps = new DavPropertySet();
        ps.add(new HrefProperty(WebdavConstants.DAV_CURRENT_USER_PRINCIPAL_PROP, CalendarUrls.principals(ClientHolder.get()).getEncoded(), false));
        if (carddavCalendarFacade.getUserDirectory(ClientHolder.getPassportUid()).isPresent()) {
            ps.add(new HrefProperty(CarddavConstants.CARDDAV_DIRECTORY_GATEWAY_PROP, CalendarUrls.directory().getEncoded(), false));
        }
        return ps;
    }

    public DavPropertySet getPrincipalUrlProperties(String user) {
        DavPropertySet ps = new DavPropertySet();
        ps.add(new HrefProperty(WebdavConstants.DAV_OWNER_PROP, CalendarUrls.principals(ClientHolder.get()).getEncoded(), false));
        ps.add(new ResourceType(new int[] { WebdavConstants.DAV_COLLECTION_RT, WebdavConstants.DAV_PRINCIPAL_RT }));
        ps.addAll(getGlobalProperties());
        ps.add(new DefaultDavProperty<>(WebdavConstants.DAV_DISPLAYNAME_PROP, ccCalendarFacade.getUserName(user)));
        ps.add(getDavPrincipalUrlProperty(user));
        ps.add(new SupportedReportSetProperty(CcDavUtils.getSupportedReports()));
        ps.add(new HrefProperty(WebdavConstants.DAV_PRINCIPAL_COLLECTION_SET_PROP, CalendarUrls.principals().getEncoded(), false));
        ps.addAll(getCaldavUrlProperties(user));
        ps.addAll(getCarddavUrlProperties(user));
        return ps;
    }

    private void expandProperty(String user, WebdavRequest request, WebdavResponse response, ReportRequestExpandProperty reportRequestExpandProperty) throws DavException, IOException {
        CaldavRequestContext requestContext = new CaldavRequestContext(this, request, response, registry);

        final PassportUid passportUid = caldavCalendarFacade.checkAndGetPassportUid(user);
        ListF<CalendarDescription> calendars = caldavCalendarFacade.getUserVisibleExternalCalendars(passportUid);

        if (requestContext.getUserAgentType() == UserAgentType.CALDAVSYNCADAPTER_ANDROID) {
            calendars = calendars.filter(CalendarDescription.supportsVeventComponentF()); // CAL-6600
        }

        Tuple2<ListF<CalendarDescription>, ListF<CalendarDescription>> wr = calendars.partition(CalendarDescription.writableF());
        ListF<CalendarDescription> writableCalendars = wr._1;
        ListF<CalendarDescription> readonlyCalendars = wr._2;

        SetF<String> writableCalendarsOwners = writableCalendars.filterMap(CalendarDescription::getOwner).unique();
        // if another user let you write to at least one his calendar, show him as read&write delegator
        SetF<String> readonlyCalendarsOwners = readonlyCalendars.filterMap(CalendarDescription::getOwner).unique()
                .minus(writableCalendarsOwners);

        MultiStatusResponse msr = new MultiStatusResponse(request.getRequestURI(), "??");

        for (ReportRequestExpandPropertyProperty prop : reportRequestExpandProperty.getProperties()) {
            if (passportAuthDomainsHolder.containsYandexTeamRu() &&
                    (prop.is(CaldavConstants.CALENDARSERVER_CALENDAR_PROXY_READ_FOR_PROP) ||
                    prop.is(CaldavConstants.CALENDARSERVER_CALENDAR_PROXY_WRITE_FOR_PROP)))
            {
                boolean write = prop.is(CaldavConstants.CALENDARSERVER_CALENDAR_PROXY_WRITE_FOR_PROP);

                ListF<MultiStatusResponse> calendarResponses = Cf.arrayList();
                SetF<String> externalCalendarsOwners = write ? writableCalendarsOwners : readonlyCalendarsOwners;

                for (String externalCalendarsOwner : externalCalendarsOwners.sorted()) {
                    CcResourcePrincipalsUser ccResourcePrincipalsUser = new CcResourcePrincipalsUser(requestContext, externalCalendarsOwner);
                    MultiStatusResponse cmsr = new MultiStatusResponse(ccResourcePrincipalsUser, prop.toDavPropertyNameSetFromChildren());
                    calendarResponses.add(cmsr);
                }

                msr.add(new DefaultDavProperty<>(prop.toDavPropertyName(), calendarResponses));
            } else {
                msr.add(prop.toDavPropertyName(), DavServletResponse.SC_NOT_FOUND);
            }
        }

        MultiStatus ms = new MultiStatus();
        ms.addResponse(msr);
        MultiStatusUtils.sendMultiStatus(ms, request, response, registry);
    }

    public void report(String user, WebdavRequest request0, WebdavResponse response) throws DavException, IOException {
        CaldavRequest request = new CaldavRequest(request0);

        ReportRequest reportRequest = ReportRequestParser.parse(request.getWebdavRequest().getRequestDocument().getDocumentElement());

        if (reportRequest instanceof ReportRequestExpandProperty) {
            ReportRequestExpandProperty reportRequestExpandProperty = (ReportRequestExpandProperty) reportRequest;
            expandProperty(user, request.getWebdavRequest(), response, reportRequestExpandProperty);
        } else {
            response.sendError(JackrabbitUtils.supportedReportError());
        }
    }

    public CaldavTruncationConfig getCaldavTruncationConfig() {
        return caldavTruncationConfig.get();
    }
}
