package ru.yandex.calendar.frontend.caldav.proto.tree;

import io.micrometer.core.instrument.MeterRegistry;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceFactory;
import org.apache.jackrabbit.webdav.DavResourceLocator;
import org.apache.jackrabbit.webdav.DavServletRequest;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.DavSession;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.calendar.frontend.caldav.proto.facade.CaldavCalendarFacade;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.UrlUtils;

public class CaldavResourceFactory implements DavResourceFactory {
    @Autowired
    private CaldavContext caldavContext;
    @Autowired
    private CaldavCalendarFacade caldavCalendarFacade;
    @Autowired
    private MeterRegistry registry;

    @Override
    public DavResource createResource(DavResourceLocator locator, DavSession session) throws DavException {
        throw new UnsupportedOperationException();
    }

    @Override
    public DavResource createResource(DavResourceLocator locator, DavServletRequest request, DavServletResponse response) throws DavException {
        return createResource(DavHref.fromEncoded(locator.getHref(false)), request, response);
    }

    public DavResource createResource(final DavHref href, DavServletRequest request, DavServletResponse response) throws DavException {
        final CaldavRequestContext caldavRequestContext = new CaldavRequestContext(this.caldavContext, request, response, registry);

        try {
            return CalendarUrls.parse(href, new CalendarUrls.Handler<>() {

                @Override
                public DavResource root() {
                    return new CcResourceRoot(caldavRequestContext);
                }

                @Override
                public DavResource calendarsUserEventsEvent(String user, String collectionId, String id) {
                    return new CaldavResourceUserEventsEntry(caldavRequestContext, user, collectionId, id, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserEvents(String user, String collectionId) {
                    return new CaldavResourceUserEvents(caldavRequestContext, user, collectionId, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserEventsUser(String user, String collectionId) {
                    return new CaldavResourceUserEventsUser(caldavRequestContext, user, collectionId, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserInbox(String user) {
                    return new CaldavResourceUserInbox(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserOutbox(String user) {
                    return new CaldavResourceUserOutbox(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserTodos(String user, String collectionId) {
                    return new CaldavResourceUserTodos(caldavRequestContext, user, collectionId, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserTodosTodo(String user, String collectionId, String id) {
                    return new CaldavResourceUserTodosEntry(caldavRequestContext, user, collectionId, id, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUser(String user) {
                    return new CaldavResourceUserRoot(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserUser(String user) {
                    return new CaldavResourceUserUser(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource principals() {
                    return new CcResourcePrincipals(caldavRequestContext);
                }

                @Override
                public DavResource principalsUser(String user) {
                    if (!isValidUser(user)) {
                        user = UrlUtils.urlDecode(user); // Lightning encodes twice
                        checkAndGetPassportUid(user);
                    }
                    return new CcResourcePrincipalsUser(caldavRequestContext, user);
                }

                @Override
                public DavResource directory() {
                    if (caldavRequestContext.getRequest().getDepth() > 0) {
                        // Hack for iOS 4.3.X that expects the nested directories inside directory gateway
                        return new CarddavResourceDirectoryGhost(caldavRequestContext, false);
                    }
                    return new CarddavResourceDirectory(caldavRequestContext);
                }

                @Override
                public DavResource directoryContact(String id) {
                    if (id.equals("x")) {
                        // Hack for iOS 4.3.X that expects the nested directories inside directory gateway
                        return new CarddavResourceDirectoryGhost(caldavRequestContext, true);
                    }
                    return new CarddavResourceDirectoryContact(caldavRequestContext, id);
                }

                @Override
                public DavResource addressbooksUser(String user) {
                    return new CarddavResourceUserRoot(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource addressbooksUserAddressbook(String user) {
                    return new CarddavResourceUserAddressbook(caldavRequestContext, user, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource addressbooksUserAddressbookCard(String user, String id) {
                    return new CarddavResourceUserAddressbookCard(caldavRequestContext, user, id, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserUnknown(String user, String dir) {
                    return new CaldavResourceUserUnknown(caldavRequestContext, user, dir, checkAndGetPassportUid(user));
                }

                @Override
                public DavResource calendarsUserUnknownEntry(String user, String collectionId, String id) {
                    return new CaldavResourceUserUnknownEntry(caldavRequestContext, user, collectionId, id, checkAndGetPassportUid(user));
                }

                private PassportUid checkAndGetPassportUid(String user) {
                    try {
                        return caldavCalendarFacade.checkAndGetPassportUid(user);
                    } catch (Exception e) {
                        throw new ResourceNotFoundException("User not found: " + user);
                    }
                }

                private boolean isValidUser(String user) {
                    try {
                        caldavCalendarFacade.checkAndGetPassportUid(user);
                        return true;
                    } catch (Exception e) {
                        return false;
                    }
                }
            });
        } catch (ResourceNotFoundException e) {
            throw new DavException(404, e);
        }
    }
}
