package ru.yandex.calendar.frontend.caldav.proto.tree;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.NotImplementedException;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceIterator;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.io.InputContext;
import org.apache.jackrabbit.webdav.io.OutputContext;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.PropEntry;
import org.apache.jackrabbit.webdav.property.ResourceType;
import org.apache.jackrabbit.webdav.security.CurrentUserPrivilegeSetProperty;
import org.apache.jackrabbit.webdav.security.Privilege;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.frontend.caldav.impl.LayerSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavSupportedCalendarComponentSet;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentsFilter;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestCalendar;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestCalendarMultiget;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestCalendarQuery;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestSyncCollection;
import ru.yandex.calendar.frontend.caldav.proto.ccdav.CcDavUtils;
import ru.yandex.calendar.frontend.caldav.proto.facade.CalendarComponent;
import ru.yandex.calendar.frontend.caldav.proto.facade.CalendarDescription;
import ru.yandex.calendar.frontend.caldav.proto.facade.CalendarProperties;
import ru.yandex.calendar.frontend.caldav.proto.facade.ComponentGetResult;
import ru.yandex.calendar.frontend.caldav.proto.facade.ComponentModified;
import ru.yandex.calendar.frontend.caldav.proto.facade.IcalColor;
import ru.yandex.calendar.frontend.caldav.proto.facade.IcalColorUtils;
import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncInfo;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavSyncToken;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.ReportRequestParser;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.SupportedReportSetProperty;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatus2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusResponse2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusUtils;
import ru.yandex.calendar.frontend.caldav.userAgent.UserAgentType;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.xml.dom.DomUtils;

import static java.util.Collections.emptyList;
import static java.util.function.Predicate.not;

@Slf4j
public abstract class CaldavResourceUserCalendarBase extends CaldavResourceUserChildBase {
    private final CollectionId collectionId;
    private static final ConcurrentHashMap<String, Pattern> AGENTS_PATTERNS = new ConcurrentHashMap<>();

    public CaldavResourceUserCalendarBase(CaldavRequestContext caldavContext, CollectionId collectionId) {
        super(caldavContext, collectionId.getUser(), collectionId.getPassportUid());
        this.collectionId = collectionId;
    }

    public CollectionId getCollectionId() {
        return collectionId;
    }

    @Override
    public boolean exists() {
        boolean exists = caldavCalendarFacade.existsCalendarWithId(getCollectionId());

        if (caldavRequestContext.getUserAgentType() == UserAgentType.CALDAVSYNCADAPTER_ANDROID) {
            return exists && getDescription().supportsVeventComponent(); // CAL-6600
        }
        return exists;
    }

    protected ListF<CalendarComponent> getComponents(CalendarComponentsFilter filter, boolean includeIcs) {
        return caldavCalendarFacade.getUserCalendarEvents(
                ClientHolder.getPassportUid(), getCollectionId(), filter.getVeventConditions(),
                filter.getVtodoConditions(), includeIcs, caldavRequestContext.getUserAgentType());
    }

    protected ListF<ComponentGetResult> getComponentsByIds(ListF<String> ids, boolean includeIcs) {
        return caldavCalendarFacade.getUserCalendarEvents(
                ClientHolder.getPassportUid(), getCollectionId(), ids, includeIcs, caldavRequestContext.getUserAgentType());
    }

    protected List<ComponentModified> getModifiedComponents(DavSyncToken syncToken, boolean includeIcs) {
        return caldavCalendarFacade.getUserCalendarModifiedEvents(
                ClientHolder.getPassportUid(), getCollectionId(), syncToken, includeIcs, caldavRequestContext.getUserAgentType());
    }

    protected DavSyncToken getSyncToken() {
        return caldavCalendarFacade.getCalendarSyncToken(getCollectionId());
    }

    private static Comparator<ComponentModified> sortByEtagAndFilename() {
        return Comparator.comparing(ComponentModified::getDate)
                .thenComparing(ComponentModified::getFileName);
    }

    private static List<ComponentModified> getTrunkedSortedList(List<ComponentModified> components, Optional<Integer> limit) {
        val lim = limit.orElseGet(components::size);
        return StreamEx.of(components)
                .sorted(sortByEtagAndFilename())
                .limit(lim)
                .toImmutableList();
    }

    private static Comparator<DavSyncInfo> sortTokens() {
        return Comparator.comparing(DavSyncInfo::getMillis)
                .thenComparing(x -> x.getExternalId().orElse(""));
    }

    private static DavSyncToken recalculateSyncToken(
            DavSyncToken collectionToken,
            List<ComponentModified> components,
            DavSyncToken queryToken,
            boolean overflow
    ) {
        final StreamEx<DavSyncToken> stream;

        if (!overflow) {
            stream = StreamEx.of(collectionToken);
        } else {
            val component = components.get(components.size()-1);
            stream = StreamEx.of(LayerSyncToken.getSyncToken(component.getDate(), component.getFileName()), queryToken);
        }

        return stream
                .filter(not(DavSyncToken::isEmpty))
                .map(LayerSyncToken::parseSyncToken)
                .max(sortTokens())
                .map(LayerSyncToken::getSyncToken)
                .get();
    }

    private final Function0<CalendarDescription> description = ((Function0<CalendarDescription>) () -> caldavCalendarFacade.getUserCalendar(ClientHolder.getPassportUid(), getCollectionId())).memoize();

    protected CalendarDescription getDescription() {
        return description.apply();
    }

    @Override
    public boolean isCollection() {
        return true;
    }

    @Override
    public MultiStatusResponse alterProperties(List<? extends PropEntry> changeList) {
        MultiStatusResponse response = new MultiStatusResponse(getHref(), "PROPPATCH events");

        Option<IcalColor> color = Option.empty();
        Option<String> displayName = Option.empty();
        Option<Boolean> affectsAvailability = Option.empty();

        for (PropEntry prop0 : changeList) {
            if (prop0 instanceof DavPropertyName) {
                // delete
                DavPropertyName propName = (DavPropertyName) prop0;
                log.warn("Asked to delete property {}", propName);
                response.add(propName, HttpStatus.SC_404_NOT_FOUND);
            } else if (prop0 instanceof DavProperty) {
                // add/set
                DavProperty<?> prop = (DavProperty<?>) prop0;
                DavPropertyName propName = prop.getName();
                if (propName.equals(CaldavConstants.CALENDAR_COLOR_PROP)) {
                    log.info("Updating color to: {}", prop.getValue());
                    color = Option.of(IcalColorUtils.parseDavProperty(prop.getValue().toString()));
                    response.add(propName, HttpStatus.SC_200_OK);
                } else if (propName.equals(CaldavConstants.CALENDAR_ORDER_PROP)) {
                    log.info("Updating order to: {}", prop.getValue());
                    response.add(propName, HttpStatus.SC_403_FORBIDDEN);
                } else if (propName.equals(CaldavConstants.CALDAV_TIMEZONE_PROP)) {
                    log.info("Not updating timezone");
                    response.add(propName, HttpStatus.SC_403_FORBIDDEN);
                } else if (propName.equals(WebdavConstants.DAV_DISPLAYNAME_PROP)) {
                    log.info("Updating displayname to: {}", prop.getValue());
                    displayName = Option.of(prop.getValue().toString());
                    response.add(propName, HttpStatus.SC_200_OK);
                } else if (propName.equals(CaldavConstants.CALDAV_SCHEDULE_CALENDAR_TRANSP)) {
                    boolean isOpaque = !prop.getValue().toString().contains("transparent");
                    log.info("Updating affects availability to: {}", affectsAvailability);
                    affectsAvailability = Option.of(isOpaque);
                    response.add(propName, HttpStatus.SC_200_OK);
                } else {
                    log.warn("Unsupported property: {}", prop.getName());
                    response.add(propName, HttpStatus.SC_404_NOT_FOUND);
                }
            } else {
                throw new IllegalArgumentException("unsupported element in list: " + prop0);
            }
        }
        CalendarProperties properties = new CalendarProperties(displayName, color, affectsAvailability);
        caldavCalendarFacade.changeCalendar(ClientHolder.getPassportUid(), getCollectionId(), properties);
        return response;
    }

    @Override
    public void addMember(DavResource resource0, InputContext inputContext) {
        CaldavResourceUserCalendarEntryBase entry = (CaldavResourceUserCalendarEntryBase) resource0;
        caldavCalendarFacade.putUserCalendarEvent(
                ClientHolder.getPassportUid(),
                getCollectionId(),
                entry.getName(),
                CcDavUtils.getIfMatchETag(inputContext),
                new InputStreamX(inputContext.getInputStream()).readBytes());
    }

    @Override
    public void removeMember(DavResource member) {
        CaldavResourceUserCalendarEntryBase entry = (CaldavResourceUserCalendarEntryBase) member;
        Check.equals(getCollectionId(), entry.getCollectionId());
        caldavCalendarFacade.removeUserCalendarEvent(ClientHolder.getPassportUid(), getCollectionId(), entry.getName());
    }

    /**
     * @see #propFindChildren(PropertiesRequest, int)
     */
    @Override
    public DavResourceIterator getMembers() {
        throw new NotImplementedException("not needed");
    }

    @Override
    public void spool(OutputContext outputContext) throws IOException {
        if (outputContext.hasStream()) {
            outputContext.setContentType("text/plain");
            PrintWriter pw = new PrintWriter(outputContext.getOutputStream());

            ListF<CalendarComponent> components = getComponents(CalendarComponentsFilter.any(), false);
            Function<CalendarComponent, String> getHrefF =
                    CalendarComponent.getFileNameF().andThen(getComponentHrefF()).andThen(DavHref.getEncodedF());

            pw.println(components.map(getHrefF).mkString("", "\n", "\n$"));
            pw.flush();
        }
    }

    @Override
    public DavPropertySet getProperties() {
        DavPropertySet ps = super.getProperties();
        ps.add(new ResourceType(new int[] { WebdavConstants.DAV_COLLECTION_RT, CaldavConstants.CALDAV_CALENDAR_RT }));

        CalendarDescription desc = getDescription();
        Option<DavSyncToken> syncTokenO = desc.getSyncToken();

        if (syncTokenO.isPresent()) {
            ps.add(new DefaultDavProperty<>(WebdavConstants.DAV_SYNC_TOKEN_PROP, syncTokenO.get().getValue()));
            ps.add(new SupportedReportSetProperty(CcDavUtils.getSupportedReports()));
        } else {
            ps.add(new SupportedReportSetProperty(CcDavUtils.getSupportedReportsWithoutSyncReport()));
        }
        ps.add(new DefaultDavProperty<>(WebdavConstants.DAV_DISPLAYNAME_PROP, desc.getName()));
        ps.add(new DefaultDavProperty<>(CaldavConstants.CALENDARSERVER_GETCTAG_PROP, desc.getCtag().getValue()));

        Privilege[] privileges = desc.isWritable() ?
                new Privilege[] { Privilege.PRIVILEGE_READ, Privilege.PRIVILEGE_WRITE } :
                new Privilege[] { Privilege.PRIVILEGE_READ, Privilege.PRIVILEGE_WRITE_PROPERTIES };
        ps.add(new CurrentUserPrivilegeSetProperty(privileges));

        ps.add(new CaldavSupportedCalendarComponentSet(desc.getSupportedComponentNames().toArray(String.class)));

        ps.add(new DefaultDavProperty<>(
                CaldavConstants.CALENDAR_COLOR_PROP,
                IcalColorUtils.toDavProperty(desc.getColor())));

        return ps;
    }

    /**
     * @see CarddavResourceUserAddressbook#report(WebdavRequest, WebdavResponse)
     * @see CaldavResourceUserInbox#report(WebdavRequest, WebdavResponse)
     */
    @Override
    public void report(WebdavRequest request, WebdavResponse response) throws DavException, IOException {
        Document doc = request.getRequestDocument();
        Element root = doc.getDocumentElement();

        ReportRequest caldavRequest;
        try {
            caldavRequest = ReportRequestParser.parse(root);
        } catch (Exception e) {
            log.error("Error on parsing request: {}", DomUtils.writeToString(doc));
            throw e;
        }
        log.debug("Report req: {}", caldavRequest);

        if (caldavRequest instanceof ReportRequestCalendarQuery) {
            reportQuery(request, (ReportRequestCalendarQuery) caldavRequest, response);
        } else if (caldavRequest instanceof ReportRequestCalendarMultiget) {
            reportMultiget(request, (ReportRequestCalendarMultiget) caldavRequest, response);
        } else if (caldavRequest instanceof ReportRequestSyncCollection) {
            reportSyncCollection(request, (ReportRequestSyncCollection) caldavRequest, response);
        } else {
            response.sendError(JackrabbitUtils.supportedReportError());
        }
    }

    /**
     * @see #reportMultiget(WebdavRequest, ReportRequestCalendarMultiget, WebdavResponse)
     */
    @Override
    public ListF<MultiStatusResponse2> propFindChildren(final PropertiesRequest propertiesRequest, int depth) {
        boolean includeIcs = propertiesRequest.contains(CaldavConstants.CALDAV_CALENDAR_DATA_PROP);
        return Cf.toList(prepareResponses(propertiesRequest, getComponents(CalendarComponentsFilter.any(), includeIcs)));
    }

    protected void reportQuery(WebdavRequest request, ReportRequestCalendarQuery query, WebdavResponse response) {
        ListF<CalendarComponent> components = getComponents(query.getFilter(), getIsIncludeIcs(query));

        if (components.size() < 4) {
            log.debug("Returning components: {}", components);
        } else {
            log.debug("Returning {} components", components.size());
        }
        val responses = prepareResponses(query.getPropertiesRequest(), components);
        MultiStatusUtils.sendMultiStatus(new MultiStatus2(responses), request, response, getMeterRegistry());
    }

    protected void reportMultiget(WebdavRequest request, ReportRequestCalendarMultiget multiget, WebdavResponse response) {
        Tuple2List<DavHref, Option<String>> parsed = multiget.getHrefs().zipWith(
                CalendarUrls.parseUserCalendarEntryIdSafeF(getCollectionId().getId()));

        ListF<String> parsedIds = Cf2.flatBy2(parsed).get2();

        ListF<ComponentGetResult> results = getComponentsByIds(parsedIds, getIsIncludeIcs(multiget));
        Check.sameSize(parsedIds, results);

        val found = results.filter(ComponentGetResult::isFound)
                .map(ComponentGetResult.getEventDescriptionF()::apply);
        ListF<DavHref> notFound = parsed.filterBy2(Cf2.isSomeOfF(found.map(CalendarComponent.getFileNameF())).notF()).get1();

        val rs = prepareResponses(multiget.getPropertiesRequest(), found, notFound);
        MultiStatusUtils.sendMultiStatus(new MultiStatus2(rs), request, response, getMeterRegistry());
    }

    private List<MultiStatusResponse2> addInsufficientStorage(List<MultiStatusResponse2> responses, boolean overflow) {
        if (!overflow) {
            return responses;
        }

        val response = MultiStatusResponse2.insufficientStorageResponse(DavHref.fromEncoded(getHref()));

        return StreamEx.of(responses)
                .append(response)
                .toImmutableList();
    }

    private static boolean isOverflow(int size, Optional<Integer> limit) {
        return limit.stream().anyMatch(lim -> lim < size);
    }

    private Optional<Integer> getLimit(Optional<Integer> limit) {
        val config = caldavContext.getCaldavTruncationConfig();
        log.debug("caldav.truncation.config is equal to {}", config);
        if (config.getTruncationMode() == TruncationMode.NEVER_TRUNCATE) {
            return Optional.empty();
        }
        return limit.or(() -> getDefaultLimit(config));
    }

    private Optional<Integer> getDefaultLimit(CaldavTruncationConfig config) {
        if (config.getTruncationMode() == TruncationMode.REQUEST_ONLY || !doesAgentSupportTruncation(config)) {
            return Optional.empty();
        }

        return Optional.of(config.getDefaultLimit());
    }

    private boolean doesAgentSupportTruncation(CaldavTruncationConfig config) {
        if (config.getTruncationMode() != TruncationMode.ALL_EXCEPT_SPECIFIED) {
            return true;
        }

        val userAgent = caldavRequestContext.getUserAgent();
        return StreamEx.of(config.getAgentRegexps())
                .map(agent -> AGENTS_PATTERNS.computeIfAbsent(agent, Pattern::compile))
                .noneMatch(pattern -> pattern.matcher(userAgent).matches());
    }

    protected void reportSyncCollection(WebdavRequest request, ReportRequestSyncCollection query, WebdavResponse response) {
        val queryToken = query.getSyncToken();
        val serverToken = getSyncToken();

        boolean tokensMatch = queryToken.equals(serverToken);

        log.debug("About to compare tokens: requested '{}' {} actual '{}'",
                queryToken, tokensMatch ? "matches" : "mismatches", serverToken);

        if (tokensMatch) {
            MultiStatusUtils.sendMultiStatus(
                    new MultiStatus2(Cf.list(), queryToken),
                    request, response, getMeterRegistry());
            return;
        }

        val limit = getLimit(query.getLimit());

        log.debug("Actual truncation limit is {}", limit);

        val modifiedAndDeleted = queryToken.isEmpty()
                ? getComponents(CalendarComponentsFilter.any(), getIsIncludeIcs(query)).map(ComponentModified::found)
                : getModifiedComponents(queryToken, getIsIncludeIcs(query));

        val overflow = isOverflow(modifiedAndDeleted.size(), limit);

        val truncated = getTrunkedSortedList(modifiedAndDeleted, limit);

        val rs = addInsufficientStorage(prepareResponsesMixed(query.getPropertiesRequest(), truncated), overflow);

        val token = recalculateSyncToken(serverToken, truncated, queryToken, overflow);

        MultiStatusUtils.sendMultiStatus(new MultiStatus2(rs, token), request, response, getMeterRegistry());
    }

    private List<MultiStatusResponse2> prepareResponses(
            PropertiesRequest properties, List<CalendarComponent> components)
    {
        return prepareResponses(properties, components, emptyList());
    }

    private List<MultiStatusResponse2> prepareResponsesMixed(
            PropertiesRequest properties, List<ComponentModified> components) {
        val partition = StreamEx.of(components).partitioningBy(ComponentGetResult::isFound);

        val found = StreamEx.of(partition.getOrDefault(true, emptyList()))
                .map(ComponentGetResult::getEventDescription)
                .flatMap(Optional::stream)
                .toImmutableList();

        val notFound = StreamEx.of(partition.getOrDefault(false, emptyList()))
                .map(ComponentGetResult::getFileName)
                .map(this::getComponentHref)
                .toImmutableList();

        return prepareResponses(properties, found, notFound);
    }

    private List<MultiStatusResponse2> prepareResponses(
            PropertiesRequest propertiesRequest, List<CalendarComponent> found, List<DavHref> notFound)
    {
        val foundFunction = propStatResponse(propertiesRequest);
        val foundResponse = StreamEx.of(found).map(foundFunction);
        val notFoundResponse = StreamEx.of(notFound)
                .map(x -> MultiStatusResponse2.hrefResponse(x, HttpStatus.SC_404_NOT_FOUND));
        return foundResponse.append(notFoundResponse).toImmutableList();
    }

    private static boolean getIsIncludeIcs(ReportRequestCalendar request) {
        return request.getPropertiesRequest().contains(CaldavConstants.CALDAV_CALENDAR_DATA_PROP);
    }

    private DavHref getComponentHref(String fileName) {
        return DavHref.fromEncoded(getHref()).addDecodedChild(fileName);
    }

    private Function<String, DavHref> getComponentHrefF() {
        return this::getComponentHref;
    }

    private Function<CalendarComponent, MultiStatusResponse2> propStatResponse(PropertiesRequest request) {
        return (child) -> {
            DavHref href = getComponentHref(child.getFileName());
            return MultiStatusResponse2.propStatResponse(href, JackrabbitUtils.getProperties(child, request));
        };
    }
}
