package ru.yandex.calendar.frontend.caldav.proto.tree;

import java.io.IOException;

import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.io.OutputContext;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.DefaultDavProperty;
import org.apache.jackrabbit.webdav.property.ResourceType;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.frontend.caldav.WebdavUtils;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.caldav.LazyDavProperty;
import ru.yandex.calendar.frontend.caldav.proto.facade.CalendarComponent;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * @author Stepan Koltsov
 * @see CarddavResourceUserAddressbookCard
 */
public abstract class CaldavResourceUserCalendarEntryBase extends CaldavResourceUserBase {

    private final String name;
    private final CollectionId collectionId;

    public CaldavResourceUserCalendarEntryBase(
            CaldavRequestContext caldavContext, CollectionId collectionId, String name)
    {
        super(caldavContext, collectionId.getUser(), collectionId.getPassportUid());
        Validate.notEmpty(name);
        this.name = name;
        this.collectionId = collectionId;
    }

    public String getName() {
        return name;
    }

    public CollectionId getCollectionId() {
        return collectionId;
    }

    @Override
    public abstract String getHref();

    @Override
    public abstract DavResource getCollection();

    /**
     * @see CarddavResourceUserAddressbookCard#spool(OutputContext)
     */
    @Override
    public void spool(OutputContext outputContext) throws IOException {
        outputContext.setETag(getEtag().getValue());
        if (outputContext.hasStream()) {
            outputContext.setContentType("text/calendar");
            outputContext.getOutputStream().write(getCalendarData());
        }
    }

    private final Function0<Option<CalendarComponent>> calendarData = new Function0<Option<CalendarComponent>>() {
        public Option<CalendarComponent> apply() {
            // XXX: optimize, ics may not be needed
            return caldavCalendarFacade.getUserCalendarEvent(
                    ClientHolder.getPassportUid(), collectionId, getName(), true, caldavRequestContext.getUserAgentType());
        }
    }.memoize();

    @Override
    public boolean exists() {
        return calendarData.apply().isPresent();
    }

    private byte[] getCalendarData() {
        return CharsetUtils.encodeUtf8ToArray(calendarData.apply().getOrThrow("getCanendarData must not be called if not exists for " + getHref()).getIcal().get());
    }

    private ETag getEtag() {
        return calendarData.apply().get().getEtag();
    }


    /**
     * @see CarddavResourceUserAddressbookCard#getPropertyNames()
     */
    @Override
    public DavPropertyName[] getPropertyNames() {
        return propertyNames();
    }

    public static DavPropertyName[] propertyNames() {
        return new DavPropertyName[] {
                WebdavConstants.DAV_RESOURCETYPE_PROP,
                WebdavConstants.DAV_GETETAG_PROP,
                CaldavConstants.CALDAV_CALENDAR_DATA_PROP,
                WebdavConstants.DAV_GETCONTENTTYPE_PROP,
        };
    }

    /**
     * @see CarddavResourceUserAddressbookCard#getProperty(DavPropertyName)
     * @see ru.yandex.calendar.frontend.caldav.proto.facade.CalendarComponent#getProperty(DavPropertyName)
     */
    @Override
    public DavProperty<?> getProperty(DavPropertyName name) {
        if (name.equals(WebdavConstants.DAV_RESOURCETYPE_PROP)) {
            return new ResourceType(new int[0]);
        } else if (name.equals(WebdavConstants.DAV_GETETAG_PROP)) {
            return new LazyDavProperty<>(WebdavConstants.DAV_GETETAG_PROP, () -> getEtag().getValue(), false);
        } else if (name.equals(CaldavConstants.CALDAV_CALENDAR_DATA_PROP)) {
            return new LazyDavProperty<>(CaldavConstants.CALDAV_CALENDAR_DATA_PROP, () -> StringUtils.decodeUtf8(getCalendarData()), false);
        } else if (name.equals(WebdavConstants.DAV_GETCONTENTTYPE_PROP)) {
            return new DefaultDavProperty<>(WebdavConstants.DAV_GETCONTENTTYPE_PROP, "text/calendar");
        } else {
            return null;
        }
    }

    /**
     * @see CarddavResourceUserAddressbookCard#getProperties()
     */
    @Override
    public DavPropertySet getProperties() {
        return WebdavUtils.getPropertiesFromNames(this);
    }

    @Override
    public void move(DavResource destination) {
        CaldavResourceUserCalendarEntryBase destinationEntry = (CaldavResourceUserCalendarEntryBase) destination;
        caldavCalendarFacade.moveUserCalendarEvent(
                ClientHolder.getPassportUid(), getCollectionId(), destinationEntry.getCollectionId(), getName());
    }

} //~
