package ru.yandex.calendar.frontend.caldav.proto.tree;

import java.lang.reflect.Proxy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

public class CalendarUrls {

    public static String root() {
        return "/";
    }

    private static String calendarsRootStr(String user) {
        return "/calendars/" + user + "/";
    }

    public static DavHref calendarsRoot(String user) {
        return DavHref.fromDecoded(calendarsRootStr(user));
    }

    public static DavHref user(String user) {
        return DavHref.fromDecoded(calendarsRootStr(user) + "user/");
    }

    public static DavHref inbox(String user) {
        return DavHref.fromDecoded(calendarsRootStr(user) + "inbox/");
    }

    public static DavHref outbox(String user) {
        return DavHref.fromDecoded(calendarsRootStr(user) + "outbox/");
    }

    public static DavHref unknown(String user, String id) {
        return DavHref.fromDecoded(calendarsRootStr(user) + "/" + id + "/");
    }

    public static DavHref unknownEntry(String user, String collectionId, String id) {
        return unknown(user, collectionId).addDecodedChild(id);
    }

    private static String eventsStr(String user, String collectionId) {
        return calendarsRootStr(user) + "events-" + collectionId + "/";
    }

    public static DavHref events(String user, String collectionId) {
        return DavHref.fromDecoded(calendarsRootStr(user) + "events-" + collectionId + "/");
    }

    public static DavHref event(String user, String collectionId, String eventId) {
        return DavHref.fromDecoded(eventsStr(user, collectionId) + eventId);
    }

    public static DavHref eventsUser(String user, String collectionId) {
        return DavHref.fromDecoded(eventsStr(user, collectionId) + "user/");
    }

    public static String todosStr(String user, String collectionId) {
        return calendarsRootStr(user) + "todos-" + collectionId + "/";
    }

    public static DavHref todos(String user, String collectionId) {
        return DavHref.fromDecoded(todosStr(user, collectionId));
    }

    public static DavHref todo(String user, String collectionId, String todoId) {
        return DavHref.fromDecoded(todosStr(user, collectionId) + todoId);
    }

    public static DavHref principals() {
        return DavHref.fromDecoded("/principals/");
    }

    public static DavHref principals(String user) {
        return DavHref.fromDecoded("/principals/users/" + user + "/");
    }

    public static DavHref directory() {
        return DavHref.fromDecoded("/directory/");
    }

    public static DavHref directoryContact(String id) {
        return DavHref.fromDecoded("/directory/" + id);
    }

    public static DavHref addressbooksUser(String user) {
        return DavHref.fromDecoded("/addressbook/" + user + "/");
    }

    public static DavHref addressbooksUserAddressbook(String user) {
        return DavHref.fromDecoded("/addressbook/" + user + "/addressbook/");
    }

    public static DavHref addressbookUserAddressbookCard(String user, String id) {
        return DavHref.fromDecoded("/addressbook/" + user + "/addressbook/" + id);
    }


    interface Handler<T> {
        T root();

        T principals();
        T principalsUser(String user);

        T calendarsUser(String user);
        T calendarsUserUser(String user);
        T calendarsUserInbox(String user);
        T calendarsUserOutbox(String user);
        T calendarsUserEvents(String user, String collectionId);
        T calendarsUserEventsEvent(String user, String collectionId, String id);
        T calendarsUserEventsUser(String user, String collectionId);
        T calendarsUserTodos(String user, String collectionId);
        T calendarsUserTodosTodo(String user, String collectionId, String id);

        T directory();
        T directoryContact(String id);

        T addressbooksUser(String user);
        T addressbooksUserAddressbook(String user);
        T addressbooksUserAddressbookCard(String user, String id);

        T calendarsUserUnknown(String user, String collectionId);
        T calendarsUserUnknownEntry(String user, String collectionId, String id);
    }

    private static final Pattern2 pp = Pattern2.compile("/principals/users/([^/]+)/?");
    private static final Pattern2 calendarsPattern = Pattern2.compile("/calendars/([^/]+)/?(.*)");
    private static final Pattern2 addressbookPattern = Pattern2.compile("/addressbook/([^/]+)/?(.*)");

    public static <T> T parse(DavHref href, Handler<T> handler) {
        String uri = href.getDecoded();

        ListF<String> path = Pattern2.compile("/+").split(uri).filter(Cf.String.notEmptyF());

        if (path.isEmpty())
            return handler.root();

        if (path.first().equals("principals")) {
            if (path.length() == 1) {
                return handler.principals();
            }

            Matcher2 principalsMatcher = pp.matcher2(uri);
            if (principalsMatcher.matches()) {
                return handler.principalsUser(principalsMatcher.group(1).get());
            }
        }

        Matcher2 calendarsMatcher = calendarsPattern.matcher2(uri);
        if (calendarsMatcher.matches()) {
            return parseCalendars(uri, calendarsMatcher, handler);
        }

        if (path.get(0).equals("directory")) {
            if (path.length() == 1) {
                return handler.directory();
            } else if (path.length() == 2) {
                return handler.directoryContact(path.get(1));
            }
        }

        Matcher2 addressbookMatcher = addressbookPattern.matcher2(uri);
        if (addressbookMatcher.matches()) {
            return parseAddressbook(uri, addressbookMatcher, handler);
        }

        throw new ResourceNotFoundException("resource not found: " + uri);
    }

    @SuppressWarnings({"SuspiciousInvocationHandlerImplementation", "unchecked"})
    private static final Handler<String> methodNameHandler =
            (Handler<String>) Proxy.newProxyInstance(
                    Handler.class.getClassLoader(),
                    new Class[] { Handler.class },
                    (proxy, method, args) -> method.getName());

    public static String covertToTagForLog(DavHref href) {
        try {
            return parse(href, methodNameHandler);
        } catch (ResourceNotFoundException e) {
            return "unknown";
        }
    }

    public static Option<String> parseCalendarEntryIdSafe(String collectionId, DavHref href) {
        try {
            return parse(href, calendarEntryIdUrlHandler(collectionId));
        } catch (ResourceNotFoundException e) {
            return Option.empty();
        }
    }

    public static Function<DavHref, Option<String>> parseUserCalendarEntryIdSafeF(final String collectionId) {
        return new Function<DavHref, Option<String>>() {
            public Option<String> apply(DavHref h) {
                return parseCalendarEntryIdSafe(collectionId, h);
            }
        };
    }

    private static <T> T parseAddressbook(String uri, Matcher2 matcher, Handler<T> handler) {
        ListF<String> groups = matcher.groups();
        Check.isTrue(groups.size() == 2);
        String user = groups.get(0);
        String rest = groups.get(1);

        if (rest.equals("")) {
            return handler.addressbooksUser(user);
        } else if (rest.matches("addressbook/?")) {
            return handler.addressbooksUserAddressbook(user);
        } else if (rest.matches("addressbook/[^/]+")) {
            return handler.addressbooksUserAddressbookCard(user, rest.replaceFirst(".*/", ""));
        } else {
            throw new ResourceNotFoundException("resource not found: " + uri);
        }
    }

    private static final Pattern2 ep = Pattern2.compile("events-([^/]+)/?(.*)");
    private static final Pattern2 tp = Pattern2.compile("todos-([^/]+)/?(.*)");
    private static final Pattern2 up = Pattern2.compile("([^/]+)/?([^/]*)");

    private static <T> T parseCalendars(String uri, Matcher2 matcher, Handler<T> handler) {
        ListF<String> groups = matcher.groups();
        Check.isTrue(groups.size() == 2);
        String user = groups.get(0);
        String rest = groups.get(1);

        if (rest.equals(""))
            return handler.calendarsUser(user);
        else if (rest.matches("user/?"))
            return handler.calendarsUserUser(user);
        else if (rest.matches("inbox/?"))
            return handler.calendarsUserInbox(user);
        else if (rest.matches("outbox/?"))
            return handler.calendarsUserOutbox(user);
        else if (rest.matches("events-.*")) {
            ListF<String> groups2 = ep.matcherGroups(rest);
            String eventsId = groups2.get(0);
            String eventId = groups2.get(1);
            if (eventId.isEmpty()) {
                return handler.calendarsUserEvents(user, eventsId);
            } else {
                if (eventId.matches("user/?"))
                    return handler.calendarsUserEventsUser(user, eventsId);
                else
                    return handler.calendarsUserEventsEvent(user, eventsId, eventId);
            }
        } else if (rest.matches("todos-.*")) {
            ListF<String> groups2 = tp.matcherGroups(rest);
            String collectionId = groups2.get(0);
            String entryId = groups2.get(1);
            return entryId.isEmpty() ?
                handler.calendarsUserTodos(user, collectionId) :
                handler.calendarsUserTodosTodo(user, collectionId, entryId);
        } else {
            matcher = up.matcher2(rest);
            if (matcher.matches()) {
                ListF<String> groups2 = matcher.groups();
                String collectionId = groups2.get(0);
                String entryId = groups2.get(1);

                return entryId.isEmpty() ?
                        handler.calendarsUserUnknown(user, collectionId) :
                        handler.calendarsUserUnknownEntry(user, collectionId, entryId);
            }
        }

        throw new ResourceNotFoundException("resource not found: " + uri);
    }

    private static Handler<Option<String>> calendarEntryIdUrlHandler(final String collId) {
        return new Handler<Option<String>>() {
            public Option<String> calendarsUserEventsEvent(String user, String collectionId, String id) {
                return Option.when(collectionId.equals(collId), id);
            }

            public Option<String> calendarsUserTodosTodo(String user, String collectionId, String id) {
                return Option.when(collectionId.equals(collId), id);
            }

            public Option<String> calendarsUserUnknownEntry(String user, String collectionId, String id) {
                return Option.when(collectionId.equals(collId), id);
            }

            public Option<String> root() { return Option.empty(); }
            public Option<String> principals() { return Option.empty(); }
            public Option<String> principalsUser(String user) { return Option.empty(); }
            public Option<String> calendarsUser(String user) { return Option.empty(); }
            public Option<String> calendarsUserUser(String user) { return Option.empty(); }
            public Option<String> calendarsUserInbox(String user) { return Option.empty(); }
            public Option<String> calendarsUserOutbox(String user) { return Option.empty(); }
            public Option<String> calendarsUserEvents(String user, String collectionId) { return Option.empty(); }
            public Option<String> calendarsUserEventsUser(String user, String collectionId) { return Option.empty(); }
            public Option<String> calendarsUserTodos(String user, String collectionId) { return Option.empty(); }
            public Option<String> directory() { return Option.empty(); }
            public Option<String> directoryContact(String id) { return Option.empty(); }
            public Option<String> addressbooksUser(String user) { return Option.empty(); }
            public Option<String> addressbooksUserAddressbook(String user) { return Option.empty(); }
            public Option<String> addressbooksUserAddressbookCard(String user, String id) { return Option.empty(); }
            public Option<String> calendarsUserUnknown(String user, String collectionId) { return Option.empty(); }
        };
    }
} //~
