package ru.yandex.calendar.frontend.caldav.proto.tree;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

import io.micrometer.core.instrument.MeterRegistry;
import org.apache.jackrabbit.webdav.DavConstants;
import org.apache.jackrabbit.webdav.DavException;
import org.apache.jackrabbit.webdav.DavResource;
import org.apache.jackrabbit.webdav.DavResourceFactory;
import org.apache.jackrabbit.webdav.DavResourceIterator;
import org.apache.jackrabbit.webdav.DavResourceLocator;
import org.apache.jackrabbit.webdav.DavServletResponse;
import org.apache.jackrabbit.webdav.DavSession;
import org.apache.jackrabbit.webdav.MultiStatusResponse;
import org.apache.jackrabbit.webdav.WebdavRequest;
import org.apache.jackrabbit.webdav.WebdavResponse;
import org.apache.jackrabbit.webdav.io.InputContext;
import org.apache.jackrabbit.webdav.io.OutputContext;
import org.apache.jackrabbit.webdav.lock.ActiveLock;
import org.apache.jackrabbit.webdav.lock.LockInfo;
import org.apache.jackrabbit.webdav.lock.LockManager;
import org.apache.jackrabbit.webdav.lock.Scope;
import org.apache.jackrabbit.webdav.lock.Type;
import org.apache.jackrabbit.webdav.property.DavProperty;
import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertySet;
import org.apache.jackrabbit.webdav.property.PropEntry;
import org.apache.jackrabbit.webdav.server.AbstractWebdavServlet;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.caldav.proto.ClientHolder;
import ru.yandex.calendar.frontend.caldav.proto.ETag;
import ru.yandex.calendar.frontend.caldav.proto.PutResponse;
import ru.yandex.calendar.frontend.caldav.proto.ccdav.CcDavUtils;
import ru.yandex.calendar.frontend.caldav.proto.jackrabbit.JackrabbitUtils;
import ru.yandex.calendar.frontend.caldav.proto.webdav.report.PropertiesRequest;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatus2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.MultiStatusResponse2;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Stepan Koltsov
 */
public abstract class WebdavResourceBase extends PropertiesCollectionBase implements WebdavResource {
    private static final Logger logger = LoggerFactory.getLogger(WebdavResourceBase.class);

    public static final String METHODS = DavResource.METHODS + ", " + (Cf.list("REPORT", "MKCALENDAR").mkString(", "));

    protected final CaldavRequestContext caldavRequestContext;
    protected final CaldavContext caldavContext;

    public WebdavResourceBase(CaldavRequestContext caldavRequestContext) {
        this.caldavRequestContext = caldavRequestContext;
        this.caldavContext = caldavRequestContext.getCaldavContext();
    }

    protected UnsupportedOperationException unsupportedOperationException() {
        return new UnsupportedOperationException("on " + this.getClass().getName());
    }

    @Override
    public String getComplianceClass() {
        return CcDavUtils.complianceClass(caldavRequestContext.getUserAgentType());
    }

    @Override
    public void addLockManager(LockManager lockmgr) {
        throw unsupportedOperationException();
    }

    @Override
    public void addMember(DavResource resource, InputContext inputContext) throws DavException {
        throw unsupportedOperationException();
    }

    public MeterRegistry getMeterRegistry() {
        return caldavRequestContext.getRegistry();
    }

    @Override
    public PutResponse put(WebdavResource resource, InputContext inputContext) throws DavException {
        addMember(resource, inputContext);
        // 201 response is important for CalDAV Sync Adapter (Android) (CAL-6518)
        return new PutResponse(DavServletResponse.SC_CREATED, Option.<ETag>empty(), Option.<String>empty());
    }

    /**
     * @url http://tools.ietf.org/html/rfc4918#section-9.2 (PROPPATCH Method)
     */
    @Override
    public MultiStatusResponse alterProperties(List<? extends PropEntry> changeList) throws DavException {
        MultiStatusResponse response = new MultiStatusResponse(getHref(), "PROPPATCH not implemented");

        for (PropEntry prop0 : changeList) {
            if (prop0 instanceof DavPropertyName) {
                // delete
                DavPropertyName propName = (DavPropertyName) prop0;
                logger.warn("Requested does not implement alterProperties, returning 404 for all, to delete " + propName + " on " + getHref());
                response.add(propName, HttpStatus.SC_404_NOT_FOUND);
            } else if (prop0 instanceof DavProperty) {
                // add/set
                DavProperty<?> prop = (DavProperty<?>) prop0;
                DavPropertyName propName = prop.getName();
                logger.warn("Requested does not implement alterProperties, returning 404 for all, to add/set " + propName + " -> " + prop.getValue() + " on " + getHref());
                response.add(propName, HttpStatus.SC_404_NOT_FOUND);
            } else {
                logger.warn(getHref() + " does not implement alterProperties, returning 404 for all: " + prop0);
                throw new IllegalArgumentException("unsupported element in list: " + prop0);
            }
        }

        return response;
    }

    @Override
    public void copy(DavResource destination, boolean shallow) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public boolean exists() {
        return true;
    }

    @Override
    public boolean isAccessibleTo(String client) {
        return true;
    }

    @Override
    public DavResource getCollection() {
        throw unsupportedOperationException();
    }

    @Override
    public String getDisplayName() {
        throw unsupportedOperationException();
    }

    @Override
    public DavResourceFactory getFactory() {
        throw unsupportedOperationException();
    }

    @Override
    public DavPropertySet getProperties() {
        throw unsupportedOperationException();
    }

    @Override
    public String getResourcePath() {
        throw unsupportedOperationException();
    }

    @Override
    public DavSession getSession() {
        throw unsupportedOperationException();
    }

    @Override
    public String getSupportedMethods() {
        return METHODS;
    }

    @Override
    public boolean hasLock(Type type, Scope scope) {
        throw unsupportedOperationException();
    }

    @Override
    public DavResourceLocator getLocator() {
        return caldavContext.getCaldavLocatorFactory().createResourceLocator("", getHref());
    }

    @Override
    public ActiveLock getLock(Type type, Scope scope) {
        throw unsupportedOperationException();
    }

    @Override
    public ActiveLock[] getLocks() {
        throw unsupportedOperationException();
    }

    @Override
    public DavResourceIterator getMembers() {
        throw unsupportedOperationException();
    }

    @Override
    public ListF<WebdavResource> getMembersList() {
        return Cf.x(getMembers()).toList().uncheckedCast();
    }

    @Override
    public long getModificationTime() {
        return DavConstants.UNDEFINED_TIME;
    }


    @Override
    public boolean isLockable(Type type, Scope scope) {
        throw unsupportedOperationException();
    }

    @Override
    public ActiveLock lock(LockInfo reqLockInfo) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public void move(DavResource destination) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public ActiveLock refreshLock(LockInfo reqLockInfo, String lockToken) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public void removeMember(DavResource member) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public void removeProperty(DavPropertyName propertyName) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public void setProperty(DavProperty<?> property) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public void unlock(String lockToken) throws DavException {
        throw unsupportedOperationException();
    }

    @Override
    public boolean hasCustomGet() {
        return false;
    }

    @Override
    public void get(WebdavRequest request, WebdavResponse response) throws DavException, IOException {
        throw unsupportedOperationException();
    }

    @Override
    public void spool(OutputContext outputContext) throws IOException {
        outputContext.setContentType("text/plain");
        PrintWriter pw = new PrintWriter(outputContext.getOutputStream());
        if (isCollection()) {
            for (WebdavResource r : getMembersList()) {
                if (!r.isAccessibleTo(ClientHolder.getSafe())) continue;

                pw.println(r.getHref());
            }
            pw.println("$");
        } else {
            pw.println("leaf");
        }
        pw.flush();
    }

    @Override
    public boolean isCollection() {
        return false;
    }

    @Override
    public void post(WebdavRequest request, WebdavResponse response) throws DavException, IOException {
        throw unsupportedOperationException();
    }

    @Override
    public void report(WebdavRequest request, WebdavResponse response) throws DavException, IOException {
        response.sendError(JackrabbitUtils.supportedReportError());
    }

    protected MultiStatusResponse2 propFindDepth0(PropertiesRequest propertiesRequest)
        throws DavException, IOException
    {
        return JackrabbitUtils.getPropertiesResponse(this, propertiesRequest);
    }

    protected ListF<MultiStatusResponse2> propFindChildren(PropertiesRequest propertiesRequest, int depth)
        throws DavException, IOException
    {
        ListF<MultiStatusResponse2> responses = Cf.arrayList();
        for (WebdavResource child : this.getMembersList()) {
            if (!child.isAccessibleTo(ClientHolder.getSafe())) continue;

            MultiStatus2 childMultiStatus = child.propFind(propertiesRequest, depth);
            responses.addAll(childMultiStatus.getResponse2s());

            // XXX: missed children of children
        }
        return responses;
    }

    /**
     * @see AbstractWebdavServlet#doPropFind(WebdavRequest, WebdavResponse, DavResource)
     */
    @Override
    public MultiStatus2 propFind(PropertiesRequest propertiesRequest, int depth)
        throws DavException, IOException
    {
        Validate.isTrue(depth >= 0);

        ListF<MultiStatusResponse2> responses = Cf.arrayList();

        responses.add(propFindDepth0(propertiesRequest));

        if (depth > 0 && isCollection()) {
            responses.addAll(propFindChildren(propertiesRequest, depth - 1));
        }

        return new MultiStatus2(responses);

    }

}
