package ru.yandex.calendar.frontend.caldav.proto.webdav.report;

import org.apache.jackrabbit.webdav.property.DavPropertyName;
import org.apache.jackrabbit.webdav.property.DavPropertyNameSet;
import org.w3c.dom.Element;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.misc.bender.annotation.BenderParseSubclasses;
import ru.yandex.misc.bender.annotation.BendingMethod;
import ru.yandex.misc.bender.annotation.GlobalPartName;

/**
 * @author Stepan Koltsov
 */
@BenderParseSubclasses({ PropertiesRequest.All.class, PropertiesRequest.PropNames.class, PropertiesRequest.Set.class })
public abstract class PropertiesRequest {

    public abstract boolean contains(DavPropertyName propertyName);

    @GlobalPartName(name="allprop", namespace=WebdavConstants.DAV_NS_URI)
    public static final class All extends PropertiesRequest {
        private All() { }

        @Override
        public PropFindType getType() {
            return PropFindType.ALL_PROP;
        }

        @Override
        public boolean contains(DavPropertyName propertyName) {
            return true;
        }
    }

    @GlobalPartName(name="propname", namespace=WebdavConstants.DAV_NS_URI)
    public static final class PropNames extends PropertiesRequest {
        private PropNames() { }

        @Override
        public PropFindType getType() {
            return PropFindType.PROPERTY_NAMES;
        }

        @Override
        public boolean contains(DavPropertyName propertyName) {
            return true;
        }
    }

    public Option<DavPropertyNameSet> getDavPropertyNameSetO() {
        return Option.empty();
    }

    public abstract PropFindType getType();

    @GlobalPartName(name="prop", namespace=WebdavConstants.DAV_NS_URI)
    public static final class Set extends PropertiesRequest {
        private final DavPropertyNameSet davPropertyNameSet;

        public Set(DavPropertyNameSet davPropertyNameSet) {
            this.davPropertyNameSet = davPropertyNameSet;
        }

        @BendingMethod
        public static Set parse(Element element) {
            return new Set(new DavPropertyNameSet(element));
        }

        @Override
        public Option<DavPropertyNameSet> getDavPropertyNameSetO() {
            return Option.of(davPropertyNameSet);
        }

        public DavPropertyNameSet getDavPropertyNameSet() {
            return davPropertyNameSet;
        }

        @Override
        public boolean contains(DavPropertyName propertyName) {
            return davPropertyNameSet.contains(propertyName);
        }

        @Override
        public PropFindType getType() {
            return PropFindType.BY_PROPERTY;
        }

    }

    public static PropertiesRequest all() {
        return new All();
    }

    public static PropertiesRequest names() {
        return new PropNames();
    }

    public static PropertiesRequest set(DavPropertyNameSet davPropertyNameSet) {
        return new Set(davPropertyNameSet);
    }

} //~
