package ru.yandex.calendar.frontend.caldav.proto.webdav.report;

import org.w3c.dom.Element;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.caldav.proto.caldav.CaldavConstants;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentCondition;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentConditions;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.CalendarComponentsFilter;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestCalendarParser;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.ReportRequestSyncCollectionParser;
import ru.yandex.calendar.frontend.caldav.proto.caldav.report.TimeRange;
import ru.yandex.calendar.frontend.caldav.proto.carddav.CarddavConstants;
import ru.yandex.calendar.frontend.caldav.proto.carddav.report.ReportRequestAddressbookParser;
import ru.yandex.calendar.frontend.caldav.proto.carddav.report.ReportRequestAddressbookQueryParser;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.calendar.frontend.caldav.proto.webdav.xml.WebdavXmlizers;
import ru.yandex.calendar.logic.ics.IcsUtils;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderXmlParser;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.xml.dom.DomUtils;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @author Stepan Koltsov
 *
 * XXX: AFAIR, unknown elements should be ignored
 */
public class ReportRequestParser {
    public static final Logger logger = LoggerFactory.getLogger(ReportRequestParser.class);

    private static TimeRange parseTimeRange(Element element) {
        DomUtils.validateName(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_TIME_RANGE_QNAME));
        Option<String> start = StringUtils.notEmptyO(element.getAttribute("start"));
        Option<String> end = StringUtils.notEmptyO(element.getAttribute("end"));
        return new TimeRange(start.map(IcsUtils.parseTimestampF()), end.map(IcsUtils.parseTimestampF()));
    }

    /**
     *    <!ELEMENT prop-filter (is-not-defined |
     *                           ((time-range | text-match)?,
     *                            param-filter*))>
     *
     *    <!ATTLIST prop-filter name CDATA #REQUIRED>
     *    name value: a calendar property name (e.g., ATTENDEE)
     *
     * @url http://tools.ietf.org/html/rfc4791#section-9.7.2
     */
    private static CalendarComponentCondition parsePropFilter(Element element) {
        String propertyName = element.getAttribute("name");
        return new CalendarComponentCondition.PropertyCondition(propertyName);
    }

    private static CalendarComponentCondition parseVeventCompFilterChild(Element element) {
        if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_TIME_RANGE_QNAME))) {
            return new CalendarComponentCondition.TimeRangeCondition(parseTimeRange(element));
        } else if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_PROP_FILTER_QNAME))) {
            return parsePropFilter(element);
        } else {
            throw new IllegalArgumentException("unknown element: " + DomUtils.qname(element));
        }
    }

    private static CalendarComponentsFilter parseVcalendarCompFilter(Element element) {
        DomUtils.validateName(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_COMP_FILTER_QNAME));
        Validate.V.equals("VCALENDAR", element.getAttribute("name"));

        ListF<Element> compFilters = DomUtils.childElements(element);

        Option<CalendarComponentConditions> veventConditions = Option.empty();
        Option<CalendarComponentConditions> vtodoConditions = Option.empty();

        for (Element compFilter : compFilters) {
            CalendarComponentConditions condition = parseVeventCompFilter(compFilter);
            String name = compFilter.getAttribute("name");
            if (name.equals("VEVENT")) {
                veventConditions = Option.of(condition);
            } else if (name.equals("VTODO")) {
                vtodoConditions = Option.of(condition);
            }
        }

        return new CalendarComponentsFilter(veventConditions, vtodoConditions);
    }

    /**
     *
     *   <!ELEMENT comp-filter (is-not-defined | (time-range?,
     *                           prop-filter*, comp-filter*))>
     *
     *   <!ATTLIST comp-filter name CDATA #REQUIRED>
     *   name value: a calendar object or calendar component
     *                type (e.g., VEVENT)
     *
     */
    private static CalendarComponentConditions parseVeventCompFilter(Element compFilter) {
        DomUtils.validateName(compFilter, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_COMP_FILTER_QNAME));

        return CalendarComponentConditions.all(DomUtils.childElements(compFilter).map(new Function<Element, CalendarComponentCondition>() {
            public CalendarComponentCondition apply(Element element11) {
                return parseVeventCompFilterChild(element11);
            }
        }));
    }

    public static CalendarComponentsFilter parseFilter(Element element) {
        DomUtils.validateName(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_FILTER_QNAME));
        if (DomUtils.childElements(element).isEmpty())
            return CalendarComponentsFilter.any();
        else
            return parseVcalendarCompFilter(DomUtils.childElements(element).single());
    }

    public static DavHref parseHref(Element element) {
        return WebdavXmlizers.hrefXmlizer.parseXml(element);
    }

    private static final BenderXmlParser<ReportRequest> reportRequestParser = Bender.xmlParser(ReportRequest.class);

    public static ReportRequest parse(Element element) {
        // XXX: implement all parsers using {@link #reportRequestParser}
        if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_CALENDAR_QUERY_QNAME))) {
            return ReportRequestCalendarParser.parseQuery(element);
        } else if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(CaldavConstants.CALDAV_CALENDAR_MULTIGET_QNAME))) {
            return ReportRequestCalendarParser.parseMultiget(element);
        } else if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(WebdavConstants.DAV_SYNC_COLLECTION_QNAME))) {
            return ReportRequestSyncCollectionParser.parseSyncCollection(element);
        } else if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(WebdavConstants.DAV_EXPAND_PROPERTY_QNAME))) {
            return ReportRequestExpandPropertyParser.parse(element);
        } else if (DomUtils.nameIs(element, Dom4jUtils.toJavaxXmlNamespaceQName(CarddavConstants.CARDDAV_ADDRESSBOOK_QUERY_QNAME))) {
            return ReportRequestAddressbookQueryParser.parse(element);
        } else if (StringUtils.equals(element.getNamespaceURI(), CarddavConstants.CARDDAV_NS_URI)) {
            return ReportRequestAddressbookParser.P.parse(element);
        } else {
            return reportRequestParser.parseXml(element);
        }
    }

    public static ReportRequest parseXmlString(String string) {
        return parse(DomUtils.I.readRootElement(string));
    }

} //~
