package ru.yandex.calendar.frontend.caldav.proto.webdav.xml;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import lombok.val;
import org.apache.http.HttpStatus;
import org.dom4j.DocumentFactory;
import org.dom4j.Element;
import org.dom4j.QName;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.caldav.proto.webdav.DavHref;
import ru.yandex.calendar.frontend.caldav.proto.webdav.HttpReasonPhrases;
import ru.yandex.calendar.frontend.caldav.proto.webdav.WebdavConstants;
import ru.yandex.misc.xml.dom4j.Dom4jUtils;

/**
 * @url http://tools.ietf.org/html/rfc2518#section-12.9.1
 * <pre>
 *      <!ELEMENT response (href, ((href*, status)|(propstat+)), responsedescription?) >
 * </pre>
 *
 * @see MultiStatus2
 */
public class MultiStatusResponse2 {
    private abstract static class ResponseContent {
        protected abstract List<Integer> getStatuses();
        protected abstract ListF<Element> toXml();
    }

    private static class HrefResponseContent extends ResponseContent {
        private final DavHref href;
        private final int status;

        private HrefResponseContent(DavHref href, int status) {
            this.href = href;
            this.status = status;
        }

        @Override
        protected List<Integer> getStatuses() {
            return Arrays.asList(status);
        }

        protected Element createElement(QName name) {
            return DocumentFactory.getInstance().createElement(name);
        }

        @Override
        protected ListF<Element> toXml() {
            val r = Dom4jUtils.readRootElement(WebdavXmlizers.hrefXmlizer.serializeXml(href));
            val statusElement = createElement(WebdavConstants.DAV_STATUS_QNAME);
            val statusLine = "HTTP/1.1 " + status + " " + HttpReasonPhrases.getReasonPhrase(status);
            statusElement.addText(statusLine);
            return Cf.list(r, statusElement);
        }
    }

    private static class InsufficientStorageResponseContent extends HrefResponseContent {
        private InsufficientStorageResponseContent(DavHref href) {
            super(href, HttpStatus.SC_INSUFFICIENT_STORAGE);
        }

        @Override
        protected ListF<Element> toXml() {
            val list = super.toXml();
            val error = createElement(WebdavConstants.DAV_ERROR_QNAME);
            val limits = createElement(WebdavConstants.DAV_LIMITS_MATCHES_QNAME);

            error.add(limits);
            return list.plus(error);
        }
    }

    private static class PropStatResponseContent extends ResponseContent {
        private final DavHref href;
        private final List<PropStat> propStats;

        private PropStatResponseContent(DavHref href, List<PropStat> propStats) {
            this.href = href;
            this.propStats = propStats;
        }

        @Override
        protected List<Integer> getStatuses() {
            return propStats.stream().map(PropStat::getStatus).collect(Collectors.toList());
        }

        @Override
        protected ListF<Element> toXml() {
            ListF<Element> r = Cf.arrayList();
            r.add(Dom4jUtils.readRootElement(WebdavXmlizers.hrefXmlizer.serializeXml(href)));
            for (PropStat propStat : propStats) {
                r.add(propStat.toXml());
            }
            return r;
        }
    }

    public List<Integer> getStatuses() {
        return responseContent.getStatuses();
    }

    private final ResponseContent responseContent;
    private final Optional<String> responseDescription;


    private MultiStatusResponse2(ResponseContent responseContent,
            Optional<String> responseDescription)
    {
        this.responseContent = responseContent;
        this.responseDescription = responseDescription;
    }

    public static MultiStatusResponse2 hrefResponse(DavHref href, int status, Optional<String> responseDescription) {
        return new MultiStatusResponse2(new HrefResponseContent(href, status), responseDescription);
    }

    public static MultiStatusResponse2 hrefResponse(DavHref href, int status) {
        return hrefResponse(href, status, Optional.empty());
    }

    public static MultiStatusResponse2 insufficientStorageResponse(DavHref href) {
        return new MultiStatusResponse2(new InsufficientStorageResponseContent(href), Optional.empty());
    }

    public static MultiStatusResponse2 hrefResponse(DavHref href, int status, String responseDescription) {
        return hrefResponse(href, status, Optional.of(responseDescription));
    }

    public static MultiStatusResponse2 propStatResponse(DavHref href, List<PropStat> propStats, Optional<String> responseDescription) {
        return new MultiStatusResponse2(new PropStatResponseContent(href, propStats), responseDescription);
    }

    public static MultiStatusResponse2 propStatResponse(DavHref href, List<PropStat> propStats, String responseDescription) {
        return propStatResponse(href, propStats, Optional.of(responseDescription));
    }

    public static MultiStatusResponse2 propStatResponse(DavHref href, List<PropStat> propStats) {
        return propStatResponse(href, propStats, Optional.empty());
    }

    @SuppressWarnings("unchecked")
    public Element toXml() {
        Element r = DocumentFactory.getInstance().createElement(WebdavConstants.DAV_RESPONSE_QNAME);
        r.content().addAll(responseContent.toXml());
        responseDescription.ifPresent(s -> r.addElement(WebdavConstants.DAV_RESPONSEDESCRIPTION_QNAME).addText(s));
        return r;
    }

    public static Function2<DavHref, Integer, MultiStatusResponse2> hrefResponseF2() {
        return MultiStatusResponse2::hrefResponse;
    }

    public static Function<DavHref, MultiStatusResponse2> hrefResponseF(int statusCode) {
        return hrefResponseF2().bind2(statusCode);
    }
} //~
