package ru.yandex.calendar.frontend.display;

import java.util.EnumSet;
import java.util.Optional;

import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.display.dto.MeetingData;
import ru.yandex.calendar.frontend.display.dto.MeetingIdInfo;
import ru.yandex.calendar.frontend.display.dto.MeetingInfo;
import ru.yandex.calendar.frontend.display.dto.MeetingParticipantInfo;
import ru.yandex.calendar.frontend.display.dto.RoomInfo;
import ru.yandex.calendar.frontend.display.dto.RoomWithScheduleInfo;
import ru.yandex.calendar.frontend.display.dto.RoomsInfo;
import ru.yandex.calendar.frontend.display.dto.ScheduleIntervalInfo;
import ru.yandex.calendar.frontend.display.dto.StateInfo;
import ru.yandex.calendar.frontend.display.dto.TokenInfo;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventResourceUncheckin;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActorId;
import ru.yandex.calendar.logic.event.ChangedEventInfoForMails;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.dao.EventResourceUncheckinDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventInvitationUpdateData;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.ics.exp.EventInstanceParameters;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.SpecialResources;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationInfo;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationManager;
import ru.yandex.calendar.logic.sending.EventSendingInfo;
import ru.yandex.calendar.logic.sending.param.EventMessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.calendar.logic.suggest.IntervalSet;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.EventAction;
import ru.yandex.calendar.util.base.UidGen;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.calendar.util.dates.DateOrDateTime;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;

@Slf4j
public class DisplayManager {
    public static final PassportUid ROBOT_RES_MASTER_UID = PassportUid.cons(1120000000118099L);
    public static final Email ROBOT_RES_MASTER_EMAIL = new Email("robot-resmaster@yandex-team.ru");

    private enum UpdateOptions {
        SEND_UNCHECKIN_MAIL,
        SUPPRESS_GENERAL_MAILS
    }

    @Value
    private static class EmailEventInfo {
        Event event;
        EventInstanceParameters instance;
    }

    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private EventResourceUncheckinDao eventResourceUncheckinDao;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private ResourceReservationManager resourceReservationManager;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private Authorizer authorizer;

    public RoomsInfo findUnassignedRooms(Language lang, ActionInfo actionInfo) {
        Comparator<Resource> resourceC = Resource.getF(ResourceFields.OFFICE_ID).andThenNaturalComparator()
                .thenComparing(Resource.getF(ResourceFields.FLOOR_NUM).andThenNaturalComparator().nullLowC())
                .thenComparing(Resource.getF(ResourceFields.EXCHANGE_NAME).andThenNaturalComparator().nullLowC())
                .uncheckedCastC();

        ListF<ResourceInfo> resources = resourceRoutines.getYtActiveRoomsWithoutDisplayToken()
                .sorted(ResourceInfo.resourceF().andThen(resourceC));

        return new RoomsInfo(resources.map(res -> toRoom(res, lang, actionInfo)));
    }

    public TokenInfo assignRoom(long id) {
        Option<String> resourceToken = resourceRoutines.loadById(id).getDisplayToken();
        if (resourceToken.isPresent()) {
            throw CommandRunException.createSituation("already assigned resource", Situation.PERMISSION_ERROR);
        }
        String generatedToken = UidGen.createPrivateToken();
        resourceRoutines.updateResourceDisplayTokenById(id, generatedToken);

        return new TokenInfo(generatedToken);
    }

    public StateInfo getState(String token, Language lang, ActionInfo actionInfo) {
        Instant now = actionInfo.getNow();

        ResourceInfo resource = getResourceInfo(token);
        DateTimeZone officeTz = OfficeManager.getOfficeTimeZone(resource.getOffice());

        updateLastPingTs(resource.getResourceId(), now);

        EventGetProps egp = EventGetProps.any();
        InstantInterval interval = new InstantInterval(
                now, new DateTime(now, officeTz).plusDays(1).withTimeAtStartOfDay());
        EventLoadLimits limits = EventLoadLimits.intersectsInterval(interval);

        ListF<EventInstanceInfo> events = eventRoutines.getSortedInstancesOnResource(
                Option.empty(), egp, Cf.list(resource.getResourceId()),
                limits, actionInfo.getActionSource());

        ListF<ResourceReservationInfo> reservations = resourceReservationManager.findReservations(
                Cf.list(resource.getResourceId()), interval, Option.empty(), actionInfo);

        ListF<MeetingInfo> restrictions = SpecialResources.getRestrictionDates(resource)
                .filterMap(m -> toMeeting(m, new LocalDate(now, officeTz), lang));

        ListF<MeetingInfo> meetings = events.map(event -> toMeeting(event, officeTz, lang))
                .plus(reservations.flatMap(r -> toMeetings(r, interval, officeTz, lang)))
                .plus(restrictions);

        IntervalSet gapsSet = IntervalSet.cons(meetings.map(m -> m.getInterval(officeTz))).invert(interval);

        ListF<ScheduleIntervalInfo> schedule = meetings.map(ScheduleIntervalInfo.meetingF())
                .plus(gapsSet.getIntervals().map(ScheduleIntervalInfo.gapF(officeTz)))
                .sortedBy(ScheduleIntervalInfo.getStartF());

        SpecialResources.RoomsDisplaySettings settings = SpecialResources.getDisplaySettings(resource.getResource());
        Option<Duration> maxDuration = SpecialResources.getDurationLimit(resource.getResource());
        Option<Duration> minDuration = SpecialResources.getMinimalDurationLimit(resource.getResource());

        return new StateInfo(
                new RoomWithScheduleInfo(toRoom(resource, lang, actionInfo), schedule),
                resourceRoutines.isAutoDeclineFromDisplay(resource.getResourceId()),
                settings.getCheckInStart(), settings.getCheckInEnd(),
                maxDuration.map(Duration::getStandardMinutes), minDuration.map(Duration::getStandardMinutes));
    }

    public MeetingIdInfo createMeeting(String token, MeetingData data, ActionInfo actionInfo) {
        val resource = getResourceInfo(token);
        val officeTz = OfficeManager.getOfficeTimeZone(resource.getOffice());
        val attendees = singletonList(resource.getEmail());

        val notifications = new NotificationsData.CreateWithData(Cf.list(), Option.empty());
        val eventData = data.toEventData(officeTz, ROBOT_RES_MASTER_EMAIL, attendees);
        val eventInfo = eventWebManager.createUserEvent(ROBOT_RES_MASTER_UID, eventData, notifications,
                InvitationProcessingMode.SAVE_ONLY, actionInfo);

        val instanceStartTs = eventInfo.getEvent().getStartTs().toDateTime(officeTz).toLocalDateTime();
        return MeetingIdInfo.event(eventInfo.getEventId(), instanceStartTs);
    }

    private void doIfLastResourceOrElse(Event event, Runnable onTrue, Runnable onFalse) {
        doIfLastResourceOrElse(eventResourceDao.findEventResourcesByEventId(event.getId()).size(), onTrue, onFalse);
    }

    private void doIfLastResourceOrElse(int resourcesCount, Runnable onTrue, Runnable onFalse) {
        if (resourcesCount == 1) {
            onTrue.run();
        } else {
            onFalse.run();
        }
    }

    public void moveMeetingEnd(String token, long eventId, LocalDateTime instanceStart, LocalDateTime newEnd,
                               ActionInfo actionInfo) {
        val resource = getResourceInfo(token);
        val resourceId = resource.getResourceId();

        val officeTz = OfficeManager.getOfficeTimeZone(resource.getOffice());
        val startTs = instanceStart.toDateTime(officeTz).toInstant();
        val endTs = newEnd.toDateTime(officeTz).toInstant();

        transactionTemplate.execute(s -> {
            resourceRoutines.lockResourcesByIds(Cf.list(resourceId));

            val eventResources = eventResourceDao.findEventResourcesByEventId(eventId);
            if (eventResources.stream().noneMatch(res -> res.getResourceId() == resourceId)) {
                throw CommandRunException.createSituation("event not found id " + eventId, Situation.EVENT_NOT_FOUND);
            }

            val event = eventDbManager.getEventByIdSafe(eventId)
                    .getOrThrow(CommandRunException.createSituationF("not found by id " + eventId,
                            Situation.EVENT_NOT_FOUND));

            val eventInstance = eventRoutines.getSingleInstance(Option.of(ROBOT_RES_MASTER_UID), Option.of(startTs),
                    eventId,
                    actionInfo.getActionSource());
            if (eventInstance.getInterval().getEnd().isAfter(endTs)) {
                declineInstance(resource, event, startTs, EnumSet.noneOf(UpdateOptions.class), actionInfo);
            }

            return null;
        });
    }

    public void declineMeeting(String token, long eventId, LocalDateTime instanceStart, ActionInfo actionInfo) {
        declineOrDeleteMeeting(token, eventId, instanceStart, false, actionInfo);
    }

    public void deleteMeeting(String token, long eventId, LocalDateTime instanceStart, ActionInfo actionInfo) {
        declineOrDeleteMeeting(token, eventId, instanceStart, true, actionInfo);
    }

    private void updateLastPingTs(long resourceId, Instant ts) {
        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
        try {
            Resource data = new Resource();
            data.setId(resourceId);
            data.setDisplayLastPingTs(ts);

            resourceRoutines.updateResource(data);
        } catch (Exception e) {
            log.error("Failed to update last ping ts", e);
        } finally {
            handle.popSafely();
        }
    }

    private void declineOrDeleteMeeting(String token, long eventId, LocalDateTime instanceStart, boolean isDelete,
                                        ActionInfo actionInfo) {
        val resource = getResourceInfo(token);

        val resourceId = resource.getResourceId();
        val officeTz = OfficeManager.getOfficeTimeZone(resource.getOffice());
        val startTs = instanceStart.toDateTime(officeTz).toInstant();

        transactionTemplate.execute(s -> {
            resourceRoutines.lockResourcesByIds(Cf.list(resourceId));
            {
                int countryId = resource.getOffice().getCityName()
                        .map(OfficeManager::getCountryIdByCityName).getOrElse(GeobaseIds.RUSSIA);
                if (HolidayRoutines.isDayOff(instanceStart.toLocalDate(), countryId)) {
                    log.info("Skip declineOrDeleteMeeting for resource {} because of holiday", resourceId);
                    return null;
                }

                Event event = eventDbManager.getEventByIdSafe(eventId)
                        .getOrThrow(CommandRunException.createSituationF("not found by id " + eventId,
                                Situation.EVENT_NOT_FOUND));

                boolean isRobotResMasterOrganizer = eventDbManager.getParticipantsByEventIds(Cf.list(eventId), false)
                        .stream()
                        .anyMatch(p -> p.getParticipants().isOrganizer(ROBOT_RES_MASTER_UID));

                boolean eventContainsRoom = StreamEx.of(eventDbManager.findResourcesByEventIds(Cf.list(eventId)).get2())
                        .flatCollection(identity())
                        .anyMatch(res -> res.getResourceId() == resourceId);
                if (!eventContainsRoom) {
                    throw CommandRunException.createSituation("not participant", Situation.EVENT_NOT_FOUND);
                }
                final UserInfo user = userManager.getUserInfo(ROBOT_RES_MASTER_UID);
                final EventWithRelations eventWithRelations = eventDbManager.getEventWithRelationsById(eventId);
                EventInfoForPermsCheck eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, eventWithRelations);
                EnumSet<EventAction> eventPermissions = authorizer.getEventPermissions(user, eventAuthInfo,
                        actionInfo.getActionSource());
                if (!eventPermissions.contains(EventAction.EDIT)) {
                    log.info("Skip declineOrDeleteMeeting for resource {} cause not enough permissions {} for {} from" +
                                    " {} ",
                            resourceId, eventPermissions, event.getId(), user.getUid());
                    return null;
                }

                Option<Event> master = eventRoutines.getMainInstOfEvent(event);
                boolean missingCheckInLimitExceeded = false;

                int timesToTotal = SpecialResources.getDisplaySettings(resource.getResource()).getTimesToTotalDecline();
                if (master.isPresent() && !isDelete && timesToTotal > 0) {
                    int uncheckinTimesInARow = registerUncheckinGetTimesInARow(resourceId, event, master.get(),
                            startTs);
                    missingCheckInLimitExceeded = uncheckinTimesInARow >= timesToTotal;
                }

                if (isRobotResMasterOrganizer || isDelete) {
                    if (missingCheckInLimitExceeded) {
                        declineMasterAndRecurrencesOnTotalUnchekIn(resource, master.get(), startTs, actionInfo);
                    } else {
                        // CAL-8029
                        doIfLastResourceOrElse(event,
                                () -> cancelInstance(resource, event, startTs, actionInfo),
                                () -> declineInstance(resource, event, startTs,
                                        EnumSet.of(UpdateOptions.SUPPRESS_GENERAL_MAILS), actionInfo));
                    }
                } else {
                    if (missingCheckInLimitExceeded) {
                        declineMasterAndRecurrencesOnTotalUnchekIn(resource, master.get(), startTs, actionInfo);
                    } else {
                        val options = EnumSet.of(UpdateOptions.SUPPRESS_GENERAL_MAILS,
                                UpdateOptions.SEND_UNCHECKIN_MAIL);
                        declineInstance(resource, event, startTs, options, actionInfo);
                    }
                }
            }
            return null;
        });
    }

    private void decline(ResourceInfo resourceInfo, Event event, Instant instanceStart, EnumSet<UpdateOptions> options,
                         Optional<Repetition> repetition, boolean applyToFuture, ActionInfo actionInfo) {
        ListF<EventMessageParameters> emails;
        if (options.contains(UpdateOptions.SEND_UNCHECKIN_MAIL)) {
            val eventInfo = prepareEmailEventInfo(event, instanceStart, applyToFuture);
            emails = createEventEmails(resourceInfo, eventInfo.getEvent(), eventInfo.getInstance(),
                    MailType.RESOURCE_UNCHECKIN, actionInfo);
        } else {
            emails = Cf.list();
        }

        val userInfo = userManager.getUserInfo(ROBOT_RES_MASTER_UID);
        val eventData = new EventData();
        eventData.setEvent(event.copy());
        eventData.setInstanceStartTs(instanceStart);
        repetition.ifPresent(eventData::setRepetition);
        eventData.setInvData(new EventInvitationUpdateData(Option.empty(), Cf.list(resourceInfo.getEmail())));
        val sendMails = !options.contains(UpdateOptions.SUPPRESS_GENERAL_MAILS);
        eventWebManager.update(userInfo, eventData, new NotificationsData.Update(Cf.map()), Option.empty(),
                Option.empty(), applyToFuture, Option.of(sendMails), !sendMails, actionInfo);

        sendMailTasks(actionInfo, emails);
    }

    @NotNull
    private ListF<EventMessageParameters> filterExternalUsers(ListF<EventMessageParameters> emails) {
        ListF<EventMessageParameters> result = Cf.arrayList();
        for (EventMessageParameters email : emails) {
            final Option<PassportUid> recipientUid = email.getRecipientUid();
            if (recipientUid.isPresent() && !userManager.isExternalYtUser(recipientUid.get())) {
                result.add(email);
            }
        }
        return result;
    }

    private void declineSeries(ResourceInfo resourceInfo, Event master, Instant instanceStart,
                               EnumSet<UpdateOptions> options,
                               ActionInfo actionInfo) {
        val eventAndRepetition = eventDbManager.getEventAndRepetitionById(master.getId());
        val repetition = eventAndRepetition.getRepetitionInfo().getRepetition().toOptional();
        decline(resourceInfo, master, instanceStart, options, repetition, true, actionInfo);
    }

    private void declineInstance(ResourceInfo resourceInfo, Event instance, Instant instanceStart,
                                 EnumSet<UpdateOptions> options,
                                 ActionInfo actionInfo) {
        val recurrenceOrInstance =
                eventRoutines.getSingleEventInstanceForModifierOrCreateRecurrence(Option.of(ROBOT_RES_MASTER_UID),
                instance.getId(), instanceStart, Option.empty(), actionInfo);
        decline(resourceInfo, recurrenceOrInstance.getEvent(), instanceStart, options, Optional.empty(), false,
                actionInfo);
    }

    private void declineMasterAndRecurrencesOnTotalUnchekIn(ResourceInfo resource, Event master, Instant instanceStart,
                                                            ActionInfo actionInfo) {
        val options = EnumSet.of(
                UpdateOptions.SEND_UNCHECKIN_MAIL,
                UpdateOptions.SUPPRESS_GENERAL_MAILS);
        declineSeries(resource, master, instanceStart, options, actionInfo);
    }

    private EmailEventInfo prepareEmailEventInfo(Event event, Instant instanceStart, boolean applyToFuture) {
        val master = eventRoutines.getMainInstOfEvent(event);
        if (master.isPresent() && master.get().getRepetitionId().isPresent() && master.get().getId().equals(event.getId())) {
            val duration = EventRoutines.getInstantInterval(event).getDuration();
            val instance = applyToFuture
                    ? EventInstanceParameters.fromEvent(master.get())
                    : new EventInstanceParameters(instanceStart, instanceStart.plus(duration),
                    Option.of(instanceStart));
            return new EmailEventInfo(master.get(), instance);
        } else {
            return new EmailEventInfo(event, EventInstanceParameters.fromEvent(event));
        }
    }

    private void cancelInstance(ResourceInfo resource, Event event, Instant instanceStart, ActionInfo actionInfo) {
        val eventInfo = prepareEmailEventInfo(event, instanceStart, false);
        val emails = createEventEmails(resource, eventInfo.getEvent(), eventInfo.getInstance(), MailType.EVENT_CANCEL,
                actionInfo);

        val userInfo = userManager.getUserInfo(ROBOT_RES_MASTER_UID);
        eventWebManager.deleteEvent(userInfo, eventInfo.getEvent().getId(),
                Option.of(eventInfo.getInstance().getStartTs()),
                true, false, actionInfo);

        sendMailTasks(actionInfo, emails);
    }

    private void sendMailTasks(ActionInfo actionInfo, ListF<EventMessageParameters> emails) {
        ListF<EventMessageParameters> ytUsers = filterExternalUsers(emails);
        if (ytUsers.isNotEmpty()) {
            mailSender.sendEmailsViaTask(ytUsers, actionInfo);
        }
    }

    private int registerUncheckinGetTimesInARow(long resourceId, Event master, Event instance, Instant instanceStart) {
        RepetitionInstanceInfo repetition = repetitionRoutines.getRepetitionInstanceInfoByEvent(master);
        if (repetition.isEmpty()) {
            return 0;
        }

        Option<EventResourceUncheckin> uncheckinO = eventResourceUncheckinDao.findByMainEventIdAndResourceId(
                master.getMainEventId(), resourceId);

        LocalDate recurrenceDate = new LocalDate(instance.getRecurrenceId().getOrElse(instanceStart),
                repetition.getTz());
        final EventResourceUncheckin uncheckin;
        if (!uncheckinO.isPresent()) {
            uncheckin = new EventResourceUncheckin();
            uncheckin.setMainEventId(master.getMainEventId());
            uncheckin.setResourceId(resourceId);
            uncheckin.setPreviousDate(recurrenceDate);
            uncheckin.setTimesInARow(1);

            eventResourceUncheckinDao.save(uncheckin);
        } else {
            uncheckin = uncheckinO.get().copy();
            LocalDate previousDate = uncheckin.getPreviousDate();

            if (!previousDate.equals(recurrenceDate)) {
                Option<InstantInterval> nextInterval = RepetitionUtils.getInstanceIntervalStartingAfter(
                        repetition.withoutRecurrences(),
                        previousDate.plusDays(1).toDateTimeAtStartOfDay(repetition.getTz()).toInstant());

                Option<LocalDate> nextDate = nextInterval.map(i -> new LocalDate(i.getStart(), repetition.getTz()));

                if (nextDate.isSome(recurrenceDate)) {
                    uncheckin.setTimesInARow(uncheckin.getTimesInARow() + 1);
                } else {
                    uncheckin.setTimesInARow(1);
                }
                uncheckin.setPreviousDate(recurrenceDate);
                eventResourceUncheckinDao.update(uncheckin);
            }
        }
        return uncheckin.getTimesInARow();
    }

    private ListF<EventMessageParameters> createEventEmails(
            ResourceInfo resource, Event event, EventInstanceParameters instance, MailType type,
            ActionInfo actionInfo) {
        return createEventEmails(resource, event, instance, type, Option.empty(), actionInfo);
    }

    private ListF<EventMessageParameters> createEventEmails(
            ResourceInfo resource, Event event, EventInstanceParameters instance, MailType type,
            Option<ChangedEventInfoForMails> changesInfo, ActionInfo actionInfo) {
        long resourceId = resource.getResourceId();

        RepetitionInstanceInfo repetition = eventDbManager.getEventAndRepetitionByEvent(event).getRepetitionInfo();
        EventWithRelations eventWR = eventDbManager.getEventWithRelationsByEvent(event);

        ListF<ParticipantInfo> invitations = EventInvitationManager.getInvitationsExceptSenderAndRejected(
                eventWR.getParticipants(), Option.of(UidOrResourceId.resource(resourceId)));

        ListF<EventSendingInfo> sendingInfos = eventInvitationManager.prepareSendingInfoForParticipants(
                invitations, type, instance, changesInfo,
                type != MailType.RESOURCE_UNCHECKIN && eventWR.isExportedWithEws());

        if (type == MailType.RESOURCE_UNCHECKIN) {
            sendingInfos = sendingInfos.map(s -> s.withSubscriptionResource(resource));
        }

        return eventInvitationManager.createEventInvitationOrCancelMails(
                ActorId.resource(resourceId), eventWR, repetition, sendingInfos, actionInfo);
    }

    private ResourceInfo getResourceInfo(String token) {
        return resourceRoutines.findResourceInfoByDisplayToken(token).getOrThrow(CommandRunException.createSituationF(
                "resource cannot be found by token " + token, Situation.INVALID_TOKEN));
    }

    private RoomInfo toRoom(ResourceInfo resourceInfo, Language lang, ActionInfo actionInfo) {
        long id = resourceInfo.getResourceId();

        String name = ResourceRoutines.getNameI18n(resourceInfo.getResource(), lang).getOrElse("");
        String alterName = ResourceRoutines.getAlterNameI18n(resourceInfo.getResource(), lang).getOrElse("");
        String officeName = ResourceRoutines.getNameI18n(resourceInfo.getOffice(), lang);

        LocalDateTime currentTime = new LocalDateTime(
                actionInfo.getNow(), OfficeManager.getOfficeTimeZone(resourceInfo.getOffice()));

        boolean isActive = resourceInfo.getResource().getIsActive()
                || resourceInfo.getEmail().equalsIgnoreCase(SpecialResources.DISPLAY_ROOM_EMAIL);

        return new RoomInfo(id, name, alterName, officeName, currentTime, isActive);
    }

    private MeetingInfo toMeeting(EventInstanceInfo ei, DateTimeZone tz, Language lang) {
        EventWithRelations event = ei.getEventWithRelations();
        EventInterval interval = event.getEventInterval(ei.getInterval());

        boolean mayView = ei.getMayView();
        Participants participants = event.getParticipants();

        long id = event.getId();
        LocalDateTime instanceStart = new LocalDateTime(interval.getInstanceStart(), tz);

        String name = mayView ? event.getEvent().getName() : "";
        LocalDateTime start = interval.getStart().toLocalDateTime(tz);
        LocalDateTime end = interval.getEnd().toLocalDateTime(tz);

        Option<MeetingParticipantInfo> organizer = participants.getOrganizerSafe()
                .filterMap(p -> toParticipantIfYandexUser(p, lang));

        ListF<MeetingParticipantInfo> attendees = participants.getAllAttendeesButNotOrganizerSafe()
                .filterMap(p -> toParticipantIfYandexUser(p, lang));

        return new MeetingInfo(MeetingIdInfo.event(id, instanceStart), name, start, end, organizer, attendees);
    }

    private ListF<MeetingInfo> toMeetings(ResourceReservationInfo reservation, InstantInterval interval,
                                          DateTimeZone tz,
                                          Language lang) {
        val user = userManager.getYtUserByUid(reservation.getCreatorUid());
        final var organizer = user.map(u -> new MeetingParticipantInfo(
                lang == Language.RUSSIAN
                        ? UserManager.extractPrettyUserName(u)
                        : UserManager.getFullNameEn(u).orElseGet(() -> UserManager.extractPrettyUserName(u)),
                u.getLogin()));

        ListF<MeetingInfo> result = Cf.arrayList();
        for (InstantInterval instance : reservation.getInstanceIntervalsInInterval(interval)) {
            result.add(new MeetingInfo(
                    MeetingIdInfo.temporaryReservation(), "",
                    new LocalDateTime(instance.getStart(), tz),
                    new LocalDateTime(instance.getEnd(), tz),
                    Option.x(organizer), Cf.list()));
        }
        return result;
    }

    private Option<MeetingInfo> toMeeting(DateInterval restriction, LocalDate date, Language lang) {
        LocalDateTime start = date.toLocalDateTime(LocalTime.MIDNIGHT);
        LocalDateTime end = start.plusDays(1);

        return Option.when(restriction.overlaps(date), new MeetingInfo(
                MeetingIdInfo.bookingRestriction(),
                restriction.title.map(t -> t.getName(lang)).getOrElse(""),
                restriction.start.map(DateOrDateTime::toLocalDateTime).getOrElse(start),
                restriction.end.map(DateOrDateTime::toLocalDateTime).getOrElse(end),
                Option.empty(), Cf.list()));
    }

    private Option<MeetingParticipantInfo> toParticipant(YandexUserParticipantInfo p, Language lang) {
        if (!p.getLogin().isPresent()) {
            return Option.empty();
        }

        String login = p.getLogin().get();
        String name = p.getHiddenName();

        if (lang != Language.RUSSIAN) {
            val user = userManager.getYtUserByLogin(login);
            if (user.isPresent()) {
                val nameEn = UserManager.getFullNameEn(user.get());
                if (nameEn.isPresent()) {
                    name = nameEn.get();
                }
            }
        }
        return Option.of(new MeetingParticipantInfo(name, login));
    }

    private Option<MeetingParticipantInfo> toParticipantIfYandexUser(ParticipantInfo p, Language lang) {
        return p instanceof YandexUserParticipantInfo
                ? toParticipant((YandexUserParticipantInfo) p, lang)
                : Option.empty();
    }
}
