package ru.yandex.calendar.frontend.ews;

import javax.xml.bind.JAXBElement;

import com.microsoft.schemas.exchange.services._2006.types.AndType;
import com.microsoft.schemas.exchange.services._2006.types.BasePathToElementType;
import com.microsoft.schemas.exchange.services._2006.types.ConstantValueType;
import com.microsoft.schemas.exchange.services._2006.types.DistinguishedPropertySetType;
import com.microsoft.schemas.exchange.services._2006.types.FieldURIOrConstantType;
import com.microsoft.schemas.exchange.services._2006.types.IsEqualToType;
import com.microsoft.schemas.exchange.services._2006.types.IsGreaterThanOrEqualToType;
import com.microsoft.schemas.exchange.services._2006.types.IsGreaterThanType;
import com.microsoft.schemas.exchange.services._2006.types.IsLessThanOrEqualToType;
import com.microsoft.schemas.exchange.services._2006.types.IsLessThanType;
import com.microsoft.schemas.exchange.services._2006.types.MapiPropertyTypeType;
import com.microsoft.schemas.exchange.services._2006.types.ObjectFactory;
import com.microsoft.schemas.exchange.services._2006.types.PathToExtendedFieldType;
import com.microsoft.schemas.exchange.services._2006.types.PathToUnindexedFieldType;
import com.microsoft.schemas.exchange.services._2006.types.TwoOperandExpressionType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author ssytnik
 */
public class EwsConditions {
    private static final ObjectFactory typesObjectFactory = new ObjectFactory();

    public static JAXBElement<? extends BasePathToElementType> createUnindexedFieldPath(UnindexedFieldURIType fieldUri) {
        PathToUnindexedFieldType pathToField = new PathToUnindexedFieldType();
        pathToField.setFieldURI(fieldUri);
        return typesObjectFactory.createFieldURI(pathToField);
    }

    public static FieldURIOrConstantType createConstant(String value) {
        FieldURIOrConstantType res = new FieldURIOrConstantType();
        ConstantValueType fieldValue = new ConstantValueType();
        fieldValue.setValue(value);
        res.setConstant(fieldValue);
        return res;
    }

    public static JAXBElement<IsEqualToType> createFieldEqualsCondition(UnindexedFieldURIType fieldUri, String value) {
        IsEqualToType newCondition = new IsEqualToType();
        newCondition.setPath(createUnindexedFieldPath(fieldUri));
        newCondition.setFieldURIOrConstant(createConstant(value));
        return typesObjectFactory.createIsEqualTo(newCondition);
    }

    public static JAXBElement<? extends TwoOperandExpressionType> createFieldCompareCondition(
            UnindexedFieldURIType fieldUri, boolean isFieldLess, boolean isStrict, String value)
    {
        TwoOperandExpressionType res = isFieldLess ?
                (isStrict ? new IsLessThanType() : new IsLessThanOrEqualToType()) :
                (isStrict ? new IsGreaterThanType() : new IsGreaterThanOrEqualToType());

        res.setPath(createUnindexedFieldPath(fieldUri));
        res.setFieldURIOrConstant(createConstant(value));

        return (
            isFieldLess && isStrict ? typesObjectFactory.createIsLessThan((IsLessThanType) res) :
            isFieldLess && !isStrict ? typesObjectFactory.createIsLessThanOrEqualTo((IsLessThanOrEqualToType) res) :
            !isFieldLess && isStrict ? typesObjectFactory.createIsGreaterThan((IsGreaterThanType) res) :
            typesObjectFactory.createIsGreaterThanOrEqualTo((IsGreaterThanOrEqualToType) res)
        );
    }

    public static JAXBElement<AndType> createEventIntervalOverlapCondition(InstantInterval interval) {

        String start = EwsUtils.instantToXMLGregorianCalendar(interval.getStart(), DateTimeZone.UTC).toString();
        String end = EwsUtils.instantToXMLGregorianCalendar(interval.getEnd(), DateTimeZone.UTC).toString();

        AndType newCondition = new AndType();
        newCondition.getSearchExpression().addAll(Cf.list(
            createFieldCompareCondition(UnindexedFieldURIType.CALENDAR_END, false, true, start),
            createFieldCompareCondition(UnindexedFieldURIType.CALENDAR_START, true, true, end)
        ));

        return typesObjectFactory.createAnd(newCondition);
    }

    // can be useful for searching events by exchange's hex UID; useless for other UIDs. Details are here:
    // http://www.infinitec.de/post/2009/04/13/Searching-a-meeting-with-a-specific-UID-using-Exchange-Web-Services-2007.aspx
    @Deprecated
    public static JAXBElement<IsEqualToType> createGlobalObjectIdEqualsCondition(String globalObjectIdValue) {
        IsEqualToType newCondition = new IsEqualToType();

        PathToExtendedFieldType pathToField = new PathToExtendedFieldType();
        pathToField.setDistinguishedPropertySetId(DistinguishedPropertySetType.MEETING);
        pathToField.setPropertyId(0x03);
        pathToField.setPropertyType(MapiPropertyTypeType.BINARY);
        newCondition.setPath(typesObjectFactory.createExtendedFieldURI(pathToField));

        newCondition.setFieldURIOrConstant(createConstant(globalObjectIdValue));

        return typesObjectFactory.createIsEqualTo(newCondition);
    }
}
