package ru.yandex.calendar.frontend.ews;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.microsoft.schemas.exchange.services._2006.messages.ResponseMessageType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseClassType;
import lombok.val;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.admin.universal.CalendarReflectionToStringStyle;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

public class EwsErrorResponseException extends EwsException {
    private final ResponseMessageType responseMessageType;

    public EwsErrorResponseException(ResponseMessageType response) {
        this(null, response);
    }

    public EwsErrorResponseException(@Nullable String message, ResponseMessageType response) {
        super(message(message, response));
        this.responseMessageType = response;
        Validate.equals(ResponseClassType.ERROR, response.getResponseClass());
    }

    private static String message(@Nullable String message, ResponseMessageType response) {
        val sb = new StringBuilder();

        if (StringUtils.isNotEmpty(message)) {
            sb.append(message).append(": ");
        }
        sb.append(message(response));

        return sb.toString();
    }

    public ResponseMessageType getResponseMessageType() {
        return responseMessageType;
    }

    public static String message(ResponseMessageType response) {
        val sb = new StringBuilder();

        sb.append(response.getResponseCode());
        sb.append(": ").append(response.getMessageText());

        val values = getValues(response.getMessageXml());

        if (!values.isEmpty()) {
            sb.append(", values: ").append(values);
        }
        return sb.toString();
    }

    public static List<String> getValues(@Nullable ResponseMessageType.MessageXml xml) {
        if (xml == null) {
            return Cf.list();
        }
        val style = new CalendarReflectionToStringStyle();
        val values = xml.getAny().stream().map(object -> ReflectionToStringBuilder.toString(Option.of(object), style))
                .map(value -> StringUtils.substringBeforeLast(StringUtils.substringAfter(value, "[x="), "]"))
                .collect(Collectors.toList());
        return values;
    }
}
