package ru.yandex.calendar.frontend.ews;

import java.util.List;

import javax.annotation.Nullable;
import javax.xml.bind.JAXBElement;
import javax.xml.datatype.DatatypeConstants;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import com.microsoft.schemas.exchange.services._2006.types.BaseItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.BasePathToElementType;
import com.microsoft.schemas.exchange.services._2006.types.BodyType;
import com.microsoft.schemas.exchange.services._2006.types.BodyTypeType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemType;
import com.microsoft.schemas.exchange.services._2006.types.CalendarItemTypeType;
import com.microsoft.schemas.exchange.services._2006.types.DayOfWeekIndexType;
import com.microsoft.schemas.exchange.services._2006.types.DeleteItemFieldType;
import com.microsoft.schemas.exchange.services._2006.types.DistinguishedPropertySetType;
import com.microsoft.schemas.exchange.services._2006.types.EmailAddressType;
import com.microsoft.schemas.exchange.services._2006.types.ExtendedPropertyType;
import com.microsoft.schemas.exchange.services._2006.types.ItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.ItemType;
import com.microsoft.schemas.exchange.services._2006.types.MapiPropertyTypeType;
import com.microsoft.schemas.exchange.services._2006.types.MeetingRequestMessageType;
import com.microsoft.schemas.exchange.services._2006.types.MonthNamesType;
import com.microsoft.schemas.exchange.services._2006.types.ObjectFactory;
import com.microsoft.schemas.exchange.services._2006.types.OccurrenceItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.PathToExtendedFieldType;
import com.microsoft.schemas.exchange.services._2006.types.PathToUnindexedFieldType;
import com.microsoft.schemas.exchange.services._2006.types.RecurringMasterItemIdType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseObjectCoreType;
import com.microsoft.schemas.exchange.services._2006.types.ResponseTypeType;
import com.microsoft.schemas.exchange.services._2006.types.SetItemFieldType;
import com.microsoft.schemas.exchange.services._2006.types.TimeZoneDefinitionType;
import com.microsoft.schemas.exchange.services._2006.types.TimeZoneType;
import com.microsoft.schemas.exchange.services._2006.types.UnindexedFieldURIType;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.YtEwsSubscription;
import ru.yandex.calendar.logic.event.ExchangeEventSynchData;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;

public class EwsUtils {
    public static final int STATUS_FREQUENCY_IN_MINUTES = 1;
    public static final int PULL_SUBSCRIPTION_TIMEOUT_IN_MINUTES = 120;

    public static final String EXTENDED_PROPERTY_SOURCE = "EX-SOURCE";
    public static final String EXTENDED_PROPERTY_ORGANIZER = "YA-TEAM-ORGANIZER";
    public static final String YA_TEAM_CALENDAR_SOURCE = "YA-TEAM CALENDAR";

    public static final String EXTENDED_PROPERTY_RECURRENCE_ID = "YA-TEAM-RECURRENCE-ID";
    private static final DateTimeFormatter RECURRENCE_ID_FORMATTER =
            DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss'Z'").withZoneUTC();

    private static final ObjectFactory typesObjectFactory = new ObjectFactory();

    public static EmailAddressType createEmailAddressType(Email email) {
        EmailAddressType calendarUser = new EmailAddressType();
        calendarUser.setEmailAddress(email.getEmail());
        return calendarUser;
    }

    private static PathToUnindexedFieldType createUnindexedFieldUri(UnindexedFieldURIType fieldUri) {
        PathToUnindexedFieldType pathToUnindexedField = new PathToUnindexedFieldType();
        pathToUnindexedField.setFieldURI(fieldUri);
        return pathToUnindexedField;
    }

    private static PathToExtendedFieldType createExtendedFieldUri(String fieldName) {
        PathToExtendedFieldType pathToExtendedField = new PathToExtendedFieldType();
        pathToExtendedField.setPropertyName(fieldName);
        pathToExtendedField.setPropertyType(MapiPropertyTypeType.STRING);
        pathToExtendedField.setDistinguishedPropertySetId(DistinguishedPropertySetType.PUBLIC_STRINGS);
        return pathToExtendedField;
    }

    public static JAXBElement<? extends BasePathToElementType> createUnindexedFieldPath(UnindexedFieldURIType field) {
        return typesObjectFactory.createFieldURI(createUnindexedFieldUri(field));
    }

    public static JAXBElement<? extends BasePathToElementType> createExtendedFieldPath(String fieldName) {
        return typesObjectFactory.createExtendedFieldURI(createExtendedFieldUri(fieldName));
    }

    public static ExtendedPropertyType createExtendedProperty(String name, String value) {
        ExtendedPropertyType extProperty = new ExtendedPropertyType();
        extProperty.setExtendedFieldURI(createExtendedFieldUri(name));
        extProperty.setValue(value);
        return extProperty;
    }

    public static ExtendedPropertyType createSourceExtendedProperty() {
        return createExtendedProperty(EXTENDED_PROPERTY_SOURCE, YA_TEAM_CALENDAR_SOURCE);
    }

    public static ExtendedPropertyType createOrganizerExtendedProperty(Email organizerEmail) {
        return createExtendedProperty(EXTENDED_PROPERTY_ORGANIZER, organizerEmail.getEmail());
    }

    // https://jira.yandex-team.ru/browse/CAL-3742
    public static ExtendedPropertyType createRecurrenceIdExtendedProperty(Instant recurrenceId) {
        return createExtendedProperty(EXTENDED_PROPERTY_RECURRENCE_ID, recurrenceId.toString(RECURRENCE_ID_FORMATTER));
    }

    public static Option<Instant> toInstantO(XMLGregorianCalendar dt) {
        if (dt == null) {
            return Option.empty();
        } else {
            return Option.of(xmlGregorianCalendarInstantToInstant(dt));
        }
    }

    public static Option<Instant> getRecurrenceId(CalendarItemType item) {
        return EwsUtils.toInstantO(item.getRecurrenceId()).orElse(convertExtendedProperties(item).getRecurrenceId());
    }

    /**
     * XXX ssytnik@: potential bug (I did not check): dt might not have timezone in it.
     * For details, see {@link XMLGregorianCalendar#toGregorianCalendar()}'s javadoc.
     */
    public static Instant xmlGregorianCalendarInstantToInstant(XMLGregorianCalendar dt) {
        Validate.isTrue(dt.getTimezone() != DatatypeConstants.FIELD_UNDEFINED);
        validateIfNotInProduction(dt.getHour() != DatatypeConstants.FIELD_UNDEFINED, "Hour undefined");
        validateIfNotInProduction(dt.getMinute() != DatatypeConstants.FIELD_UNDEFINED, "Minute undefined");
        validateIfNotInProduction(dt.getSecond() != DatatypeConstants.FIELD_UNDEFINED, "Second undefined");
        return new Instant(dt.toGregorianCalendar().getTimeInMillis());
    }

    public static LocalDate xmlGregorianCalendarLocalDateToLocalDate(XMLGregorianCalendar dt) {
        validateIfNotInProduction(dt.getHour() == DatatypeConstants.FIELD_UNDEFINED, "Hour defined");
        validateIfNotInProduction(dt.getMinute() == DatatypeConstants.FIELD_UNDEFINED, "Minute defined");
        validateIfNotInProduction(dt.getSecond() == DatatypeConstants.FIELD_UNDEFINED, "Second defined");
        return new LocalDate(dt.getYear(), dt.getMonth(), dt.getDay());
    }

    public static DatatypeFactory newDatatypeFactory() {
        try {
            return DatatypeFactory.newInstance();
        } catch (Exception e) {
            throw translate(e);
        }
    }

    public static RuntimeException translate(Exception e) {
        return ExceptionUtils.translate(e);
    }

    public static TimeZoneType getOrDefaultZoneType(DateTimeZone zone) {
        TimeZoneType tz = new TimeZoneType();
        String zoneName = WindowsTimeZones.getWinNameByZone(zone).getOrElse("UTC"); // default can be any timezone // ssytnik@
        tz.setTimeZoneName(zoneName);
        return tz;
    }

    // ssytnik@: workaround exchange behavior. One should:
    // when setting timezone - use MeetingTimeZone property (TimeZone is read-only);
    // when reading timezone - use TimeZone property (MeetingTimeZone is not set for some reason)
    //
    // TODO unify with IcsTimeZones#forId, in particular, support parse timezone by offset
    private static DateTimeZone getOrDefaultZone(
            @Nullable String timeZoneProp,
            @Nullable TimeZoneType meetingTimeZoneProp,
            @Nullable TimeZoneDefinitionType startTimezoneProp)
    {
        Option<DateTimeZone> tz = Option.empty();

        Function<String, Option<DateTimeZone>> lookup = name -> WindowsTimeZones.getZoneByWinName(name)
                .orElse(Option.ofNullable(name).filter(DateTimeZone.getAvailableIDs()::contains).map(DateTimeZone::forID));

        if (startTimezoneProp != null) {
            tz = lookup.apply(startTimezoneProp.getId());
        }
        if (!tz.isPresent() && startTimezoneProp != null) {
            tz = lookup.apply(startTimezoneProp.getName());
        }
        if (!tz.isPresent() && timeZoneProp != null) {
            tz = lookup.apply(timeZoneProp);
        }
        if (!tz.isPresent() && meetingTimeZoneProp != null) {
            tz = lookup.apply(meetingTimeZoneProp.getTimeZoneName());
        }
        if (!tz.isPresent()) {
            tz = Option.of(DateTimeZone.UTC); // UTC here is important // ssytnik@
        }
        return tz.get();
    }

    public static DateTimeZone getOrDefaultZone(CalendarItemType itemType) {
        return getOrDefaultZone(itemType.getTimeZone(), itemType.getMeetingTimeZone(), itemType.getStartTimeZone());
    }

    public static DateTimeZone getOrDefaultZone(MeetingRequestMessageType itemType) {
        return getOrDefaultZone(itemType.getTimeZone(), itemType.getMeetingTimeZone(), itemType.getStartTimeZone());
    }

    public static Option<String> getMeetingTimezoneName(CalendarItemType itemType) {
        return Option.ofNullable(itemType.getMeetingTimeZone()).map(TimeZoneType::getTimeZoneName).filterNotNull();
    }

    public static XMLGregorianCalendar dateTimeToXMLGregorianCalendar(DateTime dateTime) {
        return instantToXMLGregorianCalendar(dateTime.toInstant(), dateTime.getZone());
    }

    public static XMLGregorianCalendar localDateToXmlGregorianCalendarInUtc(LocalDate localDate) {
        int utcZoneOffsetMinutes = 0;
        return newDatatypeFactory().newXMLGregorianCalendarDate(
                localDate.getYear(), localDate.getMonthOfYear(), localDate.getDayOfMonth(), utcZoneOffsetMinutes);
    }

    public static XMLGregorianCalendar instantToXMLGregorianCalendar(Instant instant, DateTimeZone zone) {
        DateTime dateTime = new DateTime(instant.getMillis(), zone);
        return newDatatypeFactory().newXMLGregorianCalendar(dateTime.toGregorianCalendar());
    }

    public static XMLGregorianCalendar localDateTimeToXmlGregorianCalendar(LocalDateTime l) {
        return newDatatypeFactory().newXMLGregorianCalendar(
                l.getYear(), l.getMonthOfYear(), l.getDayOfMonth(),
                l.getHourOfDay(), l.getMinuteOfHour(), l.getSecondOfMinute(), l.getMillisOfSecond(),
                DatatypeConstants.FIELD_UNDEFINED);
    }

    public static XMLGregorianCalendar localTimeToXmlGregorianCalendar(LocalTime t) {
        return newDatatypeFactory().newXMLGregorianCalendarTime(
                t.getHourOfDay(), t.getMinuteOfHour(), t.getSecondOfMinute(),
                DatatypeConstants.FIELD_UNDEFINED);
    }

    public static XMLGregorianCalendar localDateToXMLGregorianCalendar(LocalDate localDate, DateTimeZone zone) {
        DatatypeFactory dataTypeFactory = newDatatypeFactory();
        int zoneOffsetMinutes = DatatypeConstants.FIELD_UNDEFINED;
        return dataTypeFactory.newXMLGregorianCalendarDate(
                localDate.getYear(),
                localDate.getMonthOfYear(),
                localDate.getDayOfMonth(),
                zoneOffsetMinutes);
    }

    /**
     * @param weekOfMonth 1-based week of month
     */
    public static DayOfWeekIndexType getDayOfWeekIndexTypeByWeekOfMonth(int weekOfMonth) {
        return DayOfWeekIndexType.values()[weekOfMonth - 1];
    }

    /**
     * @param jodaMonthOfYear 1-based month of year
     */
    public static MonthNamesType getMonthNamesTypeByJodaMonthOfYear(int jodaMonthOfYear) {
        return MonthNamesType.values()[jodaMonthOfYear - 1];
    }

    public static ItemIdType createItemId(String itemExchangeId) {
        ItemIdType itemIdType = new ItemIdType();
        itemIdType.setId(itemExchangeId);
        return itemIdType; // and no changeKey
    }

    public static BodyType createTextBody(String value) {
        BodyType body = new BodyType();
        body.setBodyType(BodyTypeType.TEXT);
        body.setValue(value);
        return body;
    }

    public static Function<String, BaseItemIdType> createItemIdF() {
        return EwsUtils::createItemId;
    };

    public static RecurringMasterItemIdType createRecurringMasterItemIdType(String occurrenceExchangeId) {
        RecurringMasterItemIdType itemIdType = new RecurringMasterItemIdType();
        itemIdType.setOccurrenceId(occurrenceExchangeId);
        return itemIdType; // and no changeKey
    }

    public static Function<String, BaseItemIdType> createRecurringMasterItemIdTypeF() {
        return EwsUtils::createRecurringMasterItemIdType;
    };

    public static OccurrenceItemIdType createOccurrenceItemIdType(String recurringMasterExchangeId, int index) {
        OccurrenceItemIdType occurrenceItemIdType = new OccurrenceItemIdType();
        occurrenceItemIdType.setRecurringMasterId(recurringMasterExchangeId);
        occurrenceItemIdType.setInstanceIndex(index);
        return occurrenceItemIdType; // and no changeKey
    }

    public static Function<String, BaseItemIdType> createOccurrenceItemIdTypeF(final int index) {
        return a -> createOccurrenceItemIdType(a, index);
    };

    public static Option<Email> getEmailOSafe(EmailAddressType emailAddress) {
        try {
            return getEmailO(emailAddress);
        } catch (EwsBadEmailException e) {
            return Option.empty();
        }
    }

    public static Option<Email> getEmailO(EmailAddressType emailAddress) {
        // hack for testing with development exchange
        final String email = emailAddress.getEmailAddress();
        if (StringUtils.isNotEmpty(email)) {
            String normalizedEmail = ExchangeEmailManager.getEmailValueByExchangeEmailValue(email, false);
            try {
                return Option.of(new Email(normalizedEmail));
            } catch (Exception e) {
                throw new EwsBadEmailException(normalizedEmail, e);
            }
        } else {
            return Option.empty();
        }
    }

    public static Email getEmail(EmailAddressType emailAddress) {
        Option<Email> o = getEmailO(emailAddress);
        if (o.isPresent()) {
            return o.get();
        } else {
            throw new RuntimeException("e-mail not found by " + YandexToStringBuilder.reflectionToString(emailAddress));
        }
    }

    public static Function<ItemType, String> itemIdF() {
        return item -> item.getItemId().getId();
    }

    public static Function<CalendarItemType, String> calendarItemExchangeIdF() {
        return calendarItemType -> calendarItemType.getItemId().getId();
    }

    public static Function<CalendarItemType, Instant> calendarItemStartF() {
        return i -> xmlGregorianCalendarInstantToInstant(i.getStart());
    }

    public static Function<CalendarItemType, CalendarItemTypeType> calenderItemTypeF() {
        return CalendarItemType::getCalendarItemType;
    }

    public static Function<CalendarItemType, ItemIdType> calendarItemItemIdF() {
        return ItemType::getItemId;
    }

    public static Function<CalendarItemType, String> calendarItemUidF() {
        return c -> StringUtils.defaultIfEmpty(c.getUID(), "?");
    }

    public static Function<CalendarItemType, ExtendedCalendarItemProperties> convertExtendedPropertiesF() {
        return EwsUtils::convertExtendedProperties;
    }

    public static ExtendedCalendarItemProperties convertExtendedProperties(CalendarItemType item) {
        return convertExtendedProperties(item.getExtendedProperty());
    }

    public static ExtendedCalendarItemProperties convertExtendedProperties(
            List<ExtendedPropertyType> extendedProperties)
    {
        MapF<String, String> map = Cf.x(extendedProperties).toMap(
                a -> Tuple2.tuple(
                        a.getExtendedFieldURI().getPropertyName(), a.getValue()));
        Option<Email> organizer = map.getO(EXTENDED_PROPERTY_ORGANIZER).map(Email.consF());

        final Option<String> sourceValueO = map.getO(EXTENDED_PROPERTY_SOURCE);
        boolean wasCreatedFromYaTeam = sourceValueO.isPresent() && sourceValueO.get().equals(YA_TEAM_CALENDAR_SOURCE);

        // https://jira.yandex-team.ru/browse/CAL-3742
        Option<String> recurrenceIdStr = map.getO(EXTENDED_PROPERTY_RECURRENCE_ID);
        Option<Instant> recurrenceId = recurrenceIdStr.isPresent()
                ? Option.of(RECURRENCE_ID_FORMATTER.parseDateTime(recurrenceIdStr.get()).toInstant())
                : Option.empty();

        return new ExtendedCalendarItemProperties(organizer, wasCreatedFromYaTeam, recurrenceId);
    }

    public static UidOrResourceId getSubjectId(YtEwsSubscription subscription) {
        if (subscription.getUid().isPresent()) {
            return UidOrResourceId.user(subscription.getUid().get());
        } else if (subscription.getResourceId().isPresent()) {
            return UidOrResourceId.resource(subscription.getResourceId().get());
        } else {
            throw new IllegalStateException("Unknown type of subscription: " + subscription);
        }
    }

    public static Function<YtEwsSubscription, UidOrResourceId> getSubjectIdF() {
        return EwsUtils::getSubjectId;
    }

    public static SetItemFieldType createSetItemField(
            CalendarItemType calendarItem, UnindexedFieldURIType unindexedFieldUri)
    {
        SetItemFieldType setItemField = new SetItemFieldType();
        setItemField.setCalendarItem(calendarItem);
        setItemField.setPath(createUnindexedFieldPath(unindexedFieldUri));
        return setItemField;
    }

    public static SetItemFieldType createSetItemField(
            CalendarItemType calendarItem, String extendedPropertyName)
    {
        SetItemFieldType setItemField = new SetItemFieldType();
        setItemField.setCalendarItem(calendarItem);
        setItemField.setPath(createExtendedFieldPath(extendedPropertyName));
        return setItemField;
    }

    public static DeleteItemFieldType createDeleteItemField(
            UnindexedFieldURIType unindexedFieldUri)
    {
        DeleteItemFieldType deleteItemField = new DeleteItemFieldType();
        deleteItemField.setPath(createUnindexedFieldPath(unindexedFieldUri));
        return deleteItemField;
    }

    public static boolean isCancelled(CalendarItemType calendarItem) {
        return calendarItem.isIsCancelled() == Boolean.TRUE
                || Cf.list("Отменено:", "Canceled:").exists(p -> StringUtils.startsWith(calendarItem.getSubject(), p));
    }

    public static boolean isOrganizedOrNonMeeting(CalendarItemType item) {
        return item.getMyResponseType() == ResponseTypeType.ORGANIZER
                || item.isIsMeeting() == Boolean.FALSE && item.getMyResponseType() == ResponseTypeType.UNKNOWN;
    }

    public static ExchangeEventSynchData createExchangeSynchData(CalendarItemType calendarItemPartial) {
        String exchangeId = calendarItemPartial.getItemId().getId();
        int sequence = Option.ofNullable(calendarItemPartial.getAppointmentSequenceNumber()).getOrElse(0);
        Instant lastModified = EwsUtils.xmlGregorianCalendarInstantToInstant(calendarItemPartial.getLastModifiedTime());
        Instant dtstamp = EwsUtils.xmlGregorianCalendarInstantToInstant(calendarItemPartial.getDateTimeStamp());
        ExchangeEventSynchData exchangeSynchData = new ExchangeEventSynchData(exchangeId, sequence, lastModified, dtstamp);
        return exchangeSynchData;
    }

    public static ExchangeEventSynchData createExchangeSynchData(EventData eventData) {
        Validate.some(eventData.getExchangeData());

        Event event = eventData.getEvent();
        return new ExchangeEventSynchData(
                eventData.getExchangeData().get().getExchangeId(),
                event.getSequence(), event.getLastUpdateTs(), event.getDtstamp().get());
    }

    public static ItemIdType findItemId(ItemType item) {
        if (item instanceof ResponseObjectCoreType) {
            return ((ResponseObjectCoreType) item).getReferenceItemId(); // includes CancelCalendar, Decline item types
        } else {
            return item.getItemId(); // includes CalendarItemType
        }
    }

    // temp
    private static void validateIfNotInProduction(boolean b, String message) {
        if (EnvironmentType.getActive() != EnvironmentType.PRODUCTION) {
            Validate.isTrue(b, message);
        }
    }

    public static TimeZoneType createTimezoneType(String timezoneName) {
        TimeZoneType timezone = new TimeZoneType();
        timezone.setTimeZoneName(timezoneName);

        return timezone;
    }

    public static TimeZoneDefinitionType createTimezoneDefinitionType(String timezoneName) {
        TimeZoneDefinitionType timezone = new TimeZoneDefinitionType();
        timezone.setId(timezoneName);

        return timezone;
    }

    public static void setStartEndTimezone(CalendarItemType item, DateTimeZone tz) {
        TimeZoneDefinitionType timezone = EwsUtils.createTimezoneDefinitionType(
                WindowsTimeZones.getWinNameByZone(tz).getOrThrow("Unrecognized timezone"));

        item.setStartTimeZone(timezone);
        item.setEndTimeZone(timezone);
    }

    public static Decision getDecisionSafe(ResponseTypeType responseType) {
        return Decision.findByExchangeReponseType(responseType).getOrElse(Decision.UNDECIDED);
    }
}
